% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/eval_feature_representation_summary.R
\name{eval_feature_representation_summary}
\alias{eval_feature_representation_summary}
\alias{eval_feature_representation_summary,ConservationProblem,numeric-method}
\alias{eval_feature_representation_summary,ConservationProblem,matrix-method}
\alias{eval_feature_representation_summary,ConservationProblem,data.frame-method}
\alias{eval_feature_representation_summary,ConservationProblem,Spatial-method}
\alias{eval_feature_representation_summary,ConservationProblem,sf-method}
\alias{eval_feature_representation_summary,ConservationProblem,Raster-method}
\title{Evaluate feature representation}
\usage{
eval_feature_representation_summary(x, solution)

\S4method{eval_feature_representation_summary}{ConservationProblem,numeric}(x, solution)

\S4method{eval_feature_representation_summary}{ConservationProblem,matrix}(x, solution)

\S4method{eval_feature_representation_summary}{ConservationProblem,data.frame}(x, solution)

\S4method{eval_feature_representation_summary}{ConservationProblem,Spatial}(x, solution)

\S4method{eval_feature_representation_summary}{ConservationProblem,sf}(x, solution)

\S4method{eval_feature_representation_summary}{ConservationProblem,Raster}(x, solution)
}
\arguments{
\item{x}{\code{\link[=problem]{problem()}} (i.e. \code{\linkS4class{ConservationProblem}}) object.}

\item{solution}{\code{numeric}, \code{matrix}, \code{data.frame},
\code{\linkS4class{Raster}}, \code{\linkS4class{Spatial}},
or \code{\link[sf:sf]{sf::sf()}} object.
The argument should be in the same format as the planning unit cost
data in the argument to \code{x}.
See the Solution format section for more information.}
}
\value{
\code{\link[tibble:tibble]{tibble::tibble()}} object describing feature representation.
Here, each row describes a specific summary statistic
(e.g. different management zone) for a specific feature.
It contains the following columns:

\describe{

\item{summary}{\code{character} description of the summary statistic.
The statistic associated with the \code{"overall"} value
in this column is calculated using the entire solution
(including all management zones if there are multiple zones).
If multiple management zones are present, then summary statistics
are also provided for each zone separately
(indicated using zone names).}

\item{feature}{\code{character} name of the feature.}

\item{total_amount}{\code{numeric} total amount of each feature available
in the entire conservation planning problem
(not just planning units selected within the solution).
It is calculated as the sum of the feature data,
supplied when creating a \code{\link[=problem]{problem()}} object
(e.g. presence/absence values).}

\item{absolute_held}{\code{numeric} total amount of each feature secured within
the solution. It is calculated as the sum of the feature data,
supplied when creating a \code{\link[=problem]{problem()}} object
(e.g. presence/absence values), weighted by the status of each
planning unit in the solution (e.g. selected or not for prioritization).}

\item{relative_held}{\code{numeric} proportion of
each feature secured within the solution. It is calculated
by dividing values in the \code{"absolute_held"} column by those in the
\code{"total_amount"} column.}

}
}
\description{
Calculate how well features are represented by a solution
to a conservation planning \code{\link[=problem]{problem()}}.
These summary statistics are reported for each and every feature,
and each and every zone, within a conservation planning problem.
}
\section{Solution format}{

Broadly speaking, the argument to \code{solution} must be in the same format as
the planning unit data in the argument to \code{x}.
Further details on the correct format are listed separately
for each of the different planning unit data formats:

\describe{

\item{\code{x} has \code{numeric} planning units}{The argument to \code{solution} must be a
\code{numeric} vector with each element corresponding to a different planning
unit. It should have the same number of planning units as those
in the argument to \code{x}. Additionally, any planning units missing
cost (\code{NA}) values should also have missing (\code{NA}) values in the
argument to \code{solution}.
}

\item{\code{x} has \code{matrix} planning units}{The argument to \code{solution} must be a
\code{matrix} vector with each row corresponding to a different planning
unit, and each column correspond to a different management zone.
It should have the same number of planning units and zones
as those in the argument to \code{x}. Additionally, any planning units
missing cost (\code{NA}) values for a particular zone should also have a
missing (\code{NA}) values in the argument to \code{solution}.
}

\item{\code{x} has \code{\linkS4class{Raster}} planning units}{The argument to \code{solution}
be a \code{\linkS4class{Raster}} object where different grid cells (pixels) correspond
to different planning units and layers correspond to
a different management zones. It should have the same dimensionality
(rows, columns, layers), resolution, extent, and coordinate reference
system as the planning units in the argument to \code{x}. Additionally,
any planning units missing cost (\code{NA}) values for a particular zone
should also have missing (\code{NA})  values in the argument to \code{solution}.
}

\item{\code{x} has \code{data.frame} planning units}{The argument to \code{solution} must
be a \code{data.frame} with each column corresponding to a different zone,
each row corresponding to a different planning unit, and cell values
corresponding to the solution value. This means that if a \code{data.frame}
object containing the solution also contains additional columns, then
these columns will need to be subsetted prior to using this function
(see below for example with \code{\link[sf:sf]{sf::sf()}} data).
Additionally, any planning units missing cost
(\code{NA}) values for a particular zone should also have missing (\code{NA})
values in the argument to \code{solution}.
}

\item{\code{x} has \code{\linkS4class{Spatial}} planning units}{The argument to \code{solution}
must be a \code{\linkS4class{Spatial}} object with each column corresponding to a
different zone, each row corresponding to a different planning unit, and
cell values corresponding to the solution value. This means that if the
\code{\linkS4class{Spatial}} object containing the solution also contains additional
columns, then these columns will need to be subsetted prior to using this
function (see below for example with \code{\link[sf:sf]{sf::sf()}} data).
Additionally, the argument to \code{solution} must also have the same
coordinate reference system as the planning unit data.
Furthermore, any planning units missing cost
(\code{NA}) values for a particular zone should also have missing (\code{NA})
values in the argument to \code{solution}.
}

\item{\code{x} has \code{\link[sf:sf]{sf::sf()}} planning units}{The argument to \code{solution} must be
a \code{\link[sf:sf]{sf::sf()}} object with each column corresponding to a different
zone, each row corresponding to a different planning unit, and cell values
corresponding to the solution value. This means that if the
\code{\link[sf:sf]{sf::sf()}} object containing the solution also contains additional
columns, then these columns will need to be subsetted prior to using this
function (see below for example).
Additionally, the argument to \code{solution} must also have the same
coordinate reference system as the planning unit data.
Furthermore, any planning units missing cost
(\code{NA}) values for a particular zone should also have missing (\code{NA})
values in the argument to \code{solution}.
}
}
}

\examples{
# set seed for reproducibility
set.seed(500)

# load data
data(sim_pu_raster, sim_pu_polygons, sim_pu_zones_sf, sim_features,
     sim_pu_zones_stack, sim_features_zones)

# create a simple conservation planning dataset so we can see exactly
# how feature representation is calculated
pu <- data.frame(id = seq_len(10), cost = c(0.2, NA, runif(8)),
                 spp1 = runif(10), spp2 = c(rpois(9, 4), NA))

# create problem
p1 <- problem(pu, c("spp1", "spp2"), cost_column = "cost") \%>\%
      add_min_set_objective() \%>\%
      add_relative_targets(0.1) \%>\%
      add_binary_decisions() \%>\%
      add_default_solver(verbose = FALSE)

# create a solution
# specifically, a data.frame with a single column that contains
# binary values indicating if each planning units was selected or not
s1 <- data.frame(s = c(1, NA, rep(c(1, 0), 4)))
print(s1)

# calculate feature representation
r1 <- eval_feature_representation_summary(p1, s1)
print(r1)

# let's verify that feature representation calculations are correct
# by manually performing the calculations and compare the results with r1
## calculate total amount for each feature
print(setNames(
  c(sum(pu$spp1, na.rm = TRUE),
    sum(pu$spp2, na.rm = TRUE)),
  c("spp1", "spp2")))

## calculate absolute amount held for each feature
print(setNames(
  c(sum(pu$spp1 * s1$s, na.rm = TRUE),
    sum(pu$spp2 * s1$s, na.rm = TRUE)),
  c("spp1", "spp2")))

## calculate relative amount held for each feature
print(setNames(
  c(sum(pu$spp1 * s1$s, na.rm = TRUE) / sum(pu$spp1, na.rm = TRUE),
    sum(pu$spp2 * s1$s, na.rm = TRUE) / sum(pu$spp2, na.rm = TRUE)),
  c("spp1", "spp2")))

\dontrun{
# solve the problem using an exact algorithm solver
s1_2 <- solve(p1)
print(s1_2)

# calculate feature representation in this solution
r1_2 <- eval_feature_representation_summary(
  p1, s1_2[, "solution_1", drop = FALSE])
print(r1_2)

# build minimal conservation problem with raster data
p2 <- problem(sim_pu_raster, sim_features) \%>\%
      add_min_set_objective() \%>\%
      add_relative_targets(0.1) \%>\%
      add_binary_decisions() \%>\%
      add_default_solver(verbose = FALSE)

# solve the problem
s2 <- solve(p2)

# print solution
print(s2)

# calculate feature representation in the solution
r2 <- eval_feature_representation_summary(p2, s2)
print(r2)

# plot solution
plot(s2, main = "solution", axes = FALSE, box = FALSE)
}
# build minimal conservation problem with polygon (Spatial) data
p3 <- problem(sim_pu_polygons, sim_features, cost_column = "cost") \%>\%
      add_min_set_objective() \%>\%
      add_relative_targets(0.1) \%>\%
      add_binary_decisions() \%>\%
      add_default_solver(verbose = FALSE)
\dontrun{
# solve the problem
s3 <- solve(p3)

# print first six rows of the attribute table
print(head(s3))

# calculate feature representation in the solution
r3 <- eval_feature_representation_summary(p3, s3[, "solution_1"])
print(r3)

# plot solution
spplot(s3, zcol = "solution_1", main = "solution", axes = FALSE, box = FALSE)
}
# build multi-zone conservation problem with raster data
p4 <- problem(sim_pu_zones_stack, sim_features_zones) \%>\%
      add_min_set_objective() \%>\%
      add_relative_targets(matrix(runif(15, 0.1, 0.2), nrow = 5,
                                  ncol = 3)) \%>\%
      add_binary_decisions() \%>\%
      add_default_solver(verbose = FALSE)
\dontrun{
# solve the problem
s4 <- solve(p4)

# print solution
print(s4)

# calculate feature representation in the solution
r4 <- eval_feature_representation_summary(p4, s4)
print(r4)

# plot solution
plot(category_layer(s4), main = "solution", axes = FALSE, box = FALSE)
}

# build multi-zone conservation problem with polygon (sf) data
p5 <- problem(sim_pu_zones_sf, sim_features_zones,
              cost_column = c("cost_1", "cost_2", "cost_3")) \%>\%
      add_min_set_objective() \%>\%
      add_relative_targets(matrix(runif(15, 0.1, 0.2), nrow = 5,
                                  ncol = 3)) \%>\%
      add_binary_decisions() \%>\%
      add_default_solver(verbose = FALSE)
\dontrun{
# solve the problem
s5 <- solve(p5)

# print first six rows of the attribute table
print(head(s5))

# calculate feature representation in the solution
r5 <- eval_feature_representation_summary(
  p5, s5[, c("solution_1_zone_1", "solution_1_zone_2", "solution_1_zone_3")])
print(r5)

# create new column representing the zone id that each planning unit
# was allocated to in the solution
s5$solution <- category_vector(
  s5[, c("solution_1_zone_1", "solution_1_zone_2", "solution_1_zone_3")])
s5$solution <- factor(s5$solution)

# plot solution
plot(s5[, "solution"])
}
}
\seealso{
\code{\link[=problem]{problem()}}, \link{summaries}.
}
