% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/power_marginaleffect.R
\name{power_marginaleffect}
\alias{power_marginaleffect}
\title{Power approximation for estimating marginal effects in GLMs}
\usage{
power_marginaleffect(
  response,
  predictions,
  target_effect,
  exposure_prob,
  var1 = NULL,
  kappa1_squared = NULL,
  estimand_fun = "ate",
  estimand_fun_deriv0 = NULL,
  estimand_fun_deriv1 = NULL,
  inv_estimand_fun = NULL,
  margin = estimand_fun(1, 1),
  alpha = 0.025,
  tolerance = 0.001,
  verbose = options::opt("verbose"),
  ...
)
}
\arguments{
\item{response}{the response variable from comparator participants}

\item{predictions}{predictions of the response}

\item{target_effect}{a \code{numeric} minimum effect size that we should be able to detect.
See more in details.}

\item{exposure_prob}{a \code{numeric} with the probability of being in
"group 1" (rather than group 0). See more in details.}

\item{var1}{a \code{numeric} variance of the potential outcome corresponding to group 1,
or a \code{function} with a single argument meant to obtain \code{var1} as a tranformation
of the variance of the potential outcome corresponding to group 0.
See more in details.}

\item{kappa1_squared}{a \code{numeric} mean-squared error from predicting potential
outcome corresponding to group 1, or a \code{function} with a single arguments meant
to obtain \code{kappa1_squared} as a transformation of the MSE in group 0.
See more in details.}

\item{estimand_fun}{a \code{function} with arguments \code{psi1} and \code{psi0} specifying
the estimand. Alternative, specify "ate" or "rate_ratio" as a \code{character}
to use one of the default estimand functions. See
more details in the "Estimand" section of \link{rctglm}.}

\item{estimand_fun_deriv0}{a \code{function} specifying the derivative of \code{estimand_fun} wrt. \code{psi0}. As a default
the algorithm will use symbolic differentiation to automatically find the derivative from \code{estimand_fun}}

\item{estimand_fun_deriv1}{a \code{function} specifying the derivative of \code{estimand_fun} wrt. \code{psi1}. As a default
the algorithm will use symbolic differentiation to automatically find the derivative from \code{estimand_fun}}

\item{inv_estimand_fun}{(optional) a \code{function} with arguments \code{psi0} and
\code{target_effect}, so \code{estimand_fun(psi1 = y, psi0 = x) = z} and
\code{inv_estimand_fun(psi0 = x, target_effect = z) = y} for all x, y, z.
If left as \code{NULL}, the inverse will be found automatically}

\item{margin}{a \code{numeric} superiority margin. As a default, the \code{estimand_fun}
is evaluated with the same counterfactual means \code{psi1} and \code{psi0}, corresponding
to a superiority margin assuming no difference (fx. 0 for ATE and 1 for rate ratio).}

\item{alpha}{a \code{numeric} significance level. Due to regulatory guidelines when
using a one-sided test, half the specified significance level is used.
Thus, for standard significance level of 5\%, the default is \code{alpha = 0.025}.}

\item{tolerance}{passed to \link{all.equal} when comparing calculated \code{target_effect}
from derivations and given \code{target_effect}.}

\item{verbose}{\code{numeric} verbosity level. Higher values means more information is
printed in console. A value of 0 means nothing is printed to console during
execution (Defaults to \code{2}, overwritable using option 'postcard.verbose' or environment variable 'R_POSTCARD_VERBOSE')}

\item{...}{arguments passed to \verb{[stats::uniroot]}, which is used to find the
inverse of \code{estimand_fun}, when \code{inv_estimand_fun} is \code{NULL}.}
}
\value{
a \code{numeric} with the estimated power.
}
\description{
The functions implements the algorithm for power estimation described in
\href{https://arxiv.org/abs/2503.22284}{Powering RCTs for marginal effects with GLMs using prognostic score adjustment}
by Højbjerre-Frandsen et. al (2025). See a bit more context in details and
all details in reference.
}
\details{
The reference in the description shows in its "Prospective power" section a
derivation of a variance bound
\deqn{
v_\infty^{\uparrow 2} = r_0'^{\, 2}\sigma_0^2+
r_1'^{\, 2}\sigma_1^2+
\pi_0\pi_1\left(\frac{|r_0'|\kappa_0}{\pi_0} +
\frac{|r_1'|\kappa_1}{\pi_1} \right)^2
}

where \eqn{r_a'} is the derivative of the \code{estimand_fun} with respect to
\eqn{\Psi_a}, \eqn{\sigma_a^2} is the variance of the potential outcome corresponding to
group \eqn{a}, \eqn{\pi_a} is the probability of being assigned to group \eqn{a},
and \eqn{\kappa_a} is the expected mean-squared error when predicting the
potential outcome corresponding to group \eqn{a}.

The variance bound is then used for calculating a lower bound of the power using
the distributions corresponding to the null and alternative hypotheses
\eqn{\mathcal{H}_0: \hat{\Psi} \sim F_0 = \mathcal{N}(\Delta ,v_\infty^{\uparrow 2} / n)}
and
\eqn{\mathcal{H}_1: \hat{\Psi} \sim F_1 = \mathcal{N}(\Psi,v_\infty^{\uparrow 2} / n)}.
See more details in the reference.
\subsection{Relating arguments to formulas}{
\itemize{
\item \code{response}: Used to obtain both \eqn{\sigma_0^2} (by taking the sample
variance of the response) and \eqn{\kappa_0}.
\item \code{predictions}: Used when calculating the MSE \eqn{\kappa_0}.
\item \code{var1}: \eqn{\sigma_1^2}. As a default, chosen to be the same as
\eqn{\sigma_0^2}. Can specify differently through this argument fx. by
\itemize{
\item Inflating or deflating the value of \eqn{\sigma_0^2} by a scalar according
to prior beliefs. Fx. specify \code{var1 = function(x) 1.2 * x} to inflate
\eqn{\sigma_0^2} by 1.2.
\item If historical data is available for group 1, an estimate of the variance
from that data can be provided here.
}
\item \code{kappa1_squared}: \eqn{\kappa_1}. Same as for \code{var1}, default is to assume
the same value as \code{kappa0_squared}, which is found by using the \code{response}
and \code{predictions} vectors. Adjust the value according to prior beliefs if
relevant.
\item \code{target_effect}: \eqn{\Psi}.
\item \code{exposure_prob}: \eqn{\pi_1}
}
}
}
\examples{
# Generate a data set to use as an example
n <- 100
exposure_prob <- .5

dat_gaus <- glm_data(Y ~ 1+2*X1-X2+3*A+1.6*A*X2,
                X1 = rnorm(n),
                X2 = rgamma(n, shape = 2),
                A = rbinom(n, size = 1, prob = exposure_prob),
                family = gaussian())

# ---------------------------------------------------------------------------
# Obtain out-of-sample (OOS) prediction using glm model
# ---------------------------------------------------------------------------
gaus1 <- dat_gaus[1:(n/2), ]
gaus2 <- dat_gaus[(n/2+1):n, ]

glm1 <- glm(Y ~ X1 + X2 + A, data = gaus1)
glm2 <- glm(Y ~ X1 + X2 + A, data = gaus2)
preds_glm1 <- predict(glm2, newdata = gaus1, type = "response")
preds_glm2 <- predict(glm1, newdata = gaus2, type = "response")
preds_glm <- c(preds_glm1, preds_glm2)

# Obtain power
power_marginaleffect(
  response = dat_gaus$Y,
  predictions = preds_glm,
  target_effect = 2,
  exposure_prob = exposure_prob
)

# ---------------------------------------------------------------------------
# Get OOS predictions using discrete super learner and adjust variance
# ---------------------------------------------------------------------------
learners <- list(
  mars = list(
    model = parsnip::set_engine(
      parsnip::mars(
        mode = "regression", prod_degree = 3
      ),
      "earth"
    )
 ),
    lm = list(
      model = parsnip::set_engine(
        parsnip::linear_reg(),
        "lm"
      )
    )
)
lrnr1 <- fit_best_learner(preproc = list(mod = Y ~ X1 + X2 + A),
                          data = gaus1,
                          learners = learners)
lrnr2 <- fit_best_learner(preproc = list(mod = Y ~ X1 + X2 + A),
                          data = gaus2,
                          learners = learners)
preds_lrnr1 <- dplyr::pull(predict(lrnr2, new_data = gaus1))
preds_lrnr2 <- dplyr::pull(predict(lrnr1, new_data = gaus2))
preds_lrnr <- c(preds_lrnr1, preds_lrnr2)

# Estimate the power AND adjust the assumed variance in the "unknown"
# group 1 to be 20\% larger than in group 0
power_marginaleffect(
  response = dat_gaus$Y,
  predictions = preds_lrnr,
  target_effect = 2,
  exposure_prob = exposure_prob,
  var1 = function(var0) 1.2 * var0
)

}
\seealso{
See \link{power_linear} for power approximation functionalities for linear models.
}
