% Generated by roxygen2 (4.0.2): do not edit by hand
\name{spsannPPL}
\alias{objPairs}
\alias{objPoints}
\alias{pairsPerLag}
\alias{pointsPerLag}
\alias{spsannPPL}
\title{Optimization of spatial samples for variogram estimation}
\usage{
spsannPPL(points, candidates, lags = 7, lags.type = "exponential",
  lags.base = 2, cutoff = NULL, criterion = "distribution",
  pre.distri = NULL, x.max, x.min, y.max, y.min, iterations = 10000,
  acceptance = list(initial = 0.99, cooling = iterations/10),
  stopping = list(max.count = iterations/10), plotit = TRUE, boundary,
  progress = TRUE, verbose = TRUE)

objPoints(points, lags = 7, lags.type = "exponential", lags.base = 2,
  cutoff = NULL, criterion = "distribution", pre.distri = NULL)

pointsPerLag(points, lags = 7, lags.type = "exponential", lags.base = 2,
  cutoff = NULL)
}
\arguments{
\item{points}{data.frame or matrix with three columns: 1) the identification
of each point, 2) the x coordinates of the set of points, and 3) the y
coordinates of the set of points. The coordinates must be projected. This is
the set of points to be optimized. \code{points} must be a subset of
\code{candidates}. See \sQuote{Details} for more information.}

\item{candidates}{data.frame or matrix with the candidate locations for the
sample points. See \sQuote{Details} for more information.}

\item{lags}{Integer value defining the number of lag distance classes.
Alternatively, a vector of numeric values defining the lower and upper limits
of each lag distance class. The lowest value should be different from zero,
e.g. 0.0001. Defaults to \code{lags = 7}.}

\item{lags.type}{Character value defining the type of lag distance classes.
Available options are \code{"equidistant"}, for equidistant lag distance
classes, and \code{"exponential"}, for exponentially spaced lag distance
classes. Defaults to \code{lags.type = "exponential"}. See \sQuote{Details}
for more information.}

\item{lags.base}{Numeric value defining the creation of exponentially spaced
lag distance classes. Defaults to \code{lags.base = 2}. See \sQuote{Details}
for more information.}

\item{cutoff}{Numeric value defining the maximum distance up to which
lag distance classes are created. Used only when lag distance classes are
not defined. See \sQuote{Details} for more information.}

\item{criterion}{Character value defining the measure that should be
returned to describe the energy state of the current system configuration.
Available options are \code{"minimum"} and \code{"distribution"}. The first
returns the minimum number of points or point-pairs observed over all lag
distance classes. The second returns the sum of the differences between a
pre-specified distribution and the observed distribution of counts of points
or point-pairs per lag distance class. Defaults to
\code{objective = "minimum"}. See \sQuote{Details} for more information.}

\item{pre.distri}{Vector of numeric values used to pre-specify the
distribution of points or point-pair with which the observed counts of points
or point-pairs per lag distance class is compared. Used only when
\code{criterion = "distribution"}. Defaults to a uniform distribution. See
\sQuote{Details} for more information.}

\item{x.max,x.min,y.max,y.min}{The minimum and maximum quantity of random
noise to be added to the x and y coordinates. The minimum quantity should be
equal to, at least, the minimum distance between two neighboring candidate
locations. The units are the same as of the coordinates. See \sQuote{Details}
for more information.}

\item{iterations}{Integer value defining the maximum number of iterations
that should be used for the optimization. See \sQuote{Details} for more
information.}

\item{acceptance}{List with two sub-arguments: \code{initial} and
\code{cooling}. \code{initial} is a numeric value between 0 and 1 defining
the initial acceptance probability. Defaults to \code{initial = 0.99}.
\code{cooling} is a numeric value defining the exponential factor by with
the acceptance probability decreases at each iteration. Defaults to
\code{cooling = iterations / 10}. See \sQuote{Details} for more
information.}

\item{stopping}{A list with one sub-argument: \code{max.count}.
\code{max.count} is an integer value defining the maximum allowable number
of iterations without improvement of the objective function value. This is
also known as the freezing criterion. Defaults to
\code{max.count = iterations / 10}. See \sQuote{Details} for more
information.}

\item{plotit}{Logical value for ploting the optimization results. This
includes a) the progress of the objective function values and acceptance
probabilities, and b) the original points, the perturbed points and the
progress of the maximum perturbation in the x and y coordinates. The plots
are updated at each 10 iterations. The boundary of the spatial domain is
passed using the argument \code{boundary}. Defaults to \code{plotit = TRUE}.}

\item{boundary}{SpatialPolygon defining the boundary of the spatial domain.
It is mandatory if \code{plotit = TRUE}.}

\item{progress}{Logical value for printing a progress bar. Defaults to
\code{progress = TRUE}.}

\item{verbose}{Logical value for printing messages about the progress of the
optimization.}
}
\value{
\code{pointsPerLag} and \code{pairsPerLag} return a data.frame with three
columns: a) the lower and b) upper limits of each lag distance class, and
c) the number of points or point-pairs per lag distance class.

\code{objPoints} and \code{objPairs} return a numeric value depending on the
choice of \code{criterion}. If \code{criterion = "distribution"}, the sum of
the differences between the pre-specified and observed distribution of counts
of points or point-pairs per lag distance class. If
\code{criterion = "minimum"}, the inverse of the minimum count of points or
point pairs over all lag distance classes multiplied by a constant.
}
\description{
Funtion to optimize spatial samples for variogram estimation using spatial
simulated annealing. The criterion used in the optimization is the number of
points per lag distance class. Functions to counts the number of points or
point pairs per lag distance class. Functions to compute the deviation of
the observed distribution of counts from a pre-specified distribution.
Functions to compute the minimum number of points or point pairs observed
over all lag distance classes.
}
\details{
\subsection{Distances}{
Euclidean distances between points are calculated. This computation requires
the coordinates to be projected. The user is responsible for making sure that
 this requirement is attained.
}
\subsection{Distribution}{
Using the default uniform distribution means that the number of
\strong{point-pairs} per lag distance class is equal to
\eqn{n \times (n - 1) / (2 \times lag)}, where \eqn{n} is the total number
of points in \code{points}, and \eqn{lag} is the number of lag distance
classes.

Using the default uniform distribution means that the number of
\strong{points} per lag distance class is equal to the total number of points
in \code{points}. This is the same as expecting that each point contributes
to every lag distance class.

Distributions other that the available options can be easily implemented
changing the arguments \code{lags}, \code{lags.base} and \code{pre.distri}.
}
\subsection{Type of lags}{
Two types of lag distance classes can be created by default. The first
(\code{lags.type = "equidistant"}) are evenly spaced lags. They are created
by simply dividing the distance interval from 0.0001 to \code{cutoff} by the
required number of lags. The minimum value of 0.0001 guarantees that a point
does not form a pair with itself.

The second type (\code{lags.type = "exponential"}) of lag distance classes
is defined by exponential spacings. The spacings are defined by the base
\eqn{b} of the exponential expression \eqn{b^n}, where \eqn{n} is the
required number of lags. The base is defined using argument \code{lags.base}.
For example, the default \code{lags.base = 2} creates lags that are
sequentially defined as half of the immediately preceding larger lag. If
\code{cutoff = 100} and \code{lags = 4}, the upper limits of the lag distance
classes will be

\verb{
> 100 / (2 ^ c(1:4))
[1] 50.00 25.00 12.50  6.25
}
}
\subsection{Criteria}{
The functions \code{objPoints} and \code{objPairs} (to be implemented) and
were designed to be used in spatial simulated annealing to optimize spatial
sample configurations. Both of them have two criteria implemented. The first
is called using \code{criterion = "distribution"} and is used to minimize the
sum of differences between a pre-specified distribution and the observed
distribution of points or point-pairs per lag distance class.

Consider that we aim at having the following distribution of points per lag:

\code{desired <- c(10, 10, 10, 10, 10)},

and that the observed distribution of points per lag is the following:

\code{observed <- c(1, 2, 5, 10, 10)}.

The objective at each iteration of the optimization will be to match the two
distributions. This criterion is of the same type as the one proposed by
Warrick and Myers (1987).

The second criterion is called using \code{criterion = "minimum"}. It
corresponds to maximizing the minimum number of points or point-pairs
observed over all lag distance classes. Consider we observe the following
distribution of points per lag in the first iteration:

\code{observed <- c(1, 2, 5, 10, 10)}.

The objective in the next iteration will be to increase the number of points
in the first lag (\eqn{n = 1}). Consider we then have the following resulting
distribution:

\code{resulting <- c(5, 2, 5, 10, 10)}.

Now the objective will be to increse the number of points in the second lag
(\eqn{n = 2}). The optimization continues until it is not possible to
increase the number of points in any of the lags, that is, when:

\code{distribution <- c(10, 10, 10, 10, 10)}.

This shows that the result of using \code{criterion = "minimum"} is similar
to using \code{criterion = "distribution"}. However, the resulting sample
pattern can be significantly different. The running time of each iteration
can be a bit longer when using \code{criterion = "distribution"}, but since
it is a more sensitive criteria (it takes all lags into account), convergence
is likely to be attained with a smaller number of iterations. Note that this
also depends on the other parameters passed to the optimization algorithm.

It is important to note that using the first criterion
(\code{"distribution"}) in simulated annealing corresponds to a
\strong{minimization} problem. On the other hand, using the second criterion
(\code{"minimum"}) would correspond to a \strong{maximization} problem. We
solve this inconsistency substituting the criterion that has to be maximized
by its inverse. For conveninence we multiply the resulting value by a
constant (i.e. \eqn{c / x + 1}, where \code{c} is the number of points and
\code{x} is the criterion value). This procedure allows us to define both
problems as minimization problems.
}
\subsection{Utopia and nadir points}{
Knowledge of the utopia and nadir points can help in the construction of
multi-objective optimization problems.

When \code{criterion = "distribution"}, the \strong{utopia}
(\eqn{f^{\circ}_{i}}) point is exactly zero (\eqn{f^{\circ}_{i} = 0}). When
\code{criterion = "minimum"}, the utopia point is approximately 1 (0.9)
(\eqn{f^{\circ}_{i} \sim 1}). It can be calculated using the equation
\eqn{n / n + 1}, where \code{n} is the number of points
(\code{objPoints}), or the number point-pairs divided by the number of lag
distance classes (\code{objPairs}).

THIS SECTION ABOUT THE NADIR POINT HAS TO BE REVIEWED!
The \strong{nadir} (\eqn{f^{max}_{i}}) point depends on a series of
elements. For instance, when \code{criterion = "distribution"}, if the
desired distribution of point or point-pairs per lag distance class is
\code{pre.distribution <- c(10, 10, 10, 10, 10)}, the worst case scenario
would be to have all points or point-pairs in a single lag distance class,
that is, \code{obs.distribution <- c(0, 50, 0, 0, 0)}. In this case, the
nadir point is equal to the sum of the differences between the two
distributions:

\code{sum((c(10, 10, 10, 10, 10) - c(0, 50, 0, 0, 0)) ^ 2) = 2000}.

When \code{objective = "minimum"}, the nadir point is equal to
\eqn{f^{max}_{i} = n / 0 + 1 = n}.
}
}
\note{
Use \code{lags = 1} with \code{pointsPerLag} and \code{pairsPerLag} to check
that the functions are working correctly. They should return the total number
of points in \code{points} and the total possible number of point-pairs
\eqn{n \times (n - 1) / 2}, respectively.

Some of the solutions used to build this function were found in the source
code of the R-packages \strong{intamapInteractive} and \pkg{SpatialTools}.
As such, the authors of those packages (Edzer Pebesma
<\email{edzer.pebesma@uni-muenster.de}>, Jon Skoien
<\email{jon.skoien@gmail.com}>, Joshua French
<\email{joshua.french@ucdenver.edu}>) are entitled \sQuote{contributors} to
the R-package \pkg{pedometrics}.
}
\section{Spatial simulated annealing}{

\subsection{Search graph}{
The search graph corresponds to the set of effective candidate locations for
a point being jittered in a given iteration. The size of the search graph,
i.e. the maximum distance that a point can be moved around, is correlated
with the concept of \strong{temperature}. A larger search graph is equivalent
to higher temperatures, which potentially result in more movement or
\dQuote{agitation} of the set of points or \dQuote{particles}.

The current implementation of spatial simulated annealing uses a
\strong{linear cooling schedule} depending upon the number of iterations to
control the size of the search graph. The equations are as follows:

\verb{
x.max.b <- x.max.a - k / iterations * (x.max.a - x.min)
y.max.b <- y.max.a - k / iterations * (y.max.a - y.min)
}

where \code{x.max.a} and \code{y.max.a} are the maximum allowed shift in the
x and y coordinates in the current iteration, \code{x.min} and \code{y.min}
are the minimum required shift in the x and y coordinates, and \code{x.max.b}
and \code{y.max.b} are the maximum allowed shift in the x and y coordinates
in the next iteration. \code{iterations} is the total number of iterations
and \code{k} is the current iteration.
}
\subsection{Acceptance probability}{
The acceptance probability is the chance of accepting a new system
configuration that is worse than the current system configuration. The
concept of acceptance probability is related with that of
\strong{temperature}. A higher acceptance probability is equivalent to higher
temperatures, which potentially result in more movement or
\dQuote{agitation} of the set of points or \dQuote{particles}.

Using a low initial acceptance probability turns the spatial simulated
annealing into a \emph{greedy} algorithm. It will converge in a shorter time,
but the solution found is likely to be a local optimum instead of the global
optimum. Using a high initial acceptance probability (\code{>0.8}) usually is
the wisest choice.

An \strong{exponential cooling schedule} depending upon the number of
iterations is used in the current implementation of the spatial simulated
annealing to control the acceptance probability. The acceptance probability
at each iteration is calculates as follows:

\verb{actual_prob <- acceptance$initial * exp(-k / acceptance$cooling)}

where \code{actual_prob} is the acceptance probability at the \code{k}-th
iteration, \code{acceptance$initial} is the initial acceptance probability,
and \code{acceptance$cooling} is the exponential cooling factor.
}
\subsection{Starting system configuration}{
Unidimensional criterion such as the number of points per lag distance class
are dependent on the starting system configuration by definition. This means
that, depending on the parameters passed to the spatial simulated annealing
algorithm, many points will likely to stay close to their starting positions.
It would be reasonable to use a starting system configuration that is close
to the global optimal, but such thing is not feasible.

Increasing the initial acceptance probability does not guarantee the
independence from the starting system configuration. The most efficient
option in the current implementation of the spatial simulated annealing
algorithm is to start using the entire spatial domain as search graph. This
is set using the interval of the x and y coodinates to set \code{x.max}
and \code{y.max} (See above).

An alternative is to start jittering (randomly perturbing) several points at
a time and use a cooling schedule to \strong{exponentially} decrease the
number of points jittered at each iteration. The current implementation of
the spatial simulated annealing does not explore such alternative. The
cooling schedule would be as follows:

\verb{
new.size <- round(c(old.size - 1) * exp(-k / size.factor) + 1)
}

where \code{old.size} and \code{new.size} are the number of points jittered
in the previous and next iterations, \code{size.factor} is the cooling
parameter, and \code{k} is the number of the current iteration. The larger
the difference between the starting system configuration and the global
optimum, the larger the number of points that would need to be jittered in
the first iterations. This will usually increase the time spent on the first
iterations.
}
\subsection{Number of iterations}{
The number of iterations has a large influence on the performance of the
spatial simulated annealing algorithm. The larger the number of possible
system configurations, the higher should the number of iterations be.

The number of possible system configurations increases with:
\itemize{
\item a high initial acceptance probability
\item the use of an infinite set of candidate locations
\item the use of a very dense finite set of candidate locations
}
}
}
\examples{
require(sp)
data(meuse)
meuse <- as.matrix(meuse[, 1:2])
meuse <- matrix(cbind(c(1:dim(meuse)[1]), meuse), ncol = 3)
pointsPerLag(meuse, cutoff = 1000)
objPoints(meuse, cutoff = 1000)
}
\author{
Alessandro Samuel-Rosa \email{alessandrosamuelrosa@gmail.com}

Gerard Heuvelink \email{gerard.heuvelink@wur.nl}
}
\references{
Bresler, E.; Green, R. E. \emph{Soil parameters and sampling scheme for
characterizing soil hydraulic properties of a watershed}. Honolulu:
University of Hawaii at Manoa, p. 42, 1982.

Marler, R. T.; Arora, J. S. Function-transformation methods for
multi-objective optimization. \emph{Engineering Optimization}. v. 37, p.
551-570, 2005.

Russo, D. Design of an optimal sampling network for estimating the variogram.
\emph{Soil Science Society of America Journal}. v. 48, p. 708-716, 1984.

Truong, P. N.; Heuvelink, G. B. M.; Gosling, J. P. Web-based tool for expert
elicitation of the variogram. \emph{Computers and Geosciences}. v. 51, p.
390-399, 2013.

Warrick, A. W.; Myers, D. E. Optimization of sampling locations for variogram
calculations. \emph{Water Resources Research}. v. 23, p. 496-500, 1987.
}
\seealso{
\code{\link[stats]{dist}}.
}
\concept{
simulated annealing
}
\keyword{optimize}
\keyword{spatial}

