\name{tpsPower}
\Rdversion{1.7}
\alias{tpsPower}
\title{
	Simulation-based estimation of power for the two-phase study design
}
\description{
  Monte Carlo based estimation of statistical power for estimators of the components of a logistic regression model, based on balanced two-phase and case-control study designs (Breslow and Chatterjee, 1999; Prentice and Pykle, 1979).
}
\usage{
tpsPower(B=1000, betaTruth, X, N, strata, expandX="all", etaTerms=NULL,
         nII, alpha=0.05, digits=1, betaNames=NULL,
         monitor=NULL, cohort=TRUE, NI=NULL)
}
\arguments{
  \item{B}{
    The number of datasets generated by the simulation.
  }
  \item{betaTruth}{
    Regression coefficients from the logistic regression model.
  }
  \item{X}{
    Design matrix for the logistic regression model. The first column should correspond to intercept. For each exposure, the baseline group should be coded as 0, the first level as 1, and so on.
  }
  \item{N}{
    A numeric vector providing the sample size for each row of the design matrix, \code{X}.
  }
  \item{strata}{
    A numeric vector indicating which columns of the design matrix, \code{X}, are used to form the phase I stratification variable. \code{strata=1} specifies the intercept and is, therefore, equivalent to a case-control study. \code{strata=0} is not permitted in tpsPower(), although multiple two-phase stratifications can be investigated with tpsSim().
  }
  \item{expandX}{
    Character vector indicating which columns of \code{X} to expand as a series of dummy variables. Useful when at least one exposure is continuous (and should not be expanded). Default is `all'; other options include `none' or character vector of column names. See Details, below.
  }
  \item{etaTerms}{
    Character vector indicating which columns of \code{X} are to be included in the model. See Details, below.
  }
  \item{nII}{
    A numeric value indicating the phase II sample size. If a vector is provided, separate simulations are run for each element.
  }
  \item{alpha}{
    Type I error rate assumed for the evaluation of coverage probabilities and power.
  }
  \item{digits}{
    Integer indicating the precision to be used for the output.
  }
  \item{betaNames}{
    An optional character vector of names for the regression coefficients, 
    \code{betaTruth}.
  }
  \item{monitor}{
    Numeric value indicating how often \code{tpsPower()} reports real-time progress on the simulation, as the \code{B} datasets are generated and evaluated. The default of \code{NULL} indicates no output.
  }
  \item{cohort}{
    Logical flag. TRUE indicates phase I is drawn as a cohort; FALSE indicates phase I is drawn as a case-control sample.
  }
  \item{NI}{
    A pair of integers providing the outcome-specific phase I sample
    sizes when the phase I data are drawn as a case-control sample. The
    first element corresponds to the controls and the second to the
    cases.
  }
}
\details{
A simulation study is performed to estimate power for various estimators
of \code{beta}:
\itemize{
  \item{}{(a) complete data maximum likelihood (CD)}
  \item{}{(b) case-control maximum likelihood (CC)}
  \item{}{(c) two-phase weighted likelihood (WL)}
  \item{}{(d) two-phase pseudo- or profile likelihood (PL)}
  \item{}{(e) two-phase maximum likelihood (ML)}
}
The overall simulation approach is the same as that described in \code{\link{tpsSim}}.

  In each case, power is estimated as the proportion of simulated datasets for which a hypothesis test of no effect is rejected.

  The correspondence between \code{betaTruth} and \code{X}, specifically the ordering of elements, is based on successive use of \code{\link{factor}} to each column of \code{X} which is expanded via the \code{expandX} argument. Each exposure that is expanded must conform to a {0, 1, 2, ...} integer-based coding convention.
  
  The \code{etaTerms} argument is useful when only certain columns in \code{X} are to be included in the model. In the context of the two-phase design, this might be the case if phase I stratifies on some surrogate exposure and a more detailed/accurate measure is to be included in the main model.

Only balanced designs are considered by \code{tpsPower()}. For unbalanced designs, power estimates can be obtained from \code{\link{tpsSim}}.

NOTE: In some settings, the current implementation of the ML estimator returns point estimates that do not satisfy the phase I and/or phase II constraints. If this is the case a warning is printed and the "fail" elements of the returned list is set to TRUE. An example of this is phenomenon is given the help file for \code{\link{tps}}. When this occurs, \code{tpsPower()} considers ML estimation for the particular dataset to have failed.

}
\value{
  \code{tpsPower()} returns an object of class "tpsPower", a list containing all the input arguments, as well as the following components:
    \item{betaPower}{
    	Power against the null hypothesis that the regression coefficient is zero for a Wald-based test with an \code{alpha} type I error rate.
    }
    \item{failed}{
    	A vector consisting of the number of datasets excluded from the power calculations (i.e. set to \code{NA}), for each simulation performed. For power calculations, the three reasons are: (1) lack of convergence indicated by \code{NA} point estimates returned by \code{\link{glm}} or \code{\link{tps}}; (2) lack of convergence indicated by \code{NA} standard error point estimates returned by \code{\link{glm}} or \code{\link{tps}}; and (3) for the ML estimator only, the phase I and/or phase II constraints are not satisfied.
    }
}
\note{
	A generic print method provides formatted output of the results.

	A generic plot function \code{\link{plotPower}} provides plots of powers against different sample sizes for each estimate of a regression coefficient.
}
\references{
Prentice, R. and Pyke, R. (1979) "Logistic disease incidence models and case-control studies." Biometrika 66:403-411.

Breslow, N. and Chatterjee, N. (1999) "Design and analysis of two phase studies with binary outcome applied to Wilms tumour prognosis." Applied Statistics 48:457-468.

Haneuse, S. and Saegusa, T. and Lumley, T. (2011) "osDesign: An R Package for the Analysis, Evaluation, and Design of Two-Phase and Case-Control Studies." Journal of Statistical Software, 43(11), 1-29.
}
\author{
  Sebastien Haneuse, Takumi Saegusa
}
\seealso{
  \code{\link{plotPower}}.
}
\examples{
##
data(Ohio)

## 
XM   <- cbind(Int=1, Ohio[,1:3])
fitM <- glm(cbind(Death, N-Death) ~ factor(Age) + Sex + Race, data=Ohio,
            family=binomial)
betaNamesM <- c("Int", "Age1", "Age2", "Sex", "Race")

## Power for the TPS design where phase I stratification is based on Race.
##
\dontrun{
tpsResult1 <- tpsPower(B=1000, beta=fitM$coef, X=XM, N=Ohio$N, strata=4,
                       nII=seq(from=100, to=1000, by=100),
                       betaNames=betaNamesM, monitor=100)
tpsResult1}

## Power for the TPS design where phase I stratification is based on Age
##   * consider the setting where the age coefficients are halved from
##     their observed true values
##   * the intercept is modified, accordingly, using the beta0() function
##
newBetaM      <- fitM$coef
newBetaM[2:3] <- newBetaM[2:3] / 2
newBetaM[1]   <- beta0(betaX=newBetaM[-1], X=XM, N=Ohio$N,
                       rhoY=sum(Ohio$Death)/sum(Ohio$N))
##
\dontrun{
tpsResult2 <- tpsPower(B=1000, beta=fitM$coef, X=XM, N=Ohio$N, strata=2,
                       nII=seq(from=100, to=500, by=50),
                       betaNames=betaNamesM, monitor=100)
tpsResult2}
}

