% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/frames_graph.R
\name{frames_graph}
\alias{frames_graph}
\title{Create frames of movement-environment interaction graphs for animation}
\usage{
frames_graph(
  m,
  r_list,
  r_times,
  r_type = "gradient",
  fade_raster = FALSE,
  crop_raster = TRUE,
  return_data = FALSE,
  graph_type = "flow",
  path_size = 1,
  path_legend = TRUE,
  path_legend_title = "Names",
  val_min = NULL,
  val_max = NULL,
  val_by = 0.1,
  verbose = T
)
}
\arguments{
\item{m}{\code{move} or \code{moveStack} of uniform time scale and time lag, e.g. prepared with \code{\link{align_move}} (recommended). May contain a column named \code{colour} to control path colours (see \code{details}).}

\item{r_list}{list of \code{raster} or \code{rasterStack}. Each list element referrs to the times given in \code{r_times}. Use single-layer \code{raster} objects for gradient or discrete data (see \code{r_type}). Use a  \code{rasterStack} containing three bands for RGB imagery (in the order red, green, blue).}

\item{r_times}{list of \code{POSIXct} times. Each list element represents the time of the corresponding element in \code{r_list}. Must be of same length as \code{r_list}.}

\item{r_type}{character, either \code{"gradient"} or \code{"discrete"}. Ignored, if \code{r_list} contains \code{rasterStacks} of three bands, which are treated as RGB.}

\item{fade_raster}{logical, if \code{TRUE}, \code{r_list} is interpolated over time based on \code{r_times}. If \code{FALSE}, \code{r_list} elements are assigned to those frames closest to the equivalent times in \code{r_times}.}

\item{crop_raster}{logical, whether to crop rasters in \code{r_list} to plot extent before plotting or not.}

\item{return_data}{logical, if \code{TRUE}, instead of a list of frames, a \code{data.frame} containing the values extracted from \code{r_list} per individual, location and time is returned. This \code{data.frame} can be used to create your own multi- or monotemporal \code{ggplot2} movement-environemnt interaction graphs.}

\item{graph_type}{character, defines the type of multi-temporal graph that should be drawn as frames. Currently supported graphs are:
\itemize{
   \item \code{"flow"}, a time flow graph with frame time on the x axis and values of the visited cell at x on the y axis per individual track
   \item \code{"hist"}, a cumulative histogram with cell values on the x axis and time-cumulative counts of visits on the y axis per individual track.
}}

\item{path_size}{numeric, size of each path.}

\item{path_legend}{logical, wether to add a path legend from \code{m} or not. Legend tracks and colours will be ordered by the tracks' temporal apperances, not by their order in \code{m}.}

\item{path_legend_title}{character, path legend title. Default is \code{"Names"}.}

\item{val_min}{numeric, minimum value of the value axis. If undefined, the minimum is collected automatically.}

\item{val_max}{numeric, maximum value of the value axis. If undefined, the maximum is collected automatically.}

\item{val_by}{numeric, increment of the value axis sequence. Default is 0.1. If \code{graph_type = "discrete"}, this value should be an integer of 1 or greater.}

\item{verbose}{logical, if \code{TRUE}, messages and progress information are displayed on the console (default).}
}
\value{
List of ggplot2 objects, each representing a single frame. If \code{return_data} is \code{TRUE}, a \code{data.frame} is returned (see \code{return_data}).
}
\description{
\code{frames_graph} creates a list of \code{ggplot2} graphs displaying movement-environment interaction. Each object represents a single frame. Each frame can be viewed or modified individually. The returned list of frames can be animated using \code{\link{animate_frames}}.
}
\details{
To later on side-by-side join spatial frames created using \code{\link{frames_spatial}} with frames created with \code{\link{frames_graph}} for animation,
equal inputs must have been used for both function calls for each of the arguments \code{m}, \code{r_list}, \code{r_times} and \code{fade_raster}.
}
\examples{
library(moveVis)
library(move)
library(ggplot2)

data("move_data", "basemap_data")
# align movement
m <- align_move(move_data, res = 4, unit = "mins")

r_list <- basemap_data[[1]]
r_times <- basemap_data[[2]]

\donttest{
# use the same inputs to create a non-spatial graph, e.g. a flow graph:
frames.gr <- frames_graph(m, r_list = r_list, r_times = r_times, r_type = "gradient",
                          fade_raster = TRUE, graph_type = "flow")
# take a look
frames.gr[[100]]

# make a histogram graph:
frames.gr <- frames_graph(m, r_list = r_list, r_times = r_times, r_type = "gradient",
                          fade_raster = TRUE, graph_type = "hist")
# change the value interval:
frames.gr <- frames_graph(m, r_list = r_list, r_times = r_times, r_type = "gradient",
                          fade_raster = TRUE, graph_type = "hist", val_by = 0.01)
                          
frames.gr[[100]]
# manipulate the labels, since now they are very dense:
# just replace the current scale
frames.gr <- add_gg(frames.gr, expr(scale_x_continuous(breaks=seq(0,1,0.1),
                                    labels=seq(0,1,0.1), expand = c(0,0))))
frames.gr[[100]]

# the same can be done for discrete data, histogram will then be shown as bin plots

# to make your own graphs, use frames_graph to return data instead of frames
frames.gr <- frames_graph(m, r_list = r_list, r_times = r_times, r_type = "gradient",
                        fade_raster = TRUE, return_data = TRUE)

# then simply animate the frames using animate_frames 
# see all add_ functions on how to customize your frames created with frames_spatial
# or frames_graph

# see ?animate_frames on how to animate your list of frames
}
}
\seealso{
\code{\link{frames_spatial}} \code{\link{join_frames}} \code{\link{animate_frames}}
}
\author{
Jakob Schwalb-Willmann
}
