% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/modelplot.R
\name{modelplot}
\alias{modelplot}
\title{Model Summary Plots with Estimates and Confidence Intervals}
\usage{
modelplot(
  models,
  conf_level = 0.95,
  coef_map = NULL,
  coef_omit = NULL,
  coef_rename = NULL,
  vcov = NULL,
  add_rows = NULL,
  facet = FALSE,
  draw = TRUE,
  background = NULL,
  ...
)
}
\arguments{
\item{models}{a model or (optionally named) list of models}

\item{conf_level}{confidence level to use for confidence intervals}

\item{coef_map}{character vector. Subset, rename, and reorder coefficients.
Coefficients omitted from this vector are omitted from the table. The order
of the vector determines the order of the table.  \code{coef_map} can be a named
or an unnamed character vector (see the Examples section below). If
\code{coef_map} is a named vector, its values define the labels that must appear
in the table, and its names identify the original term names stored in the
model object: \code{c("hp:mpg"="HPxM/G")}.}

\item{coef_omit}{string regular expression. Omits all matching coefficients
from the table using \code{grepl(perl=TRUE)}.}

\item{coef_rename}{named character vector or function which returns a named
vector. Values of the vector refer to the variable names that will appear
in the table. Names refer to the original term names stored in the model
object, e.g. c("hp:mpg"="hp X mpg") for an interaction term.}

\item{vcov}{robust standard errors and other manual statistics. The \code{vcov}
argument accepts six types of input (see the 'Details' and 'Examples'
sections below):
\itemize{
\item NULL returns the default uncertainty estimates of the model object
\item string, vector, or (named) list of strings. The strings "classical", "iid" and "constant" are aliases for \code{NULL}, and they return the model's default uncertainty estimates. The strings "robust", "HC", "HC0", "HC1", "HC2", "HC3", "HC4", "HC4m", "HC5", "stata", "HAC", "NeweyWest", "Andrews", "panel-corrected", "outer-product", "weave" use variance-covariance matrices computed using functions from the \code{sandwich} package. The behavior of those functions can (and sometimes \emph{must}) be altered by passing arguments to \code{sandwich} directly from \code{modelsummary} through the ellipsis (\code{...}), but it is safer to define your own custom functions as described in the next bullet.
\item function or (named) list of functions which return variance-covariance matrices with row and column names equal to the names of your coefficient estimates (e.g., \code{stats::vcov}, \code{sandwich::vcovHC}, \code{function(x) vcovPC(x, cluster="country")}).
\item formula or (named) list of formulas with the cluster variable(s) on the right-hand side (e.g., ~clusterid).
\item (named) list of \code{length(models)} variance-covariance matrices with row and column names equal to the names of your coefficient estimates.
\item a (named) list of length(models) vectors with names equal to the names of your coefficient estimates. See 'Examples' section below. Warning: since this list of vectors can include arbitrary strings or numbers, \code{modelsummary} cannot automatically calculate p values. The \code{stars} argument may thus use incorrect significance thresholds when \code{vcov} is a list of vectors.
}}

\item{add_rows}{a data.frame (or tibble) with the same number of columns as
your main table. By default, rows are appended to the bottom of the table.
You can define a "position" attribute of integers to set the row positions.
See Examples section below.}

\item{facet}{TRUE or FALSE. When the 'models' argument includes several
model objects, TRUE draws terms in separate facets, and FALSE draws terms
side-by-side (dodged).}

\item{draw}{TRUE returns a 'ggplot2' object, FALSE returns the data.frame
used to draw the plot.}

\item{background}{A list of 'ggplot2' geoms to add to the background of the
plot.  This is especially useful to display annotations "behind" the
'geom_pointrange' that 'modelplot' draws.}

\item{...}{all other arguments are passed through to the extractor and
table-making functions. This allows users to pass arguments directly to
\code{modelsummary} in order to affect the behavior of other functions behind
the scenes. Examples include:
\itemize{
\item \code{broom::tidy(exponentiate=TRUE)} to exponentiate logistic regression
\item \code{kableExtra::kbl(escape=FALSE)} to avoid escaping math characters in \code{kableExtra} tables.
\item \code{performance::model_performance(metrics="RMSE")} to select goodness-of-fit statistics to extract using the \code{performance} package (must have set \code{options(modelsummary_get="easystats")} first).
}}
}
\description{
Model Summary Plots with Estimates and Confidence Intervals
}
\examples{
\dontrun{

library(modelsummary)

# single model
mod <- lm(hp ~ vs + drat, mtcars)
modelplot(mod)

# omit terms with string matches or regexes
modelplot(mod, coef_omit = 'Interc')

# rename, reorder and subset with 'coef_map'
cm <- c('vs' = 'V-shape engine',
  'drat' = 'Rear axle ratio')
modelplot(mod, coef_map = cm)

# several models
models <- list()
models[['Small model']] <- lm(hp ~ vs, mtcars)
models[['Medium model']] <- lm(hp ~ vs + factor(cyl), mtcars)
models[['Large model']] <- lm(hp ~ vs + drat + factor(cyl), mtcars)
modelplot(models)

# add_rows: add an empty reference category

mod <- lm(hp ~ factor(cyl), mtcars)

add_rows = data.frame(
  term = "factory(cyl)4",
  model = "Model 1",
  estimate = NA)
attr(add_rows, "position") = 3
modelplot(mod, add_rows = add_rows)


# customize your plots with 'ggplot2' functions
library(ggplot2)

modelplot(models) +
  scale_color_brewer(type = 'qual') +
  theme_classic()

# pass arguments to 'geom_pointrange' through the ... ellipsis
modelplot(mod, color = 'red', size = 1, fatten = .5)

# add geoms to the background, behind geom_pointrange
b <- list(geom_vline(xintercept = 0, color = 'orange'),
  annotate("rect", alpha = .1,
    xmin = -.5, xmax = .5,
    ymin = -Inf, ymax = Inf),
  geom_point(aes(y = term, x = estimate), alpha = .3,
    size = 10, color = 'red', shape = 'square'))
modelplot(mod, background = b)
}

}
