#' @title Print an `sem_outs`-Class Object
#'
#' @description Print the content of
#' an `sem_outs`-class object.
#'
#' @details
#' The print method for the output
#' of `fit_many()`.
#'
#' @return
#' `x` is returned invisibly.
#' Called for its side effect.
#'
#' @param x An `sem_outs`-class object.
#'
#' @param max_models The maximum number
#' of models to be printed. Default
#' is 20.
#'
#' @param ...  Optional arguments.
#' Ignored.
#'
#' @author Shu Fai Cheung <https://orcid.org/0000-0002-9871-9448>
#'
#' @seealso An `sem_outs`-class object
#' is generated by `fit_many()`.
#'
#' @examples
#'
#' library(lavaan)
#' dat <- dat_path_model
#' mod <-
#' "
#' x3 ~ a*x1 + b*x2
#' x4 ~ a*x1
#' ab := a*b
#' "
#' fit <- sem(mod, dat_path_model, fixed.x = TRUE)
#' mod_to_add <- get_add(fit)
#' out <- fit_many(mod_to_add, fit)
#' out
#' print(out, max_models = 1)
#'
#' @export

print.sem_outs <- function(x,
                           max_models = 20,
                           ...) {
    fit_n <- length(x$fit)
    fit_names <- names(x$fit)
    x_call <- x$call
    k_converged <- sum(sapply(x$converged, isTRUE))
    k_post_check <- sum(sapply(x$post_check, isTRUE))
    cat("\n")
    cat("Call:\n")
    print(x_call)
    cat("\n")
    cat("Number of model(s) fitted: ", fit_n, "\n", sep = "")
    cat("Number of model(s) converged: ", k_converged, "\n", sep = "")
    cat("Number of model(s) passed post.check: ", k_post_check, "\n", sep = "")
    cat("\n")
    if (fit_n > max_models) {
        x_tmp <- fit_names[seq_len(max_models)]
        cat("The first", max_models, "modifications(s)/model(s):\n")
      } else {
        x_tmp <- fit_names
        cat("The modification(s)/model(s):\n")
      }
    cat(paste(x_tmp, collapse = "\n"), "\n")
    invisible(x)
  }