% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bayesopt_ego.R
\name{mlr_loop_functions_ego}
\alias{mlr_loop_functions_ego}
\alias{bayesopt_ego}
\title{Sequential Single-Objective Bayesian Optimization}
\usage{
bayesopt_ego(
  instance,
  init_design_size = NULL,
  surrogate,
  acq_function,
  acq_optimizer,
  random_interleave_iter = 0L
)
}
\arguments{
\item{instance}{(\link[bbotk:OptimInstanceSingleCrit]{bbotk::OptimInstanceSingleCrit})\cr
The \link[bbotk:OptimInstanceSingleCrit]{bbotk::OptimInstanceSingleCrit} to be optimized.}

\item{init_design_size}{(\code{NULL} | \code{integer(1)})\cr
Size of the initial design.
If \code{NULL} and the \link[bbotk:Archive]{bbotk::Archive} contains no evaluations, \code{4 * d} is used with \code{d} being the
dimensionality of the search space.
Points are drawn uniformly at random.}

\item{surrogate}{(\link{Surrogate})\cr
\link{Surrogate} to be used as a surrogate.
Typically a \link{SurrogateLearner}.}

\item{acq_function}{(\link{AcqFunction})\cr
\link{AcqFunction} to be used as acquisition function.}

\item{acq_optimizer}{(\link{AcqOptimizer})\cr
\link{AcqOptimizer} to be used as acquisition function optimizer.}

\item{random_interleave_iter}{(\code{integer(1)})\cr
Every \code{random_interleave_iter} iteration (starting after the initial design), a point is
sampled uniformly at random and evaluated (instead of a model based proposal).
For example, if \code{random_interleave_iter = 2}, random interleaving is performed in the second,
fourth, sixth, ... iteration.
Default is \code{0}, i.e., no random interleaving is performed at all.}
}
\value{
invisible(instance)\cr
The original instance is modified in-place and returned invisible.
}
\description{
Loop function for sequential single-objective Bayesian Optimization.
Normally used inside an \link{OptimizerMbo}.

In each iteration after the initial design, the surrogate and acquisition function are updated and the next candidate
is chosen based on optimizing the acquisition function.
}
\note{
\itemize{
\item The \code{acq_function$surrogate}, even if already populated, will always be overwritten by the \code{surrogate}.
\item The \code{acq_optimizer$acq_function}, even if already populated, will always be overwritten by \code{acq_function}.
\item The \code{surrogate$archive}, even if already populated, will always be overwritten by the \link[bbotk:Archive]{bbotk::Archive} of the \link[bbotk:OptimInstanceSingleCrit]{bbotk::OptimInstanceSingleCrit}.
}
}
\examples{
\donttest{
if (requireNamespace("mlr3learners") &
    requireNamespace("DiceKriging") &
    requireNamespace("rgenoud")) {

  library(bbotk)
  library(paradox)
  library(mlr3learners)

  fun = function(xs) {
    list(y = xs$x ^ 2)
  }
  domain = ps(x = p_dbl(lower = -10, upper = 10))
  codomain = ps(y = p_dbl(tags = "minimize"))
  objective = ObjectiveRFun$new(fun = fun, domain = domain, codomain = codomain)

  instance = OptimInstanceSingleCrit$new(
    objective = objective,
    terminator = trm("evals", n_evals = 5))

  surrogate = default_surrogate(instance)

  acq_function = acqf("ei")

  acq_optimizer = acqo(
    optimizer = opt("random_search", batch_size = 100),
    terminator = trm("evals", n_evals = 100))

  optimizer = opt("mbo",
    loop_function = bayesopt_ego,
    surrogate = surrogate,
    acq_function = acq_function,
    acq_optimizer = acq_optimizer)

  optimizer$optimize(instance)

  # expected improvement per second example
  fun = function(xs) {
    list(y = xs$x ^ 2, time = abs(xs$x))
  }
  domain = ps(x = p_dbl(lower = -10, upper = 10))
  codomain = ps(y = p_dbl(tags = "minimize"), time = p_dbl(tags = "time"))
  objective = ObjectiveRFun$new(fun = fun, domain = domain, codomain = codomain)

  instance = OptimInstanceSingleCrit$new(
    objective = objective,
    terminator = trm("evals", n_evals = 5))

  surrogate = default_surrogate(instance, n_learner = 2)
  surrogate$y_cols = c("y", "time")

  optimizer = opt("mbo",
    loop_function = bayesopt_ego,
    surrogate = surrogate,
    acq_function = acqf("eips"),
    acq_optimizer = acq_optimizer)

  optimizer$optimize(instance)
}
}
}
\references{
\itemize{
\item Jones, R. D, Schonlau, Matthias, Welch, J. W (1998).
\dQuote{Efficient Global Optimization of Expensive Black-Box Functions.}
\emph{Journal of Global optimization}, \bold{13}(4), 455--492.
\item Snoek, Jasper, Larochelle, Hugo, Adams, P R (2012).
\dQuote{Practical Bayesian Optimization of Machine Learning Algorithms.}
In Pereira F, Burges CJC, Bottou L, Weinberger KQ (eds.), \emph{Advances in Neural Information Processing Systems}, volume 25, 2951--2959.
}
}
\seealso{
Other Loop Function: 
\code{\link{loop_function}},
\code{\link{mlr_loop_functions_mpcl}},
\code{\link{mlr_loop_functions_parego}},
\code{\link{mlr_loop_functions_smsego}},
\code{\link{mlr_loop_functions}}
}
\concept{Loop Function}
