#' Estimate and Standard Error of a Non-Linear Function of Estimated Model Parameters
#'
#' `deltamethod` is a function to get a first-order approximate standard error
#' for a nonlinear function of a vector of random variables with known or
#' estimated covariance matrix. [`deltamethod`] emulates the behavior of the
#' excellent and well-established [car::deltaMethod] and [car::linearHypothesis]
#' functions, but it supports more models, requires fewer dependencies, and
#' offers some convenience features like shortcuts for robust standard errors.
#'
#' Warning: For hypothesis tests on objects produced by the `marginaleffects`
#' package, it is safer to use the `hypothesis` argument of the original function.
#' Using `deltamethod()` may not work in certain environments, or when called
#' programmatically.
#' 
#' @inheritParams comparisons
#' @param model Model object or object generated by the `comparisons()`, `marginaleffects()`, `predictions()`, or `marginalmeans()` functions.
#' @param FUN `NULL` or function. 
#' * `NULL` (default): hypothesis test on the model's coefficients.
#' * Function which accepts a model object and returns a numeric vector or a data.frame with two columns called `term` and `estimate`. This argument can be useful when users want to conduct a hypothesis test on an arbitrary function of quantities held in a model object.
#' @examples
#' library(marginaleffects)
#' mod <- lm(mpg ~ hp + wt + factor(cyl), data = mtcars)
#' 
#' # When `FUN` and `hypothesis` are `NULL`, `deltamethod()` returns a data.frame of parameters
#' deltamethod(mod)
#' 
#' # Test of equality between coefficients
#' deltamethod(mod, hypothesis = "hp = wt")
#' 
#' # Non-linear function
#' deltamethod(mod, hypothesis = "exp(hp + wt) = 0.1")
#' 
#' # Robust standard errors
#' deltamethod(mod, hypothesis = "hp = wt", vcov = "HC3")
#' 
#' # b1, b2, ... shortcuts can be used to identify the position of the
#' # parameters of interest in the output of FUN
#' deltamethod(mod, hypothesis = "b2 = b3")
#' 
#' # term names with special characters have to be enclosed in backticks
#' deltamethod(mod, hypothesis = "`factor(cyl)6` = `factor(cyl)8`")
#' 
#' mod2 <- lm(mpg ~ hp * drat, data = mtcars)
#' deltamethod(mod2, hypothesis = "`hp:drat` = drat")
#' 
#' # predictions(), comparisons(), and marginaleffects()
#' mod <- glm(am ~ hp + mpg, data = mtcars, family = binomial)
#' cmp <- comparisons(mod, newdata = "mean")
#' deltamethod(cmp, hypothesis = "b1 = b2")
#' 
#' mfx <- marginaleffects(mod, newdata = "mean")
#' deltamethod(cmp, hypothesis = "b2 = 0.2")
#' 
#' pre <- predictions(mod, newdata = datagrid(hp = 110, mpg = c(30, 35)))
#' deltamethod(pre, hypothesis = "b1 = b2")
#' 
#' # The `FUN` argument can be used to compute standard errors for fitted values
#' mod <- glm(am ~ hp + mpg, data = mtcars, family = binomial)
#' 
#' f <- function(x) predict(x, type = "link", newdata = mtcars)
#' p <- deltamethod(mod, FUN = f)
#' head(p)
#' 
#' f <- function(x) predict(x, type = "response", newdata = mtcars)
#' p <- deltamethod(mod, FUN = f)
#' head(p)
#' 
#' @export
deltamethod <- function(
    model,
    hypothesis = NULL,
    FUN = NULL,
    vcov = NULL,
    conf_level = 0.95,
    ...) {


    # add `hypothesis` to the original call and re-evaluate
    if (inherits(model, c("comparisons", "marginaleffects", "predictions", "marginalmeans"))) {
        if (is.null(hypothesis)) {
            msg <- "Please specify the `hypothesis` argument."
            insight::format_error(msg)
        }
        mc <- attr(model, "call")
        if (!is.call(mc)) {
            msg <- "Call could not be retrieved from object of class %s."
            msg <- sprintf(msg, class(model)[1])
            insight::format_error(msg)
        }
        if ("hypothesis" %in% names(mc)) {
            msg <- "The `deltamethod()` function cannot be applied if the `hypothesis` argument was already used in the original call."
            insight::format_error(msg)
        }
        if (!isTRUE(checkmate::check_flag(vcov, null.ok = TRUE))) {
            mc[["vcov"]] <- vcov
        }
        mc[["hypothesis"]] <- hypothesis
        out <- eval(mc)
        return(out)
    }

    # after re-evaluation
    hypothesis <- sanitize_hypothesis(hypothesis)
    vcov <- get_vcov(model = model, vcov = vcov)
    vcov.type <- get_vcov_label(vcov = vcov)

    if (is.null(FUN)) {
        FUNinner <- function(model, ...) {
            param <- insight::get_parameters(model, ...)
            colnames(param)[1:2] <- c("term", "estimate")
            return(param)
        }
        if (is.null(hypothesis)) {
            out <- FUNinner(model, ...)
            class(out) <- c("deltamethod", class(out))
            attr(out, "model") <- model
            attr(out, "model_type") <- class(model)[1]
            attr(out, "vcov") <- vcov
            attr(out, "vcov.type") <- vcov.type
            return(out)
        }
    } else {
        FUNinner <- FUN
    }

    FUNouter <- function(model, hypothesis) {
        out <- FUNinner(model)

        if (isTRUE(checkmate::check_numeric(out))) {
            out <- data.frame(
                term = seq_along(out),
                estimate = out)
        }

        if (!inherits(out, "data.frame") || any(!c("term", "estimate") %in% colnames(out))) {
            msg <- format_msg(
            "`FUN` must return a numeric vector or a data.frame with two columns named `term` and `estimate`.")
            stop(msg, call. = FALSE)
        }

        out <- get_hypothesis(out, hypothesis = hypothesis, column = "estimate")$estimate
        return(out)
    }

    b <- FUNouter(model = model, hypothesis = hypothesis)

    se <- get_se_delta(
        model = model,
        vcov = vcov,
        hypothesis = hypothesis,
        FUN = FUNouter,
        ...)

    if (!is.null(hypothesis)) {
        hyplab <- attr(hypothesis, "label")
        if (!is.null(hyplab)) {
            out <- data.frame(
                term = hyplab,
                estimate = b,
                std.error = se)
        }  else {
            out <- data.frame(
                term = "custom",
                estimate = b,
                std.error = se)
        }
    } else {
        out <- data.frame(
            term = paste0("b", seq_along(b)),
            estimate = b,
            std.error = se)
    }


    out <- get_ci(
        out,
        conf_level = conf_level,
        # sometimes get_predicted fails on SE but succeeds on CI (e.g., betareg)
        vcov = vcov,
        overwrite = FALSE,
        draws = NULL,
        estimate = "estimate",
        ...)

    class(out) <- c("deltamethod", class(out))
    attr(out, "model") <- model
    attr(out, "model_type") <- class(model)[1]
    attr(out, "vcov") <- vcov
    attr(out, "vcov.type") <- vcov.type

    return(out)
}
