% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pseudo_jn.r
\name{pseudo_johnson_neyman}
\alias{pseudo_johnson_neyman}
\alias{print.pseudo_johnson_neyman}
\title{Pseudo Johnson-Neyman Probing}
\usage{
pseudo_johnson_neyman(
  object = NULL,
  w_lower = NULL,
  w_upper = NULL,
  optimize_method = c("uniroot", "optimize"),
  extendInt = c("no", "yes", "downX", "upX"),
  tol = .Machine$double.eps^0.25
)

\method{print}{pseudo_johnson_neyman}(x, digits = 3, ...)
}
\arguments{
\item{object}{A
\code{cond_indirect_effects}-class object,
which is the output of \code{\link[=cond_indirect_effects]{cond_indirect_effects()}}.}

\item{w_lower}{The smallest value of
the moderator when doing the search.
If set to \verb{NULL,} the default, it
will be 10 standard deviations
below mean, which should be small
enough.}

\item{w_upper}{The largest value of
the moderator when doing the search.
If set to \verb{NULL,} the default, it
will be 10 standard deviations
above mean, which should be large
enough.}

\item{optimize_method}{The optimization
method to be used. Either
\code{"uniroot"} (the default) or
\code{"optimize"}, corresponding to
\code{\link[stats:uniroot]{stats::uniroot()}} and
\code{\link[stats:optimize]{stats::optimize()}}, respectively.}

\item{extendInt}{Used by
\code{\link[stats:uniroot]{stats::uniroot()}}. If \code{"no"}, then
search will be conducted strictly
within \code{c(w_lower, w_upper)}. Otherwise,
the range is extended based on this
argument if the solution is not found.
Please refer to \code{\link[stats:uniroot]{stats::uniroot()}}
for details.}

\item{tol}{The tolerance level used
by both \code{\link[stats:uniroot]{stats::uniroot()}} and
\code{\link[stats:optimize]{stats::optimize()}}.}

\item{x}{The output of
\code{\link[=pseudo_johnson_neyman]{pseudo_johnson_neyman()}}.}

\item{digits}{Number of digits to
display. Default is 3.}

\item{...}{Other arguments. Not used.}
}
\value{
A list of the class \code{pseudo_johnson_neyman}
(with a print method, \code{\link[=print.pseudo_johnson_neyman]{print.pseudo_johnson_neyman()}}).
It has these major elements:
\itemize{
\item \code{cond_effects}: An output of
\code{\link[=cond_indirect_effects]{cond_indirect_effects()}} for the
two levels of the moderator found.
\item \code{w_min_valid}: Logical. If \code{TRUE},
the conditional effect is just
significant at the lower level of
the moderator found,
and so is significant below this point.
If \code{FALSE}, then the lower level of
the moderator found is just the
lower bound of the range searched,
that is, \code{w_lower}.
\item \code{w_max_valid}: Logical. If \code{TRUE},
the conditional effect is just
significant at the higher level of
the moderator found,
and so is significant above this point.
If \code{FALSE}, then the higher level of
the moderator found is just the
upper bound of the range searched,
that is, \code{w_upper}.
}
}
\description{
Use the pseudo
Johnson-Neyman approach (Hayes, 2022)
to find the range of values of a
moderator in which the conditional
effect is not significant.
}
\details{
This function uses the
pseudo Johnson-Neyman approach
proposed by Hayes (2022) to find the
values of a moderator at which a
conditional effect is
"nearly just significant" based on
confidence interval. If an effect is
moderated, there will be two such
points (though one can be very large
or small) forming a range.
The conditional effect
is not significant within this range,
and significant outside this range,
based on the confidence interval.

This function receives the output
of \code{\link[=cond_indirect_effects]{cond_indirect_effects()}}
and search for, within
a specific range, the two values of
the moderator at which
the conditional effect is "nearly just significant",
that is, the confidence interval
"nearly touches" zero.

Note that numerical method is used
to find the points. Therefore,
strictly speaking, the effects at
the end points are still either
significant or not significant, even
if the confidence limit is very close
to zero.
\subsection{Supported Methods}{

This function supports models fitted
by \code{\link[=lm]{lm()}}, \code{\link[lavaan:sem]{lavaan::sem()}},
and \code{\link[semTools:runMI]{semTools::sem.mi()}}. This function
also supports both bootstrapping
and Monte Carlo confidence intervals.
It also supports conditional
direct paths (no mediator) and
conditional indirect paths (with one
or more mediator), with \code{x} and/or
\code{y} standardized.
}

\subsection{Requirements}{

To be eligible for using this function,
one form of confidence intervals
(e.g, bootstrapping or Monte Carlo)
must has been requested (e.g.,
setting \code{boot_ci = TRUE} or
\code{mc_ci = TRUE}) when calling
\code{\link[=cond_indirect_effects]{cond_indirect_effects()}}.

The confidence level of the confidence
intervals adopted when calling
\code{\link[=cond_indirect_effects]{cond_indirect_effects()}} will be used
by this function.
}

\subsection{Possible failures}{

Even if a path has only one moderator,
it is possible that no solution, or
more than one solution, is/are found
if the relation between this moderator
and the conditional effect is not linear.

Solution may also be not found if
the conditional effect is significant
over a wide range of value of the
moderator.

It is advised to use \code{\link[=plot_effect_vs_w]{plot_effect_vs_w()}}
to examine the relation between the
effect and the moderator first before
calling this function.
}

\subsection{Speed}{

Note that, for conditional indirect
effects, the search can be slow
because the confidence interval needs
to be recomputed for each new value
of the moderator.
}

\subsection{Limitations}{
\itemize{
\item This function currently only supports
a path with only one moderator,
\item This function does not yet support
multigroup models.
}
}
}
\section{Methods (by generic)}{
\itemize{
\item \code{print(pseudo_johnson_neyman)}: Print
method for output of \code{\link[=pseudo_johnson_neyman]{pseudo_johnson_neyman()}}.

}}
\examples{

library(lavaan)

dat <- data_med_mod_a
dat$wx <- dat$x * dat$w
mod <-
"
m ~ x + w + wx
y ~ m + x
"
fit <- sem(mod, dat)

# In real research, R should be 2000 or even 5000
# In real research, no need to set parallel and progress to FALSE
# Parallel processing is enabled by default and
# progress is displayed by default.
boot_out <- do_boot(fit,
                    R = 50,
                    seed = 4314,
                    parallel = FALSE,
                    progress = FALSE)
out <- cond_indirect_effects(x = "x", y = "y", m = "m",
                             wlevels = "w",
                             fit = fit,
                             boot_ci = TRUE,
                             boot_out = boot_out)

# Visualize the relation first
plot_effect_vs_w(out)

out_jn <- pseudo_johnson_neyman(out)
out_jn

# Plot the range
plot_effect_vs_w(out_jn$cond_effects)

}
\references{
Hayes, A. F. (2022). \emph{Introduction to mediation, moderation, and conditional process analysis: A regression-based approach} (Third edition). The Guilford Press.
}
\seealso{
\code{\link[=cond_indirect_effects]{cond_indirect_effects()}}
}
