% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/delineate_crowns.R
\name{delineate_crowns}
\alias{delineate_crowns}
\title{Compute the hull of each tree.}
\usage{
delineate_crowns(
  las,
  type = c("convex", "concave", "bbox"),
  concavity = 3,
  length_threshold = 0,
  func = NULL,
  attribute = "treeID"
)
}
\arguments{
\item{las}{An object of class \link[lidR:LAS-class]{LAS} or \link[lidR:LAScatalog-class]{LAScatalog}.}

\item{type}{character. Hull type. Can be 'convex', 'concave' or 'bbox'.}

\item{concavity}{numeric. If \code{type = "concave"}, a relative measure of concavity. 1 results
in a relatively detailed shape, Infinity results in a convex hull.}

\item{length_threshold}{numeric. If \code{type = "concave"}, when a segment length is below this
threshold, no further detail is added. Higher values result in simpler shapes.}

\item{func}{formula. An expression to be applied to each tree. It works like in \link{grid_metrics}
\link{voxel_metrics} or \link{tree_metrics} and computes, in addition to the hulls a set of metrics
for each tree.}

\item{attribute}{character. The attribute where the ID of each tree is stored. In lidR, the default is
"treeID".}
}
\value{
A \code{SpatialPolygonsDataFrame}. If a tree has less than 4 points it is not considered.
}
\description{
Compute the hull of each segmented tree. The hull can be convex, concave or a bounding box (see
details and references).
}
\details{
The concave hull method under the hood is described in Park & Oh (2012). The function relies on
the \link[concaveman:concaveman]{concaveman} function.
}
\section{Working with a \code{LAScatalog}}{

This section appears in each function that supports a LAScatalog as input.\cr

In \code{lidR} when the input of a function is a \link[lidR:LAScatalog-class]{LAScatalog} the
function uses the LAScatalog processing engine. The user can modify the engine options using
the \link[lidR:catalog_options_tools]{available options}. A careful reading of the
\link[lidR:LAScatalog-class]{engine documentation} is recommended before processing \code{LAScatalogs}. Each
\code{lidR} function should come with a section that documents the supported engine options.\cr

The \code{LAScatalog} engine supports \code{.lax} files that \emph{significantly} improve the computation
speed of spatial queries using a spatial index. Users should really take advantage a \code{.lax} files,
but this is not mandatory.\cr
}

\section{Supported processing options}{

Supported processing options for a \code{LAScatalog} (in bold). For more details see the
\link[lidR:LAScatalog-class]{LAScatalog engine documentation}:
\itemize{
\item \strong{chunk size}: How much data is loaded at once.
\item \strong{chunk buffer*}: Mandatory to get a continuous output without edge effects. The buffer is
always removed once processed and will never be returned either in R or in files.
\item \strong{chunk alignment}: Align the processed chunks.
\item \strong{progress}: Displays a progression estimation.
\item \strong{output files}: Supported templates are \code{\{XLEFT\}}, \code{\{XRIGHT\}},
\code{\{YBOTTOM\}}, \code{\{YTOP\}}, \code{\{XCENTER\}}, \code{\{YCENTER\}} \code{\{ID\}} and,
if chunk size is equal to 0 (processing by file), \code{\{ORIGINALFILENAME\}}.
\item \strong{select}: Load only attributes of interest.
\item \strong{filter}: Read only points of interest.
}
}

\examples{
LASfile <- system.file("extdata", "MixedConifer.laz", package="lidR")
poi = "-drop_z_below 0 -inside 481280 3812940 481320 3812980"
las = readLAS(LASfile, select = "xyz0", filter = poi)

# NOTE: This dataset is already segmented

#plot(las, color = "treeID", colorPalette = pastel.colors(200))

# Only the hulls
convex_hulls = delineate_crowns(las)
plot(convex_hulls)

# The hulls + some user-defined metrics
convex_hulls = delineate_crowns(las, func = ~list(Zmean = mean(Z)))
convex_hulls

# The bounding box
bbox_hulls = delineate_crowns(las, "bbox")
plot(bbox_hulls)

\dontrun{
# With concave hull (longer to compute)
concave_hulls = delineate_crowns(las, "concave")
plot(concave_hulls)

spplot(convex_hulls, "ZTOP")
spplot(convex_hulls, "Zmean")
}
}
\references{
Park, J. S., & Oh, S. J. (2012). A new concave hull algorithm and concaveness measure
for n-dimensional datasets. Journal of Information science and engineering, 28(3), 587-600.
}
