lacm <- function(formula, data, subset, offset, contrasts = NULL, start.theta = NULL, fixed, d = 1, kernel.type = c("Rectangular", "Epanechnikov", "Triangular", "Quartic (biweight)", "Triweight", "Tricube", "Cosine"), fit = TRUE, gh.num = 20, reltol.opt = 1e-4, opt.method = c("Nelder-Mead", "BFGS"), maxit.opt = 1000, compute.jacobian = TRUE, jacob.method = c("simple", "Richardson", "complex"), print.info = TRUE) {

    ## lines below inherited from 'glm'
    call <- match.call()
    if (missing(data)) 
        data <- environment(formula)
    mf <- match.call(expand.dots = FALSE)
    m <- match(c("formula", "data", "subset", "offset"), names(mf), 0L)
    mf <- mf[c(1L, m)]
    mf$drop.unused.levels <- TRUE
    mf[[1L]] <- quote(stats::model.frame)
    mf <- eval(mf, parent.frame())
    mt <- attr(mf, "terms")
    Y <- model.response(mf, "any")
    X <- if (!is.empty.model(mt)) 
             model.matrix(mt, mf, contrasts)
         else matrix(, NROW(Y), 0L)
    offset <- as.vector(model.offset(mf))
    if (!is.null(offset)) {
        if (length(offset) != NROW(Y)) 
            stop(gettextf("number of offsets is %d should equal %d (number of observations)", length(offset), NROW(Y)), domain = NA)
    }
    ## end lines inherited from 'glm'
    if (is.null(offset))
        offset <- rep.int(0, NROW(Y))
   
    ans <- structure(list(nobs = length(Y), p = NCOL(X), N = length(Y) - d, d = d, npar = NCOL(X) + 2, Y = Y, X = X, offset = offset, call = match.call()), class = "pairfit")
    
    opt.method <- match.arg(opt.method)
    jacob.method <- match.arg(jacob.method)
    kernel.type <- match.arg(kernel.type)
    compute.kweights <- function(x) {
         switch(kernel.type,
           "Rectangular" = 1,
           "Epanechnikov" =  (3 / 4) * (1 - (x / (d + 1)) ^ 2),
           "Triangular" = (1 - x / (d + 1)),
           "Quartic (biweight)" = (15 / 16) * (1 - (x / (d + 1)) ^ 2) ^ 2,
           "Triweight" = (35 / 32) * (1 - (x / (d + 1)) ^ 2) ^ 3,
           "Tricube" = (70 / 81) * (1 - (x / (d + 1)) ^ 3) ^ 3,
           "Cosine" = (pi / 4) * cos(0.5 * pi * x / (d + 1))) * (x < (d + 1))
    }
    kw <- compute.kweights(1:d)   
    ans$kweights <- kw / sum(kw)  

    if (is.null(start.theta)) {
        ## see Davis, Dunsmuir and Wang (1999)
        mod0 <- glm.fit(X, Y, family = poisson())
        mu <- fitted(mod0)
        res <- Y - mu        
        s2 <- sum(res ^ 2 - mu, na.rm = TRUE) / sum(mu ^ 2, na.rm = TRUE)
        s2 <- ifelse(s2 > 0.05, s2, 0.05)
        tau2 <- log(s2 + 1)   
        rho <- sum(res[-1] * res[-ans$nobs], na.rm = TRUE) / (s2 * sum(mu[-1] * mu[-ans$nobs], na.rm = TRUE))
        phi <- log(s2 * rho + 1) / tau2
        phi <- ifelse(abs(phi) < 0.95, abs(phi), 0.95) * sign(phi)
        start.theta <- c(coef(mod0)[1] - 0.5 * tau2, coef(mod0)[-1], phi, tau2)
        names(start.theta)[(ans$p + 1):(ans$p + 2)] <- c("phi", "tau2")
    }
    ans$start.theta <- start.theta
    ## fixed parameters
    if (missing(fixed))
        ans$fixed <- rep(NA, length(ans$start.theta))
    else {
        if (length(fixed) != length(ans$start.theta))
            stop("fixed has a wrong length")
        ## if sigma2 is zero...
        if (is.finite(fixed[ans$npar]) & fixed[ans$npar] < sqrt(.Machine$double.eps)) {
            ## then also phi is zero
            fixed[(ans$npar - 1L):ans$npar] <- rep(0.0, 2L)
        }
        ans$fixed <- fixed
    }
    ifree <- is.na(ans$fixed)
    theta <- ans$fixed
    theta[ifree] <- ans$start.theta[ifree]

    ## Gauss-Hermite nodes and weights
    gh <-  gauss.quad(gh.num, kind="hermite")

    objfun <- function(theta.free) {
        theta[ifree] <- theta.free
        beta <- theta[1 : ans$p]
        eta <- X %*% beta + offset
        phi <- theta[ans$p + 1]    
        tau2 <- theta[ans$p + 2]
        ## if sigma2 == 0 -> independence 
        if(phi == 0.0 & tau2 == 0.0) {
            llik <- .C("pairlik", as.double(eta), as.double(phi), as.double(tau2), as.integer(Y), as.integer(ans$nobs), as.integer(d), as.double(0.0), as.double(sqrt(pi)), as.integer(1), output = as.double(rep(0.0, ans$N * d)))$output
        }
        else { ## sigma2 != 0
            ## check parameter space
            if (abs(phi) >= 0.99) return (NA)
            if (tau2 <= 0.01) return (NA)
            llik <- .C("pairlik", as.double(eta), as.double(phi), as.double(tau2), as.integer(Y), as.integer(ans$nobs), as.integer(d), as.double(gh$nodes), as.double(gh$weights), as.integer(gh.num), output = as.double(rep(0.0, ans$N * d)))$output                 
        }
        if (all(is.finite(llik)))
            llik
        else (-sqrt(.Machine$double.xmax))
    }
    w <- rep(ans$kweights, ans$N)
    if (fit) {
        if (print.info)
            cat("Computing maximum pairwise likelihood estimates...\n")
        pl.fit <- try(optim(par = theta[ifree], fn = function(x) sum(objfun(x) * w), control = list(fnscale = -1, reltol = reltol.opt, maxit = maxit.opt), method = opt.method))
        ans$convergence <- pl.fit$convergence
        if (!inherits(pl.fit, "try-error") & !pl.fit$convergence){
            theta[ifree] <- pl.fit$par
            ans$plik <- pl.fit$value
        }
        else {
            cat("Sorry, no convergence...\n")
            pl.fit
        }
    }
    else
        ans$plik <- sum(objfun(theta[ifree]) * w)
    ans$theta <- theta
    names(ans$theta) <- names(ans$start.theta)
    if (compute.jacobian) {
        if (print.info)
            cat("Computing jacobian...\n")
        ans$jacobian <- jacobian(objfun, ans$theta[ifree], method = jacob.method)
    }
    else
        ans$jacobian <- NULL
    class(ans) <- "lacm"    
    ans
}

## this is exactly the same of print.lm
print.lacm <- function(x, digits = max(3L, getOption("digits") - 3L), ...) {
    cat("\nCall:\n", paste(deparse(x$call), sep = "\n", collapse = "\n"), "\n\n", sep = "")
    if (length(coef(x))) {
        cat("Coefficients:\n")
        print.default(format(coef(x), digits = digits), print.gap = 2L, quote = FALSE)
    }
    else cat("No coefficients\n")
    cat("\n")
    invisible(x)
}

coef.lacm <- function(object, ...) {
    object$theta
}

.bread.lacm <- function(x, demean = TRUE) {
    efun <- x$jacobian
    if (demean)
        efun <- sweep(efun, 2, colMeans(efun, na.rm = TRUE), check.margin = FALSE)
    efun <- efun * sqrt(rep(x$kweights, x$N))
    H <- crossprod(efun) / x$N
    solve(H)
}

.meat.lacm <- function(x, C = 1, demean = TRUE) {
    efun <- x$jacobian
    efun <- efun * rep(x$kweights, x$N)
    if (demean)
        efun <- sweep(efun, 2, colMeans(efun, na.rm = TRUE), check.margin = FALSE)
    id <- rep(1:x$N, each = x$d)
    efun2 <- aggregate(efun, list(id), sum)[ ,-1]
    L <- ceiling(C * (x$N * x$d) ^ (1/3))  
    acf.es <- acf(efun2, lag.max = L, type = "cov", plot = FALSE, demean = FALSE)[[1]]
    vmat <- acf.es[1, , ]
    autocov <- acf.es[-1, , ]
    wb <- (1 - (1:L) / (L + 1))
    wsum <- apply(autocov, c(2,3), function(x) sum(x  * wb))
    vmat + wsum + t(wsum)
}

vcov.lacm <- function(object, C = 1, demean = TRUE, adjust = TRUE, ...) {
    H <- .bread.lacm(object, demean = demean)
    J <- .meat.lacm(object, C = C, demean = demean)
    ntheta <- nrow(H)
    sand <-  H %*% J %*% H
    if (adjust)
        sand / (object$nobs - object$npar * object$d)
    else
        sand  / object$N
}

CLIC <- function(object, C = 1, demean = TRUE) {
    H <- .bread.lacm(object, demean = demean)
    J <- .meat.lacm(object, C = C, demean = demean)
    - 2 * object$plik + 2 * sum(diag(H %*% J))
}

summary.lacm <- function(object, ...) {
    cf <- object$theta
    ifree <- is.na(object$fixed)
    se <- rep(NA, length(cf))
    se[ifree] <- sqrt(diag(vcov(object, ...)))
    cf <- cbind(cf, se, cf/se, 2 * pnorm(-abs(cf/se)))
    colnames(cf) <- c("Estimate", "Std. Error", "z value", "Pr(>|z|)")
    rownames(cf) <- names(object$theta)
    object$coefficients <- cf
    object$clic <- CLIC(object, ...)
    class(object) <- "summary.lacm"
    object
}

print.summary.lacm <- function (x, digits = max(3, getOption("digits") - 3), ...) {
    cat("\nCall:", deparse(x$call, width.cutoff = floor(getOption("width") * 0.85)), "", sep = "\n")
    cat("Pairwise likelihood order:", x$d, "\n")
    if (x$convergence) {
        cat("\nmodel did not converge\n")
    }
    else {
        cat("\nCoefficients:\n")
        printCoefmat(x$coefficients, digits = digits, signif.legend = FALSE)
        cat("\nLog pairwise likelihood = ", formatC(x$plik, digits = max(5L, digits + 1L)), ",  CLIC = ", format(x$clic, digits = max(4L, digits + 1L)), "\n", sep = "")
    }
    invisible(x)
}

select.C <- function(object, demean = TRUE, adjust = TRUE, plot = TRUE, ...) {
    ans <- matrix(0.0, 8L, object$npar)
    denom <- diag(vcov(object, C = 1, demean = demean, adjust = adjust))
    for (i in seq_len(8L))
        ans[i,] <- diag(vcov(object, C = i, demean = demean, adjust = adjust)) / denom
    ans <- rowMeans(ans)
    names(ans) <- 1:8L
    if (plot) {
        plot(ans, type = "b", xlab = "C", ylab = "Mean relative variance", ...)
        grid()
        invisible(ans)
    }
    else
        ans
}

select.order <- function(object, max.d = 10, C = 1, fit = FALSE, demean = TRUE, adjust = TRUE, plot = TRUE, ...){
    ans <- matrix(0.0, max.d, object$npar)
    denom <-  diag(vcov(object, C = 1, demean = demean, adjust = adjust))
    for (i in seq_len(max.d)) {
        fitd <- update(object, d = i, fit = fit, start.theta = coef(object), print.info = FALSE)
        ans[i,] <- diag(vcov(fitd, C = C, demean = demean, adjust = adjust)) / denom
    }
    ans <- rowMeans(ans)
    if (plot) {
        plot(ans, type = "b", xlab = "Pairwise likelihood order", ylab = "Mean relative variance", ...)
        grid()
        invisible(ans)
    }
    else
        ans
}

simulate.lacm <- function(object, nsim = 1, seed = NULL, ...) {

    ## lines below inherited from 'simulate.lm'
    if (!exists(".Random.seed", envir = .GlobalEnv, inherits = FALSE)) 
        runif(1)
    if (is.null(seed)) 
        RNGstate <- get(".Random.seed", envir = .GlobalEnv)
    else {
        R.seed <- get(".Random.seed", envir = .GlobalEnv)
        set.seed(seed)
        RNGstate <- structure(seed, kind = as.list(RNGkind()))
        on.exit(assign(".Random.seed", R.seed, envir = .GlobalEnv))
    }
    X <- object$X
    beta <- object$theta[seq_len(object$p)]
    phi <- object$theta[object$npar - 1L]
    sigma2 <- object$theta[object$npar]

    simone <- function() {
        u <- arima.sim(model = list(ar = phi), n = object$nobs, sd = sqrt(sigma2))
        rpois(n = object$nobs, lambda = exp(X %*% beta + u))
    }
    sims <- replicate(nsim, simone())
    sims <- as.data.frame(sims)
    colnames(sims) <- paste("sim_", seq_len(nsim), sep = "")
    sims
}


