% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/FeatureEffect.R
\name{FeatureEffect}
\alias{FeatureEffect}
\title{Effect of a feature on predictions}
\format{\code{\link{R6Class}} object.}
\description{
\code{FeatureEffect} computes and plots (individual) feature effects of prediction models.
}
\section{Usage}{

\preformatted{
effect = FeatureEffect$new(predictor, feature, method = "ale", 
    grid.size = 20,  center.at = NULL, run = TRUE)

plot(effect)
effect$results
print(effect)
effectd$set.feature("x2")
}
}

\section{Arguments}{


For FeatureEffect$new():
\describe{
\item{predictor: }{(Predictor)\cr 
The object (created with Predictor$new()) holding the machine learning model and the data.}
\item{feature: }{(`character(1)` | `character(2)` | `numeric(1)` | `numeric(2)`) \cr The feature name or index for which to compute the effects.}
\item{method: }{(`character(1)`)\cr 
'ale' for accumulated local effects (the default), 
'pdp' for partial dependence plot, 
'ice' for individual conditional expectation curves,
'pdp+ice' for partial dependence plot and ice curves within the same plot.}
\item{center.at: }{(`numeric(1)`)\cr Value at which the plot should be centered. Ignored in the case of two features.}
\item{grid.size: }{(`numeric(1)` | `numeric(2)`)\cr The size of the grid for evaluating the predictions}
}
}

\section{Details}{


The FeatureEffect class compute the effect a feature has on the prediction. 
Different methods are implemented:
\itemize{
\item{Accumulated Local Effect (ALE) plots}
\item{Partial Dependence Plots (PDPs)}
\item{Individual Conditional Expectation (ICE) curves}
}
 
Accumuluated local effects and partial dependence plots both show the average model prediction over the feature.
The difference is that ALE are computed as accumulated differences over the conditional distribution and partial dependence plots 
over the marginal distribution.
ALE plots preferable to PDPs, because they are faster and unbiased when features are correlated.

ALE plots for categorical features are automatically ordered by the similarity of the categories based on the distribution of the other features for instances in a category.
When the feature is an ordered factor, the ALE plot leaves the order as is.

Individual conditional expectation curves describe how, for a single
observation, the prediction changes when the feature changes and can be combined with partial dependence plots.

To learn more about accumulated local effects, read the Interpretable Machine Learning book: 
\url{https://christophm.github.io/interpretable-ml-book/ale.html}

And for the partial dependence plot:
\url{https://christophm.github.io/interpretable-ml-book/pdp.html}

And for individual conditional expectation: 
\url{https://christophm.github.io/interpretable-ml-book/ice.html}
}

\section{Fields}{

\describe{
\item{method: }{(`character(1)`)\cr
'ale' for accumulated local effects, 
'pdp' for partial dependence plot, 
'ice' for individual conditional expectation curves,
'pdp+ice' for partial dependence plot and ice curves within the same plot.}
\item{feature.name: }{(`character(1)` | `character(2)`)\cr The names of the features for which the partial dependence was computed.}
\item{feature.type: }{(`character(1)` | `character(2)`)\cr The detected types of the features, either "categorical" or "numerical".}
\item{grid.size: }{(`numeric(1)` | `numeric(2)`)\cr The size of the grid.}
\item{center.at: }{(`numeric(1)` | `character(1)`)\cr The value for the centering of the plot. Numeric for numeric features, and the level name for factors.}
\item{n.features: }{(`numeric(1)`)\cr The number of features (either 1 or 2)}
\item{predictor: }{(Predictor)\cr The prediction model that was analysed.}
\item{results: }{(data.frame)\cr data.frame with the grid of feature of interest and the predicted \eqn{\hat{y}}. 
Can be used for creating custom effect plots.}
}
}

\section{Methods}{

\describe{
\item{center()}{method to set the value at which the ice computations are centered. See examples.}
\item{set.feature()}{method to get/set feature(s) (by index) fpr  which to compute pdp. See examples for usage.}
\item{plot()}{method to plot the effects. See \link{plot.FeatureEffect}}
\item{predict()}{method to predict the marginal outcome given a feature. Accepts a data.frame with the feature or a vector. 
                Returns the values of the effect curves at the given values.}
\item{\code{clone()}}{[internal] method to clone the R6 object.}
\item{\code{initialize()}}{[internal] method to initialize the R6 object.}
}
}

\examples{
# We train a random forest on the Boston dataset:
if (require("randomForest")) {
data("Boston", package  = "MASS")
rf = randomForest(medv ~ ., data = Boston, ntree = 50)
mod = Predictor$new(rf, data = Boston)

# Compute the accumulated local effects for the first feature
eff = FeatureEffect$new(mod, feature = "rm",grid.size = 30)
eff$plot()

# Again, but this time with a partial dependence plot and ice curves
eff = FeatureEffect$new(mod, feature = "rm", method = "pdp+ice", grid.size = 30)
plot(eff)

# Since the result is a ggplot object, you can extend it: 
if (require("ggplot2")) {
 plot(eff) + 
 # Adds a title
 ggtitle("Partial dependence") + 
 # Adds original predictions
 geom_point(data = Boston, aes(y = mod$predict(Boston)[[1]], x = rm), 
 color =  "pink", size = 0.5)
}

# If you want to do your own thing, just extract the data: 
eff.dat = eff$results
head(eff.dat)

# You can also use the object to "predict" the marginal values.
eff$predict(Boston[1:3,])
# Instead of the entire data.frame, you can also use feature values:
eff$predict(c(5,6,7))

# You can reuse the pdp object for other features: 
eff$set.feature("lstat")
plot(eff)

# Only plotting the aggregated partial dependence:  
eff = FeatureEffect$new(mod, feature = "crim", method = "pdp")
eff$plot() 

# Only plotting the individual conditional expectation:  
eff = FeatureEffect$new(mod, feature = "crim", method = "ice")
eff$plot() 
  
# Accumulated local effects and partial dependence plots support up to two features: 
eff = FeatureEffect$new(mod, feature = c("crim", "lstat"))  
plot(eff)


# FeatureEffect plots also works with multiclass classification
rf = randomForest(Species ~ ., data = iris, ntree=50)
mod = Predictor$new(rf, data = iris, type = "prob")

# For some models we have to specify additional arguments for the predict function
plot(FeatureEffect$new(mod, feature = "Petal.Width"))

# FeatureEffect plots support up to two features: 
eff = FeatureEffect$new(mod, feature = c("Sepal.Length", "Petal.Length"))
eff$plot()   

# show where the actual data lies
eff$plot(show.data = TRUE)   

# For multiclass classification models, you can choose to only show one class:
mod = Predictor$new(rf, data = iris, type = "prob", class = 1)
plot(FeatureEffect$new(mod, feature = "Sepal.Length"))
}
}
\references{
Apley, D. W. 2016. "Visualizing the Effects of Predictor Variables in Black Box Supervised Learning Models." ArXiv Preprint.

Friedman, J.H. 2001. "Greedy Function Approximation: A Gradient Boosting Machine." Annals of Statistics 29: 1189-1232.

Goldstein, A., Kapelner, A., Bleich, J., and Pitkin, E. (2013). Peeking Inside the Black Box: 
Visualizing Statistical Learning with Plots of Individual Conditional Expectation, 1-22. https://doi.org/10.1080/10618600.2014.907095
}
\seealso{
\link{plot.FeatureEffect}
}
