% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/engr_survey_reverse.R
\encoding{UTF-8}
\name{engr_survey_reverse}
\alias{engr_survey_reverse}
\title{Conversion of Latitude/Longitude Coordinates to Engineering Survey Measurements (KY and TN)}
\source{
\enumerate{
   \item Win-Vector Blog. John Mount, June 11, 2018, "R Tip: use isTRUE()", \url{https://win-vector.com/2018/06/11/r-tip-use-istrue/}.
   \item Latitude Longitude Coordinates to State Code in R - Stack Overflow answered by Josh O'Brien on Jan 6 2012 and edited by Josh O'Brien on Jun 18, 2020. See \url{https://stackoverflow.com/questions/8751497/latitude-longitude-coordinates-to-state-code-in-r}.
   \item r - Convert column classes in data.table - Stack Overflow answered by Matt Dowle on Dec 27 2013. See \url{https://stackoverflow.com/questions/7813578/convert-column-classes-in-data-table}.
   \item Excel vlook up function in R for data frame - Stack Overflow answered by Tyler Rinker on Apr 8 2013 and edited by Tyler Rinker on Feb 26 2014. See \url{https://stackoverflow.com/questions/15882743/excel-vlook-up-function-in-r-for-data-frame}.
   \item r - Converting geo coordinates from degree to decimal - Stack Overflow answered by Robbes on Jan 3 2018 and edited by ayaio on Jan 3 2018. See \url{https://stackoverflow.com/questions/14404596/converting-geo-coordinates-from-degree-to-decimal}.
}
}
\usage{
engr_survey_reverse(
  latitude,
  longitude,
  units = c("survey_ft", "foot", "meters"),
  location = c("KY", "TN"),
  output = c("basic", "table"),
  utm = c(0, 1)
)
}
\arguments{
\item{latitude}{numeric vector [or character vector with spaces, degree
symbol, single quotation mark, and/or escaped quotation mark (\")] that
contains the latitude coordinate point. The following possibilities are
valid: -25.02, "25\\U00B056'50.2068\"N"; "15\\U00B056'58.7068"; "37'1'54.3'N";
"35 8 46.44496",  "35'8'46.44496". If the North designation is not
provided, then it will be added. The latitude/longitude coordinate pair
has to be either a numeric or character vector (no mixing).}

\item{longitude}{numeric vector [or character vector with spaces, degree
symbol, single quotation mark, and/or escaped quotation mark (\")] that
contains the latitude coordinate point. The following possibilities are
valid: 09.83, "25\\U00B056'50.2068\"W"; "15\\U00B056'58.7068"; "37'1'54.3'E";
"35 8 46.44496",  "35'8'46.44496". If the West designation is not
provided, then it will be added. The latitude/longitude coordinate pair
has to be either a numeric or character vector (no mixing).}

\item{units}{character vector that contains the system of units (options are
\code{survey_ft} (United States Customary System) [US survey foot],
\code{foot}, or \code{meters} (International System of Units) [meters])}

\item{location}{character vector that contains the location name ('KY' for
Kentucky or 'TN' for Tennessee)}

\item{output}{character vector that contains basic for the default result
using a simple \code{\link[data.table]{data.table}} or table for the result as a
complex \code{\link[data.table]{data.table}}}

\item{utm}{numeric vector that contains 0 or 1 only. 0 represents do not
provide the utm coordinates and 1 is to provide the utm coordinates}
}
\value{
the geodetic coordinates as projected SPCS Northing and Easting
   coordinates as a \code{\link[data.table]{data.table}}
}
\description{
Takes geodetic coordinates of the World Geodetic System (WGS) (19)84
(EPSG:4326) [MapTiler Reference] and converts those values into Kentucky or
Tennessee-based Northing and Easting engineering survey measurements
[based in the State Plane Coordinate System (SPCS)] in meters, international
foot, or US survey foot. Each latitude [Y] and longitude [X] point is
verified to be located within Kentucky or Tennessee.
}
\note{
Please Note: If you have Kentucky North/South Zone survey measurements, then
please use the Kentucky Geological Survey, University of Kentucky - Kentucky
Single Coordinate Conversion Tool
(http://kgs.uky.edu/kgsweb/CoordConversionTool.asp) instead. That tool will
give you the geographic coordinates too. This R function, \code{engr_survey_reverse}
will only be valid for NAD83 / Kentucky Single Zone.

Useful Tennessee reference Web site
Tennessee Department of Transportation Roadway Design Survey Standards
(https://www.tn.gov/tdot/roadway-design/survey-standards.html)

Useful Kentucky reference Web site
Kentucky Transportation Cabinet Survey Coordination
(https://transportation.ky.gov/Highway-Design/Pages/Survey-Coordination.aspx)
}
\examples{

# Please refer to the iemisc: Engineering Survey Examples vignette for
# additional examples

# Test against TGRN Manual (Reference 5)
# using the 1983 (1995) DATUM
# GPS 60 is the station name with these coordinates
# latitude (North) = 35 8 46.44496
# longitude (West) = 89 54 24.04763

# Northing is 97296.815 # provided in TGRN Manual
# Easting is 244089.427 # provided in TGRN Manual

library(iemisc)

latitude <- "35 8 46.44496"
longitude <- "89 54 24.04763"

Northing_test2 <- 97296.815 # provided in TGRN Manual
Easting_test2 <- 244089.427 # provided in TGRN Manual

tgrn2A <- engr_survey_reverse(latitude, longitude, "meters", "TN", output = "table",
utm = 0)
tgrn2A

tgrn2B <- engr_survey(Northing_test2, Easting_test2, "meters", "TN", output = "table",
utm = 0)
tgrn2B




# Example 1

# Tennessee

library(iemisc)

lat <- 35.8466965

long <- -88.9206794

dt1B <- engr_survey_reverse(lat, long, units = "survey_ft", location = "TN", output =
"basic", utm = 1)
dt1B




# Example 2

# Kentucky

library(iemisc)

lats <- "37'50'21.5988''N"
longs <- "84'16'12.0720'W"

dt2A <- engr_survey_reverse(lats, longs, "foot", "KY", output = "basic", utm = 0)
dt2A





}
\references{
\enumerate{
   \item udunits.dat, v 1.18 2006/09/20 18:59:18 steve Exp, \url{https://web.archive.org/web/20230202155021/https://www.unidata.ucar.edu/software/udunits/udunits-1/udunits.txt}. Retrieved thanks to the Internet Archive: Wayback Machine
   \item Spatial Reference, Aug. 13, 2004, "EPSG:3088: NAD83 / Kentucky Single Zone", \url{https://spatialreference.org/ref/epsg/3088/}.
   \item Spatial Reference, March 7, 2000, "EPSG:32136 NAD83 / Tennessee", \url{https://spatialreference.org/ref/epsg/32136/}.
   \item MapTiler Team, "EPSG:4326: WGS 84 -- WGS84 - World Geodetic System 1984, used in GPS, \url{https://epsg.io/4326}.
   \item Tennessee Department of Transportation Design Division, Tennessee Department of Transportation Tennessee Geodetic Reference Network (TGRN) Reference Manual Second Edition Issued, page ix, \url{https://www.tn.gov/content/dam/tn/tdot/documents/TgrnComposite.pdf}.
   \item Earth Point, "State Plane Coordinate System - Convert, View on Google Earth", \url{https://www.earthpoint.us/StatePlane.aspx}.
   \item National Geodetic Survey datasheet95, version 8.12.5.3, online retrieval date July 25, 2019, Mid Valley Oil Rad Relay Twr designation, HA1363 PID, Grayson County Kentucky, Clarkson (1967) USGS Quad, \url{https://www.ngs.noaa.gov/cgi-bin/ds_mark.prl?PidBox=HA1363}.
   \item National Geodetic Survey datasheet95, version 8.12.5.3, online retrieval date July 25, 2019, 2006 42 07 designation, DL4005 PID, Fayette County Kentucky, Lexington West (1993) USGS Quad, \url{https://www.ngs.noaa.gov/cgi-bin/ds_mark.prl?PidBox=DL4005}.
}
}
\author{
Irucka Embry, Berry Boessenkool (a couple of functions are sourced from OSMscale)
}
