#' Fit a model to an incidence object
#'
#' @author Tim Taylor
#'
#' @param x An [incidence2::incidence] object.
#' @param model Either an individual [`trending_model`] object generated by
#'   `lm_model`, `glm_model`, `glm_nb_model`, `brms_model` or a list of these
#'   models (see `?trending::trending_model()` for further details).
#' @param ... Not currently used.
#'
#' @return An object of class `incidence2_fit` or `incidence2_fit_list` if a
#'   list of models is given as input.
#'
#' @author Tim Taylor
#'
#' @importFrom dplyr grouped_df nest_by full_join bind_rows
#'
fit_model <- function(x, model, ...) {
    UseMethod("fit_model", model)
}


#' @rdname fit_model
#' @aliases fit_model.default
fit_model.default <- function(x, model, ...) {
    stop("`model` input must be a trending model object or a list of these.")
}


#' @rdname fit_model
#' @aliases fit_model.trending_model
fit_model.trending_model <- function(x, model, ...) {

    if (!inherits(x, "incidence2")) {
        stop(sprintf("`%s` is not an incidence object", deparse(substitute(x))))
    }

    group_vars <- incidence2::get_group_names(x)
    vars <- all.vars(trending::get_formula(model))
    idx <- group_vars %in% vars
    if (any(idx)) {
        message("Note that there are grouping variables within model formula")
        group_vars <- group_vars[!idx]
    }
    cols <- c(group_vars, vars)
    x <- x[cols]

    if (is.null(group_vars)) {
        res <- trending::fit(model, x)
        res <- tibble::tibble(.data = list(x), .fitted_model = list(res))
    } else {
        dat <- grouped_df(x, group_vars)
        dat <- nest_by(dat, .key = ".data")
        res <- lapply(dat$.data, function(x) safely(trending::fit)(model, x))
        res <- base_transpose(res)
        dat$.fitted_model <- res[[1]]
        dat$.fitting_warning <- res[[2]]
        dat$.fitting_error <- res[[3]]
        res <- dat
    }
    res
}


#' @rdname fit_model
#' @aliases fit_model.trending_model_list
fit_model.list <- function(x, model, ...) {

    if (!inherits(x, "incidence2")) {
        stop(sprintf("`%s` is not an incidence object", deparse(substitute(x))))
    }

    group_vars <- incidence2::get_group_names(x)

    if(is.null(group_vars)) {
        res <- lapply(model, function(m) safely(fit_model)(x, m))
        res <- base_transpose(res)
        out <- bind_rows(res[[1]])
        out$.fitting_warning <- res[[2]]
        out$.fitting_error <- res[[3]]
    } else {
        res <- lapply(model, function(m) fit_model(x, m))
        out <- bind_rows(res)
    }
    out
}
