/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5module.h" 

#include "H5private.h" 

#define H5_CRC_QUOTIENT 0x04C11DB7

static uint32_t H5_crc_table[256];

static bool H5_crc_table_computed = false;

uint32_t
H5_checksum_fletcher32(const void *_data, size_t _len)
{
    const uint8_t *data = (const uint8_t *)_data; 
    size_t         len  = _len / 2;               
    uint32_t       sum1 = 0, sum2 = 0;

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(_data);
    assert(_len > 0);

    
    
    while (len) {
        size_t tlen = len > 360 ? 360 : len;
        len -= tlen;
        do {
            sum1 += (uint32_t)(((uint16_t)data[0]) << 8) | ((uint16_t)data[1]);
            data += 2;
            sum2 += sum1;
        } while (--tlen);
        sum1 = (sum1 & 0xffff) + (sum1 >> 16);
        sum2 = (sum2 & 0xffff) + (sum2 >> 16);
    }

    
    if (_len % 2) {
        sum1 += (uint32_t)(((uint16_t)*data) << 8);
        sum2 += sum1;
        sum1 = (sum1 & 0xffff) + (sum1 >> 16);
        sum2 = (sum2 & 0xffff) + (sum2 >> 16);
    } 

    
    sum1 = (sum1 & 0xffff) + (sum1 >> 16);
    sum2 = (sum2 & 0xffff) + (sum2 >> 16);

    FUNC_LEAVE_NOAPI((sum2 << 16) | sum1)
} 

static void
H5__checksum_crc_make_table(void)
{
    uint32_t c;    
    unsigned n, k; 

    FUNC_ENTER_PACKAGE_NOERR

    
    for (n = 0; n < 256; n++) {
        c = (uint32_t)n;
        for (k = 0; k < 8; k++)
            if (c & 1)
                c = H5_CRC_QUOTIENT ^ (c >> 1);
            else
                c = c >> 1;
        H5_crc_table[n] = c;
    }
    H5_crc_table_computed = true;

    FUNC_LEAVE_NOAPI_VOID
} 

static uint32_t
H5__checksum_crc_update(uint32_t crc, const uint8_t *buf, size_t len)
{
    size_t n; 

    FUNC_ENTER_PACKAGE_NOERR

    
    if (!H5_crc_table_computed)
        H5__checksum_crc_make_table();

    
    for (n = 0; n < len; n++)
        crc = H5_crc_table[(crc ^ buf[n]) & 0xff] ^ (crc >> 8);

    FUNC_LEAVE_NOAPI(crc)
} 

uint32_t
H5_checksum_crc(const void *_data, size_t len)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(_data);
    assert(len > 0);

    FUNC_LEAVE_NOAPI(H5__checksum_crc_update((uint32_t)0xffffffffL, (const uint8_t *)_data, len) ^
                     0xffffffffL)
} 

#define H5_lookup3_rot(x, k) (((x) << (k)) ^ ((x) >> (32 - (k))))
#define H5_lookup3_mix(a, b, c)                                                                              \
    do {                                                                                                     \
        a -= c;                                                                                              \
        a ^= H5_lookup3_rot(c, 4);                                                                           \
        c += b;                                                                                              \
        b -= a;                                                                                              \
        b ^= H5_lookup3_rot(a, 6);                                                                           \
        a += c;                                                                                              \
        c -= b;                                                                                              \
        c ^= H5_lookup3_rot(b, 8);                                                                           \
        b += a;                                                                                              \
        a -= c;                                                                                              \
        a ^= H5_lookup3_rot(c, 16);                                                                          \
        c += b;                                                                                              \
        b -= a;                                                                                              \
        b ^= H5_lookup3_rot(a, 19);                                                                          \
        a += c;                                                                                              \
        c -= b;                                                                                              \
        c ^= H5_lookup3_rot(b, 4);                                                                           \
        b += a;                                                                                              \
    } while (0)

#define H5_lookup3_final(a, b, c)                                                                            \
    do {                                                                                                     \
        c ^= b;                                                                                              \
        c -= H5_lookup3_rot(b, 14);                                                                          \
        a ^= c;                                                                                              \
        a -= H5_lookup3_rot(c, 11);                                                                          \
        b ^= a;                                                                                              \
        b -= H5_lookup3_rot(a, 25);                                                                          \
        c ^= b;                                                                                              \
        c -= H5_lookup3_rot(b, 16);                                                                          \
        a ^= c;                                                                                              \
        a -= H5_lookup3_rot(c, 4);                                                                           \
        b ^= a;                                                                                              \
        b -= H5_lookup3_rot(a, 14);                                                                          \
        c ^= b;                                                                                              \
        c -= H5_lookup3_rot(b, 24);                                                                          \
    } while (0)

uint32_t
H5_checksum_lookup3(const void *key, size_t length, uint32_t initval)
{
    const uint8_t *k = (const uint8_t *)key;
    uint32_t       a, b, c = 0; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(key);
    assert(length > 0);

    
    a = b = c = 0xdeadbeef + ((uint32_t)length) + initval;

    
    while (length > 12) {
        a += k[0];
        a += ((uint32_t)k[1]) << 8;
        a += ((uint32_t)k[2]) << 16;
        a += ((uint32_t)k[3]) << 24;
        b += k[4];
        b += ((uint32_t)k[5]) << 8;
        b += ((uint32_t)k[6]) << 16;
        b += ((uint32_t)k[7]) << 24;
        c += k[8];
        c += ((uint32_t)k[9]) << 8;
        c += ((uint32_t)k[10]) << 16;
        c += ((uint32_t)k[11]) << 24;
        H5_lookup3_mix(a, b, c);
        length -= 12;
        k += 12;
    }

    
    switch (length) 
    {
        case 12:
            c += ((uint32_t)k[11]) << 24;
            
            H5_ATTR_FALLTHROUGH
        case 11:
            c += ((uint32_t)k[10]) << 16;
            
            H5_ATTR_FALLTHROUGH
        case 10:
            c += ((uint32_t)k[9]) << 8;
            
            H5_ATTR_FALLTHROUGH
        case 9:
            c += k[8];
            
            H5_ATTR_FALLTHROUGH
        case 8:
            b += ((uint32_t)k[7]) << 24;
            
            H5_ATTR_FALLTHROUGH
        case 7:
            b += ((uint32_t)k[6]) << 16;
            
            H5_ATTR_FALLTHROUGH
        case 6:
            b += ((uint32_t)k[5]) << 8;
            
            H5_ATTR_FALLTHROUGH
        case 5:
            b += k[4];
            
            H5_ATTR_FALLTHROUGH
        case 4:
            a += ((uint32_t)k[3]) << 24;
            
            H5_ATTR_FALLTHROUGH
        case 3:
            a += ((uint32_t)k[2]) << 16;
            
            H5_ATTR_FALLTHROUGH
        case 2:
            a += ((uint32_t)k[1]) << 8;
            
            H5_ATTR_FALLTHROUGH
        case 1:
            a += k[0];
            break;
        case 0:
            goto done;
        default:
            assert(0 && "This Should never be executed!");
    }

    H5_lookup3_final(a, b, c);

done:
    FUNC_LEAVE_NOAPI(c)
} 

uint32_t
H5_checksum_metadata(const void *data, size_t len, uint32_t initval)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(data);
    assert(len > 0);

    
    
    FUNC_LEAVE_NOAPI(H5_checksum_lookup3(data, len, initval))
} 

uint32_t
H5_hash_string(const char *str)
{
    uint32_t hash = 5381;
    int      c;

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(str);

    while ((c = *str++))
        hash = ((hash << 5) + hash) + (uint32_t)c; 

    FUNC_LEAVE_NOAPI(hash)
} 
