/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Gmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Gpkg.h"      
#include "H5HLprivate.h" 
#include "H5Iprivate.h"  
#include "H5Lprivate.h"  
#include "H5MMprivate.h" 

#include "H5VLnative_private.h" 

static int H5G__link_cmp_name_inc(const void *lnk1, const void *lnk2);
static int H5G__link_cmp_name_dec(const void *lnk1, const void *lnk2);
static int H5G__link_cmp_corder_inc(const void *lnk1, const void *lnk2);
static int H5G__link_cmp_corder_dec(const void *lnk1, const void *lnk2);

static int
H5G__link_cmp_name_inc(const void *lnk1, const void *lnk2)
{
    FUNC_ENTER_PACKAGE_NOERR

    FUNC_LEAVE_NOAPI(strcmp(((const H5O_link_t *)lnk1)->name, ((const H5O_link_t *)lnk2)->name))
} 

static int
H5G__link_cmp_name_dec(const void *lnk1, const void *lnk2)
{
    FUNC_ENTER_PACKAGE_NOERR

    FUNC_LEAVE_NOAPI(strcmp(((const H5O_link_t *)lnk2)->name, ((const H5O_link_t *)lnk1)->name))
} 

static int
H5G__link_cmp_corder_inc(const void *lnk1, const void *lnk2)
{
    int ret_value = -1; 

    FUNC_ENTER_PACKAGE_NOERR

    if (((const H5O_link_t *)lnk1)->corder < ((const H5O_link_t *)lnk2)->corder)
        ret_value = -1;
    else if (((const H5O_link_t *)lnk1)->corder > ((const H5O_link_t *)lnk2)->corder)
        ret_value = 1;
    else
        ret_value = 0;

    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5G__link_cmp_corder_dec(const void *lnk1, const void *lnk2)
{
    int ret_value = -1; 

    FUNC_ENTER_PACKAGE_NOERR

    if (((const H5O_link_t *)lnk1)->corder < ((const H5O_link_t *)lnk2)->corder)
        ret_value = 1;
    else if (((const H5O_link_t *)lnk1)->corder > ((const H5O_link_t *)lnk2)->corder)
        ret_value = -1;
    else
        ret_value = 0;

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G_link_to_info(const H5O_loc_t *link_loc, const H5O_link_t *lnk, H5L_info2_t *info)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(link_loc);
    assert(lnk);

    
    if (info) {
        info->cset         = lnk->cset;
        info->corder       = lnk->corder;
        info->corder_valid = lnk->corder_valid;
        info->type         = lnk->type;

        switch (lnk->type) {
            case H5L_TYPE_HARD:
                
                if (H5VL_native_addr_to_token(link_loc->file, H5I_FILE, lnk->u.hard.addr, &info->u.token) < 0)
                    HGOTO_ERROR(H5E_LINK, H5E_CANTSERIALIZE, FAIL,
                                "can't serialize address into object token");
                break;

            case H5L_TYPE_SOFT:
                info->u.val_size = strlen(lnk->u.soft.name) + 1; 
                break;

            case H5L_TYPE_ERROR:
            case H5L_TYPE_EXTERNAL:
            case H5L_TYPE_MAX:
            default: {
                const H5L_class_t *link_class; 

                if (lnk->type < H5L_TYPE_UD_MIN || lnk->type > H5L_TYPE_MAX)
                    HGOTO_ERROR(H5E_LINK, H5E_BADTYPE, FAIL, "unknown link class");

                
                
                link_class = H5L_find_class(lnk->type);

                if (link_class != NULL && link_class->query_func != NULL) {
                    ssize_t cb_ret; 

                    
                    H5_BEFORE_USER_CB(FAIL)
                        {
                            
                            
                            cb_ret = (link_class->query_func)(lnk->name, lnk->u.ud.udata, lnk->u.ud.size,
                                                              NULL, (size_t)0);
                        }
                    H5_AFTER_USER_CB(FAIL)
                    if (cb_ret < 0)
                        HGOTO_ERROR(H5E_LINK, H5E_CALLBACK, FAIL,
                                    "query buffer size callback returned failure");

                    info->u.val_size = (size_t)cb_ret;
                } 
                else
                    info->u.val_size = 0;
            } 
        }     
    }         

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__link_to_ent(H5F_t *f, H5HL_t *heap, const H5O_link_t *lnk, H5O_type_t obj_type, const void *crt_info,
                 H5G_entry_t *ent)
{
    size_t name_offset;         
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(heap);
    assert(lnk && lnk->name);

    
    H5G__ent_reset(ent);

    
    if (H5HL_insert(f, heap, strlen(lnk->name) + 1, lnk->name, &name_offset) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINSERT, FAIL, "unable to insert symbol name into heap");
    ent->name_off = name_offset;

    
    switch (lnk->type) {
        case H5L_TYPE_HARD:
            if (obj_type == H5O_TYPE_GROUP) {
                const H5G_obj_create_t *gcrt_info = (const H5G_obj_create_t *)crt_info;

                ent->type = gcrt_info->cache_type;
                if (ent->type != H5G_NOTHING_CACHED)
                    ent->cache = gcrt_info->cache;
#ifndef NDEBUG
                else {
                    
                    H5O_loc_t targ_oloc;   
                    htri_t    stab_exists; 

                    
                    if (H5O_loc_reset(&targ_oloc) < 0)
                        HGOTO_ERROR(H5E_SYM, H5E_CANTRESET, FAIL, "unable to initialize target location");
                    targ_oloc.file = f;
                    targ_oloc.addr = lnk->u.hard.addr;

                    
                    if ((stab_exists = H5O_msg_exists(&targ_oloc, H5O_STAB_ID)) < 0)
                        HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "unable to check for STAB message");

                    assert(!stab_exists);
                } 
#endif            
            }     
            else if (obj_type == H5O_TYPE_UNKNOWN) {
                
                H5O_loc_t  targ_oloc;   
                H5O_t     *oh;          
                H5O_stab_t stab;        
                htri_t     stab_exists; 

                
                if (H5O_loc_reset(&targ_oloc) < 0)
                    HGOTO_ERROR(H5E_SYM, H5E_CANTRESET, FAIL, "unable to initialize target location");
                targ_oloc.file = f;
                targ_oloc.addr = lnk->u.hard.addr;

                
                if (NULL == (oh = H5O_protect(&targ_oloc, H5AC__READ_ONLY_FLAG, false)))
                    HGOTO_ERROR(H5E_SYM, H5E_CANTPROTECT, FAIL, "unable to protect target object header");

                
                if ((stab_exists = H5O_msg_exists_oh(oh, H5O_STAB_ID)) < 0) {
                    if (H5O_unprotect(&targ_oloc, oh, H5AC__NO_FLAGS_SET) < 0)
                        HERROR(H5E_SYM, H5E_CANTUNPROTECT, "unable to release object header");
                    HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "unable to check for STAB message");
                } 

                if (stab_exists) {
                    
                    if (NULL == H5O_msg_read_oh(f, oh, H5O_STAB_ID, &stab)) {
                        if (H5O_unprotect(&targ_oloc, oh, H5AC__NO_FLAGS_SET) < 0)
                            HERROR(H5E_SYM, H5E_CANTUNPROTECT, "unable to release object header");
                        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "unable to read STAB message");
                    } 

                    
                    ent->type                  = H5G_CACHED_STAB;
                    ent->cache.stab.btree_addr = stab.btree_addr;
                    ent->cache.stab.heap_addr  = stab.heap_addr;
                } 
                else
                    
                    ent->type = H5G_NOTHING_CACHED;

                if (H5O_unprotect(&targ_oloc, oh, H5AC__NO_FLAGS_SET) < 0)
                    HGOTO_ERROR(H5E_SYM, H5E_CANTUNPROTECT, FAIL, "unable to release object header");
            } 
            else
                ent->type = H5G_NOTHING_CACHED;

            ent->header = lnk->u.hard.addr;
            break;

        case H5L_TYPE_SOFT: {
            size_t lnk_offset; 

            
            if (H5HL_insert(f, heap, strlen(lnk->u.soft.name) + 1, lnk->u.soft.name, &lnk_offset) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "unable to write link value to local heap");

            ent->type                    = H5G_CACHED_SLINK;
            ent->cache.slink.lval_offset = lnk_offset;
        } break;

        case H5L_TYPE_ERROR:
        case H5L_TYPE_EXTERNAL:
        case H5L_TYPE_MAX:
        default:
            HGOTO_ERROR(H5E_SYM, H5E_BADVALUE, FAIL, "unrecognized link type");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__link_to_loc(const H5G_loc_t *grp_loc, const H5O_link_t *lnk, H5G_loc_t *obj_loc)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(grp_loc);
    assert(lnk);
    assert(obj_loc);

    

    
    if (lnk->type > H5L_TYPE_BUILTIN_MAX && lnk->type < H5L_TYPE_UD_MIN)
        HGOTO_ERROR(H5E_SYM, H5E_UNSUPPORTED, FAIL, "unknown link type");

    
    if (H5G_name_set(grp_loc->path, obj_loc->path, lnk->name) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "cannot set name");

    
    obj_loc->oloc->file         = grp_loc->oloc->file;
    obj_loc->oloc->holding_file = false;
    if (lnk->type == H5L_TYPE_HARD)
        obj_loc->oloc->addr = lnk->u.hard.addr;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__link_sort_table(H5G_link_table_t *ltable, H5_index_t idx_type, H5_iter_order_t order)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(ltable);

    
    if (0 == ltable->nlinks)
        HGOTO_DONE(ret_value);

    
    assert(ltable->lnks);

    
    if (idx_type == H5_INDEX_NAME) {
        if (order == H5_ITER_INC)
            qsort(ltable->lnks, ltable->nlinks, sizeof(H5O_link_t), H5G__link_cmp_name_inc);
        else if (order == H5_ITER_DEC)
            qsort(ltable->lnks, ltable->nlinks, sizeof(H5O_link_t), H5G__link_cmp_name_dec);
        else
            assert(order == H5_ITER_NATIVE);
    } 
    else {
        assert(idx_type == H5_INDEX_CRT_ORDER);
        if (order == H5_ITER_INC)
            qsort(ltable->lnks, ltable->nlinks, sizeof(H5O_link_t), H5G__link_cmp_corder_inc);
        else if (order == H5_ITER_DEC)
            qsort(ltable->lnks, ltable->nlinks, sizeof(H5O_link_t), H5G__link_cmp_corder_dec);
        else
            assert(order == H5_ITER_NATIVE);
    } 

done:
    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5G__link_iterate_table(const H5G_link_table_t *ltable, hsize_t skip, hsize_t *last_lnk,
                        const H5G_lib_iterate_t op, void *op_data)
{
    size_t u;                        
    herr_t ret_value = H5_ITER_CONT; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(ltable);
    assert(op);

    
    if (last_lnk)
        *last_lnk += skip;

    
    H5_CHECKED_ASSIGN(u, size_t, skip, hsize_t);
    for (; u < ltable->nlinks && !ret_value; u++) {
        
        ret_value = (op)(&(ltable->lnks[u]), op_data);

        
        if (last_lnk)
            (*last_lnk)++;
    } 

    
    if (ret_value < 0)
        HERROR(H5E_SYM, H5E_CANTNEXT, "iteration operator failed");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__link_release_table(H5G_link_table_t *ltable)
{
    size_t u;                   
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(ltable);

    
    if (ltable->nlinks > 0) {
        
        for (u = 0; u < ltable->nlinks; u++)
            if (H5O_msg_reset(H5O_LINK_ID, &(ltable->lnks[u])) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_CANTFREE, FAIL, "unable to release link message");

        
        H5MM_xfree(ltable->lnks);
    } 
    else
        assert(ltable->lnks == NULL);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__link_name_replace(H5F_t *file, H5RS_str_t *grp_full_path_r, const H5O_link_t *lnk)
{
    H5RS_str_t *obj_path_r = NULL;    
    herr_t      ret_value  = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(file);

    
    if (grp_full_path_r) {
        obj_path_r = H5G_build_fullpath_refstr_str(grp_full_path_r, lnk->name);
        if (H5G_name_replace(lnk, H5G_NAME_DELETE, file, obj_path_r, NULL, NULL) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTDELETE, FAIL, "unable to replace name");
    }

done:
    if (obj_path_r)
        H5RS_decr(obj_path_r);

    FUNC_LEAVE_NOAPI(ret_value)
} 
