% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kernels.R
\name{Model}
\alias{Model}
\title{C++ abstract class for Hawkes processes}
\description{
This is a C++ abstract class for Hawkes processes, which holds
methods for the estimation of its parameters.
}
\details{
This serves as a basis for the Hawkes model and its count sequence,
with conditional intensity function
\deqn{\lambda(t) = \eta + \mu \sum_{T_i < t} h^\ast(t - T_i).}
As an abstract class, an object of class \code{Model} should never be directly
instanciated, but rather one of its derived class.
The constructor can take no argument, in which case the vector \code{param} is
initialised to sensible values and \code{binsize} defaults to 1.
Alternatively, \code{param} and/or \code{binsize} can be specified.
}
\section{Fields}{

\describe{
\item{\code{param}}{Vector of parameters of the Hawkes process, of the form \eqn{(\eta, \mu, ...)}.}

\item{\code{binsize}}{Bin size for the count sequences.}

\item{\code{new(DerivedClass,(param),(binsize))}}{Constructor for derived classes; \code{param} and/or \code{binsize}
can be safely ignored.}

\item{\code{mean()}}{Returns the expected value on \eqn{[0,\mathrm{end}]}.}

\item{\code{dmean()}}{Returns the Jacobian matrix of the expected value on \eqn{[0,\mathrm{end}]}.}

\item{\code{ddmean()}}{Returns the Hessian matrix of the expected value on \eqn{[0,\mathrm{end}]}.}

\item{\code{f(xi)}}{Returns the spectral density function of the time-continuous count sequence. \itemize{
\item \code{xi} A numeric vector of frequencies.
}}

\item{\code{f1(xi,trunc)}}{Returns the spectral density function of the discrete time count sequence. \itemize{
\item \code{xi} A numeric vector of frequencies.
\item \code{trunc} The number of foldings to take into account for the aliasing.
}}

\item{\code{whittle(I,trunc)}}{Returns the log-spectral likelihood of a discrete time count sequence. \itemize{
\item \code{I} The periodogram of the count sequence.
\item \code{trunc} The number of foldings to take into account for the aliasing.
}}

\item{\code{loglik(events,end)}}{Returns the log-likelihood of a sequence of arrival times. \itemize{
\item \code{events} The sequence of arrival times.
\item \code{end} The endpoint of the observation window \eqn{[0,\mathrm{end}]}.
}}

\item{\code{dloglik(events,end)}}{Returns the Jacobian matrix of the log-likelihood of a sequence of arrival times. \itemize{
\item \code{events} The sequence of arrival times.
\item \code{end} The endpoint of the observation window \eqn{[0,\mathrm{end}]}.
}}

\item{\code{ddloglik(events,end)}}{Returns the Hessian matrix of the log-likelihood of a sequence of arrival times. \itemize{
\item \code{events} The sequence of arrival times.
\item \code{end} The endpoint of the observation window \eqn{[0,\mathrm{end}]}.
}}
}}

\examples{
# Simulate 1000 exponential Hawkes processes on \eqn{[0, 100]},
# and average the periodogram of the count sequences with bin size 1
# at each frequency.
I = rep(0, 100)
for (k in 1:1e3) {
    x = hawkes(100, fun = 1, repr = .5, family = "exp", rate = 2)
    y = discrete(x, binsize = 1)
    I = I + Mod(fft(y - mean(y)))^2 / length(y)
}

# Check that the averaged periodogram correctly approximates the spectral
# density function of the count sequence
model = new(Exponential)
model$param = c(1, .5, 2)
model$binsize = 1

z = 2 * pi * 0:99 / 100     # Frequencies of the periodogram
plot(z, I / 1e3, type = "l") # Averaged periodogram
lines(z, model$f1(xi = z, trunc = 10L), col = "red")

}
\seealso{
\link{Exponential}
}
