% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cheby2.R
\name{cheby2}
\alias{cheby2}
\alias{cheby2.FilterSpecs}
\alias{cheby2.default}
\title{Chebyshev Type II filter design}
\usage{
cheby2(n, ...)

\method{cheby2}{FilterSpecs}(n, ...)

\method{cheby2}{default}(
  n,
  Rs,
  w,
  type = c("low", "high", "stop", "pass"),
  plane = c("z", "s"),
  output = c("Arma", "Zpg", "Sos"),
  ...
)
}
\arguments{
\item{n}{filter order.}

\item{...}{additional arguments passed to cheby1, overriding those given by
\code{n} of class \code{FilterSpecs}.}

\item{Rs}{dB of stopband ripple.}

\item{w}{critical frequencies of the filter. \code{w} must be a scalar for
low-pass and high-pass filters, and \code{w} must be a two-element vector
c(low, high) specifying the lower and upper bands in radians/second. For
digital filters, W must be between 0 and 1 where 1 is the Nyquist
frequency.}

\item{type}{filter type, one of \code{"low"}, \code{"high"}, \code{"stop"},
or \code{"pass"}.}

\item{plane}{"z" for a digital filter or "s" for an analog filter.}

\item{output}{Type of output, one of:
\describe{
  \item{"Arma"}{Autoregressive-Moving average (aka numerator/denominator, aka
  b/a)}
  \item{"Zpg"}{Zero-pole-gain format}
  \item{"Sos"}{Second-order sections}
}
Default is \code{"Arma"} compatibility with the 'signal' package and the
'Matlab' and 'Octave' equivalents, but \code{"Sos"} should be preferred for
general-purpose filtering because of numeric stability.}
}
\value{
Depending on the value of the \code{output} parameter, a list of
  class \code{\link{Arma}}, \code{\link{Zpg}}, or \code{\link{Sos}}
  containing the filter coefficients
}
\description{
Compute the transfer function coefficients of a Chebyshev Type II filter.
}
\details{
Chebyshev filters are analog or digital filters having a steeper roll-off
than Butterworth filters, and have passband ripple (type I) or stopband
ripple (type II).

Because \code{cheby2} is generic, it can be extended to accept other inputs,
using \code{cheb2ord} to generate filter criteria for example.
}
\examples{
## compare the frequency responses of 5th-order
## Butterworth and Chebyshev filters.
bf <- butter(5, 0.1)
cf <- cheby2(5, 20, 0.1)
bfr <- freqz(bf)
cfr <- freqz(cf)
plot(bfr$w / pi, 20 * log10(abs(bfr$h)), type = "l", ylim = c(-40, 0),
  xlim = c(0, .5), xlab = "Frequency", ylab = c("dB"))
lines(cfr$w / pi, 20 * log10(abs(cfr$h)), col = "red")

# compare type I and type II Chebyshev filters.
c1fr <- freqz(cheby1(5, .5, 0.5))
c2fr <- freqz(cheby2(5, 20, 0.5))
plot(c1fr$w / pi, abs(c1fr$h), type = "l", ylim = c(0, 1.1),
  xlab = "Frequency", ylab = c("Magnitude"))
lines(c2fr$w / pi, abs(c2fr$h), col = "red")

}
\references{
\url{https://en.wikipedia.org/wiki/Chebyshev_filter}
}
\seealso{
\code{\link{Arma}}, \code{\link{filter}}, \code{\link{butter}},
  \code{\link{ellip}}, \code{\link{cheb2ord}}
}
\author{
Paul Kienzle, \email{pkienzle@users.sf.net},\cr
  Doug Stewart, \email{dastew@sympatico.ca}.\cr
  Conversion to R Tom Short,\cr
  adapted by Geert van Boxtel, \email{G.J.M.vanBoxtel@gmail.com}.
}
