% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/syntable.R
\name{syntable}
\alias{syntable}
\title{Synoptic tables and calculation of fidelity and differential species character}
\usage{
syntable(spec, cluster, abund = "perc", type = "percfreq")
}
\arguments{
\item{spec}{Species matrix or dataframe with species in columns and samples in rows.
Values must be numeric ("." as decimal character) or integer.  Missing values, NA or NaN are not allowed.
Species and sample names must be defined as column- and rownames.}

\item{cluster}{Integer vector with classification cluster identity. Ensure matching order of cluster identity
and samples in dataframe for correct allocation of cluster numbers to samples.}

\item{abund}{Data input type. Define whether input species matrix or dataframe is presence/absence data
(\code{abund = "freq"}) or percentage cover (\code{abund = "perc"}, default).}

\item{type}{Type of synoptic table output, \code{type = c("percfreq", "totalfreq", "mean", "median", "diffspec", "phi")}.
See Details for description of options.}
}
\value{
The function returns a list of result components.
  \item{\code{abund }}{abundance of each species (in decreasing order)}
  \item{\code{$syntable }}{unordered synoptic table for given species and clusters}
  \item{\code{$others }}{names vector of species that are not included in synoptic table due to failing treshold values for
  \code{min1} and/or \code{min2}}
  \item{\code{$samplesize }}{total samples in clusters}
Additionally for differential taxa calculation:
  \item{\code{$onlydiff }}{synoptic table only with differentiating taxa}
  \item{\code{$differentials }}{lists diagnostic taxa for each cluster}
}
\description{
This function calculates an unordered synoptic table for vegetation unit analysis from
a species-sample dataframe and a numeric vector of cluster identity input.
Synoptic table options for displaying species in clusters is absolute or percentage frequency, mean/median cover values,
differential character (positive, negative, positive-negative, according to Tsiripidis et al. 2009) or fidelity phi
(Bruelheide 2000).
Created unordered synoptic table can be sorted automatically with \code{\link{synsort}} function in this package.
}
\section{Details}{

For synoptic table calculation, six types are available.
  \itemize{\item{\code{type = "percfreq" }}{Default, creates a percentage frequency table}
  \item{\code{type = "totalfreq" }}{Creates an absolute frequency table}
  \item{\code{type = "mean" }}{Calculates mean of species values per cluster}
  \item{\code{type = "median" }}{Calculates median of species values per cluster}
  \item{\code{type = "diffspec" }}{Calculates differential character of species according to Tsiripidis et al. 2009.
   Synoptic table gives specification of positive (p), negative (n), positive-negative (pn) or no differential character (-).
   Consider that differential character is always restricted to some and not necessarily all of the other
   units, so regarding percentage frequency is essential for correct interpretation of diagnostic character of species.}
  \item{\code{type = "phi" }}{Calculates fidelity measure phi (algorithm basing on Sokal & Rohlf 1995, Bruelheide 2000).
   Values are ranging between -1 and 1 with high values near 1 indicating high fidelity.}}

For sorting synoptic tables, use \code{\link{synsort}} function.
See also \code{\link[vegdata]{syntab}} function from \code{vegdata} package for creating ordered synoptic tables from turboveg
inputs.
}

\examples{
## Synoptic table of Scheden vegetation data
library(cluster)
pam1 <- pam(schedenveg, 4)  # PAM clustering with 4 clusters output

## 1) unordered synoptic percentage frequency table
unordered <- syntable(schedenveg, pam1$clustering, abund = "perc",
                      type = "percfreq")
unordered                   # view results

## 2) differential species analysis
differential <- syntable(schedenveg, pam1$clustering, abund = "perc",
                         type = "diffspec")
# show complete table with differential character of species
differential$syntable
# list differential species for second cluster
differential$differentials$group2

## 3) Synoptic table with phi fidelity
phitable <- syntable(schedenveg, pam1$clustering, abund = "perc",
                     type = "phi")
phitable
}
\references{
Bruelheide, H. (2000): A new measure of fidelity and its application to defining species groups. - \emph{Journal of Vegetation Science} 11: 167-178.

Chytry, M., Tichy, L., Holt, J., Botta-Dukat, Z. (2002): Determination of diagnostic species with statistical fidelity measures. \emph{Journal of Vegetation Science} 13: 79-90.

Sokal, R.R. & Rohlf, F.J. (1995): Biometry. 3rd edition Freemann, New York.

Tsiripidis, I., Bergmeier, E., Fotiadis, G. & Dimopoulos, P. (2009): A new algorithm for the determination of differential taxa. - \emph{Journal of Vegetation Science} 20: 233-240.
}
\seealso{
\code{\link{synsort}}
}
\author{
Jenny Schellenberg (\email{jschell@gwdg.de})
}
