% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/geom-label.R, R/geom-text.r
\name{geom_label}
\alias{geom_label}
\alias{geom_text}
\title{Textual annotations.}
\usage{
geom_label(mapping = NULL, data = NULL, stat = "identity",
  position = "identity", ..., parse = FALSE, nudge_x = 0, nudge_y = 0,
  label.padding = unit(0.25, "lines"), label.r = unit(0.15, "lines"),
  label.size = 0.25, na.rm = FALSE, show.legend = NA,
  inherit.aes = TRUE)

geom_text(mapping = NULL, data = NULL, stat = "identity",
  position = "identity", ..., parse = FALSE, nudge_x = 0, nudge_y = 0,
  check_overlap = FALSE, na.rm = FALSE, show.legend = NA,
  inherit.aes = TRUE)
}
\arguments{
\item{mapping}{Set of aesthetic mappings created by \code{\link{aes}} or
\code{\link{aes_}}. If specified and \code{inherit.aes = TRUE} (the
default), it is combined with the default mapping at the top level of the
plot. You must supply \code{mapping} if there is no plot mapping.}

\item{data}{The data to be displayed in this layer. There are three
   options:

   If \code{NULL}, the default, the data is inherited from the plot
   data as specified in the call to \code{\link{ggplot}}.

   A \code{data.frame}, or other object, will override the plot
   data. All objects will be fortified to produce a data frame. See
   \code{\link{fortify}} for which variables will be created.

   A \code{function} will be called with a single argument,
   the plot data. The return value must be a \code{data.frame.}, and
   will be used as the layer data.}

\item{stat}{The statistical transformation to use on the data for this
layer, as a string.}

\item{position}{Position adjustment, either as a string, or the result of
a call to a position adjustment function.}

\item{...}{other arguments passed on to \code{\link{layer}}. These are
often aesthetics, used to set an aesthetic to a fixed value, like
\code{color = "red"} or \code{size = 3}. They may also be parameters
to the paired geom/stat.}

\item{parse}{If TRUE, the labels will be parsed into expressions and
displayed as described in ?plotmath}

\item{nudge_x, nudge_y}{Horizontal and vertical adjustment to nudge labels by.
Useful for offsetting text from points, particularly on discrete scales.}

\item{label.padding}{Amount of padding around label. Defaults to 0.25 lines.}

\item{label.r}{Radius of rounded corners. Defaults to 0.15 lines.}

\item{label.size}{Size of label border, in mm.}

\item{na.rm}{If \code{FALSE} (the default), removes missing values with
a warning.  If \code{TRUE} silently removes missing values.}

\item{show.legend}{logical. Should this layer be included in the legends?
\code{NA}, the default, includes if any aesthetics are mapped.
\code{FALSE} never includes, and \code{TRUE} always includes.}

\item{inherit.aes}{If \code{FALSE}, overrides the default aesthetics,
rather than combining with them. This is most useful for helper functions
that define both data and aesthetics and shouldn't inherit behaviour from
the default plot specification, e.g. \code{\link{borders}}.}

\item{check_overlap}{If \code{TRUE}, text that overlaps previous text in the
same layer will not be plotted. A quick and dirty way}
}
\description{
\code{geom_text} adds text directly to the plot. \code{geom_label} draws
a rectangle underneath the text, making it easier to read.
}
\details{
Note the the "width" and "height" of a text element are 0, so stacking
and dodging text will not work by default, and axis limits are not
automatically expanded to include all text. Obviously, labels do have
height and width, but they are physical units, not data units. The amount of
space they occupy on that plot is not constant in data units: when you
resize a plot, labels stay the same size, but the size of the axes changes.
}
\section{Aesthetics}{

\Sexpr[results=rd,stage=build]{ggplot2:::rd_aesthetics("geom", "text")}
}

\section{\code{geom_label}}{

Currently \code{geom_label} does not support the \code{rot} parameter and
is considerably slower than \code{geom_text}. The \code{fill} aesthetic
controls the background colour of the label.
}

\section{Alignment}{

You can modify text alignment with the \code{vjust} and \code{hjust}
aesthetics. These can either be a number between 0 (right/bottom) and
1 (top/left) or a character ("left", "middle", "right", "bottom", "center",
"top"). There are two special alignments: "inward" and "outward".
Inward always aligns text towards the center, and outward aligns
it away from the center
}
\examples{
p <- ggplot(mtcars, aes(wt, mpg, label = rownames(mtcars)))

p + geom_text()
# Avoid overlaps
p + geom_text(check_overlap = TRUE)
# Labels with background
p + geom_label()
# Change size of the label
p + geom_text(size = 10)

# Set aesthetics to fixed value
p + geom_point() + geom_text(hjust = 0, nudge_x = 0.05)
p + geom_point() + geom_text(vjust = 0, nudge_y = 0.5)
p + geom_point() + geom_text(angle = 45)
\dontrun{
# Doesn't work on all systems
p + geom_text(family = "Times New Roman")
}

# Add aesthetic mappings
p + geom_text(aes(colour = factor(cyl)))
p + geom_text(aes(colour = factor(cyl))) +
  scale_colour_discrete(l = 40)
p + geom_label(aes(fill = factor(cyl)), colour = "white", fontface = "bold")

p + geom_text(aes(size = wt))
# Scale height of text, rather than sqrt(height)
p + geom_text(aes(size = wt)) + scale_radius(range = c(3,6))

# You can display expressions by setting parse = TRUE.  The
# details of the display are described in ?plotmath, but note that
# geom_text uses strings, not expressions.
p + geom_text(aes(label = paste(wt, "^(", cyl, ")", sep = "")),
  parse = TRUE)

# Add a text annotation
p +
  geom_text() +
  annotate("text", label = "plot mpg vs. wt", x = 2, y = 15, size = 8, colour = "red")

\donttest{
# Aligning labels and bars --------------------------------------------------
df <- data.frame(
  x = factor(c(1, 1, 2, 2)),
  y = c(1, 3, 2, 1),
  grp = c("a", "b", "a", "b")
)

# ggplot2 doesn't know you want to give the labels the same virtual width
# as the bars:
ggplot(data = df, aes(x, y, fill = grp, label = y)) +
  geom_bar(stat = "identity", position = "dodge") +
  geom_text(position = "dodge")
# So tell it:
ggplot(data = df, aes(x, y, fill = grp, label = y)) +
  geom_bar(stat = "identity", position = "dodge") +
  geom_text(position = position_dodge(0.9))
# Use you can't nudge and dodge text, so instead adjust the y postion
ggplot(data = df, aes(x, y, fill = grp, label = y)) +
  geom_bar(stat = "identity", position = "dodge") +
  geom_text(aes(y = y + 0.05), position = position_dodge(0.9), vjust = 0)

# To place text in the middle of each bar in a stacked barplot, you
# need to do the computation yourself
df <- transform(df, mid_y = ave(df$y, df$x, FUN = function(val) cumsum(val) - (0.5 * val)))

ggplot(data = df, aes(x, y, fill = grp, label = y)) +
 geom_bar(stat = "identity") +
 geom_text(aes(y = mid_y))

# Justification -------------------------------------------------------------
df <- data.frame(
  x = c(1, 1, 2, 2, 1.5),
  y = c(1, 2, 1, 2, 1.5),
  text = c("bottom-left", "bottom-right", "top-left", "top-right", "center")
)
ggplot(df, aes(x, y)) +
  geom_text(aes(label = text))
ggplot(df, aes(x, y)) +
  geom_text(aes(label = text), vjust = "inward", hjust = "inward")
}
}

