% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/density.R
\name{density_unbounded}
\alias{density_unbounded}
\title{Unbounded density estimator}
\usage{
density_unbounded(
  x,
  weights = NULL,
  n = 512,
  bandwidth = "nrd0",
  adjust = 1,
  kernel = "gaussian",
  trim = FALSE
)
}
\arguments{
\item{x}{numeric vector containing a sample to compute a density estimate for.}

\item{weights}{optional numeric vector of weights to apply to \code{x}.}

\item{n}{numeric: the number of grid points to evaluate the density estimator at.}

\item{bandwidth}{bandwidth of the density estimator. One of:
\itemize{
\item a numeric: the bandwidth, as the standard deviation of the kernel
\item a function: a function taking \code{x} (the sample) and returning the bandwidth
\item a string: the suffix of the name of a function starting with \code{"bw."} that
will be used to determine the bandwidth. See \code{\link[=bw.nrd0]{bw.nrd0()}} for a list.
}}

\item{adjust}{numeric: the bandwidth for the density estimator is multiplied
by this value. See \code{\link[stats:density]{stats::density()}}.}

\item{kernel}{string: the smoothing kernel to be used. This must partially
match one of \code{"gaussian"}, \code{"rectangular"}, \code{"triangular"}, \code{"epanechnikov"},
\code{"biweight"}, \code{"cosine"}, or \code{"optcosine"}. See \code{\link[stats:density]{stats::density()}}.}

\item{trim}{Should the density estimate be trimmed to the bounds of the data?}
}
\value{
An object of class \code{"density"}, mimicking the output format of
\code{stats:density()}, with the following components:
\itemize{
\item \code{x}: The grid of points at which the density was estimated.
\item \code{y}: The estimated density values.
\item \code{bw}: The bandwidth.
\item \code{n}: The sample size of the \code{x} input argument.
\item \code{call}: The call used to produce the result, as a quoted expression.
\item \code{data.name}: The deparsed name of the \code{x} input argument.
\item \code{has.na}: Always \code{FALSE} (for compatibility).
}

This allows existing methods (like \code{print()} and \code{plot()}) to work if desired.
This output format (and in particular, the \code{x} and \code{y} components) is also
the format expected by the \code{density} argument of the \code{\link[=stat_slabinterval]{stat_slabinterval()}}
and the \code{smooth_} family of functions.
}
\description{
Unbounded density estimator using \code{\link[stats:density]{stats::density()}}.
Supports \link[=automatic-partial-functions]{automatic partial function application}.
}
\examples{
library(distributional)
library(dplyr)
library(ggplot2)

# For compatibility with existing code, the return type of density_unbounded()
# is the same as stats::density(), ...
set.seed(123)
x = rbeta(5000, 1, 3)
d = density_unbounded(x)
d

# ... thus, while designed for use with the `density` argument of
# stat_slabinterval(), output from density_unbounded() can also be used with
# base::plot():
plot(d)

# here we'll use the same data as above, but pick either density_bounded()
# or density_unbounded() (which is equivalent to stats::density()). Notice
# how the bounded density (green) is biased near the boundary of the support,
# while the unbounded density is not.
data.frame(x) \%>\%
  ggplot() +
  stat_slab(
    aes(xdist = dist), data = data.frame(dist = dist_beta(1, 3)),
    alpha = 0.25
  ) +
  stat_slab(aes(x), density = "bounded", fill = NA, color = "#d95f02", alpha = 0.5) +
  stat_slab(aes(x), density = "unbounded", fill = NA, color = "#1b9e77", alpha = 0.5) +
  scale_thickness_shared() +
  theme_ggdist()
}
\seealso{
Other density estimators: 
\code{\link{density_auto}()},
\code{\link{density_bounded}()}
}
\concept{density estimators}
