% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sratoolkit_functions.R
\name{fasterqDump}
\alias{fasterqDump}
\title{Download or convert fastq data from NCBI Sequence Read Archive using multiple threads}
\usage{
fasterqDump(queryMetadata_object, sratoolkitPath = "",
  outputDirectory = "./", arguments = "-p", filenames = "accessions",
  source = "sra", cleanup = FALSE, fasterqDumpHelp = FALSE)
}
\arguments{
\item{queryMetadata_object}{A list object returned from `queryMetadata` where one of the 
entities queried was `fastqMetadata`.}

\item{sratoolkitPath}{String. A path to a local copy of sratoolkit. Only necessary if sratoolkit
is not on your $PATH. Assumes executables are inside `bin`.}

\item{outputDirectory}{String. A path to the directory where you would like the files to be stored.}

\item{arguments}{A string variable of arguments to be passed directly to `fasterq-dump`.
Defaults to "-p" to show progress. Use fasterqDumpHelp = TRUE to see a list of arguments.}

\item{filenames}{String. How would you like the downloaded fastq files to be named?
"accessions" names files with SRA accession numbers
"IDs" names files with their materialSampleID
"locality_IDs" names files with their locality and materialSampleID.}

\item{source}{String. `fasterq-dump` can retrieve files directly from SRA, or it can convert .sra files
previously downloaded with `prefetch` that are in the current working directory. 
"sra" downloads from SRA
"local" converts .sra files in the current working directory.}

\item{cleanup}{Logical. cleanup = T will delete any intermediate .sra files.}

\item{fasterqDumpHelp}{Logical. fasterqDumpHelp = T will show the help page for `fasterq-dump` and then quit.}
}
\value{
This function will not return anything within r. It simply downloads fastq files. It will print command line
stdout to the console, and also provide a start and end time and amount of time elapsed during the download.
}
\description{
`fasterqDump()` uses the SRAtoolkit command-line function `fasterq-dump` to download fastq
files from all samples returned by a \code{\link{queryMetadata}} query of GEOME, when one of the
entities queried was `fastqMetadata`
}
\details{
The `fasterq-dump` tool uses temporary files and multi-threading to speed up the extraction of FASTQ from SRA-accessions.
This function works best with sratoolkit functions of version 2.9.6 or greater. \href{https://www.ncbi.nlm.nih.gov/sra/docs/toolkitsoft/}{SRAtoolkit} functions can (ideally)
be in your $PATH, or you can supply a path to them using the sratoolkitPath argument.

`fasterqDump()` downloads files to the current working directory unless a different one is assigned through outputDirectory.

Change the number of threads by adding "-e X" to arguments where X is the number of threads.

`fasterq-dump` will automatically split paired-end data into three files with:
\itemize{
 \item file_1.fastq having read 1
 \item file_2.fastq having read 2
 \item file.fastq having unmatched reads
 }
`fasterqDump()` can then rename these files based on their materialSampleID and locality.

Note that `fasterq-dump` will store temporary files in ~/ncbi/public/sra by default unless
you pass "-t /path/to/temp/dir" to arguments. \emph{Make sure to periodically delete these temporary files.}
}
\examples{
\donttest{
# Run a query of GEOME first
acaoli <- queryMetadata(
    entity = "fastqMetadata", 
    query = "genus = Acanthurus AND specificEpithet = olivaceus AND _exists_:bioSample", 
    select=c("Event"))

#trim to 3 entries for expediency
acaoli$fastqMetadata<-acaoli$fastqMetadata[1:3,]
acaoli$Event<-acaoli$Event[1:3,]

# Download straight from SRA, naming files with their locality and materialSampleID
fasterqDump(queryMetadata_object = acaoli, filenames = "IDs", source = "sra")

# A generally faster option is to run prefetch first, followed by fasterqDump, with cleanup = T to 
# remove the prefetched .sra files.
prefetch(queryMetadata_object = acaoli)
fasterqDump(queryMetadata_object = acaoli, filenames = "IDs", source = "local", cleanup = T)
}
}
\seealso{
\url{https://www.ncbi.nlm.nih.gov/sra/docs/toolkitsoft/} to download pre-compiled executables for sratoolkit or
\url{https://github.com/ncbi/sra-tools/wiki/Building-and-Installing-from-Source>} to install from source

This function will not work on Windows systems because fasterq-dump is not currently available for Windows. 
See \code{\link{fastqDump}} if you use Windows. See \code{\link{prefetch}} to download .sra files prior to converting them locally.
}
