#' Collect the set of species observed within the specified filters
#'
#' While there are reasons why users may need to check every record meeting their
#' search criteria (i.e. using [atlas_occurrences()]), a common use case
#' is to simply identify which species occur in a specified region, time period,
#' or taxonomic group. This function returns a `data.frame` with one row
#' per species, and columns giving associated taxonomic information.
#'
#' @param request optional `data_request` object: generated by a call to
#' [galah_call()].
#' @param identify `data.frame`: generated by a call to
#' [galah_identify()].
#' @param filter `data.frame`: generated by a call to
#' [galah_filter()]
#' @param geolocate `string`: generated by a call to
#' [galah_geolocate()]
#' @param data_profile `string`: generated by a call to
#' [galah_apply_profile()]
#' @return An object of class `tbl_df` and `data.frame` (aka a tibble), 
#' returning matching species The `data.frame` object has attributes listing of 
#' the user-supplied arguments of the `data_request` 
#' (i.e., identify, filter, geolocate, columns)
#' @details
#' The primary use case of this function is to extract species-level information
#' given a set of criteria defined by [search_taxa()],
#' [galah_filter()] or [galah_geolocate()]. If the purpose
#' is simply to get taxonomic information that is not restricted by filtering,
#' then [search_taxa()] is more efficient. Similarly, if counts are
#' required that include filter but without returning taxonomic detail, then
#' [atlas_counts()] is more efficient (see examples).
#' @examples \dontrun{
#' # First register a valid email address
#' galah_config(email = "ala4r@ala.org.au")
#' 
#' # Get a list of species within genus "Heleioporus"
#' # (every row is a species with associated taxonomic data)
#' galah_call() |>
#'   galah_identify("Heleioporus") |>
#'   atlas_species()
#' 
#' # Get a list of species within family "Peramelidae"
#' galah_call() |>
#'   galah_identify("peramelidae") |>
#'   atlas_species()
#'   
#' # Or alternatively
#' request_data(type = "species") |>
#'   identify("peramelidae") |>
#'   collect()
#' 
#' # It's good idea to find how many species there are before downloading
#' galah_call() |>
#'   galah_identify("Heleioporus") |>
#'   atlas_counts(type = "species")
#' # Or alternatively
#' request_data(type = "species") |>
#'   identify("Heleioporus") |>
#'   count() |>
#'   collect()
#' }
#' @importFrom potions pour
#' @export
atlas_species <- function(request = NULL,
                          identify = NULL,
                          filter = NULL,
                          geolocate = NULL,
                          data_profile = NULL
                          ) {
                            
  # capture supplied arguments
  args <- as.list(environment())
  
  # handle type correctly
  if(!is.null(args$request)){
    args$request$type <- "species"
  }else{
    args <- c(list(type = "species"), args)
  }
  
  # convert to `data_request` object
  check_atlas_inputs(args) |>
    collapse() |>
    collect()
}