#' Return occurrence records
#'
#' The most common form of data stored by living atlases are observations of
#' individual life forms, known as 'occurrences'. This function allows the
#' user to search for occurrence records that match their specific criteria,
#' and return them as a `data.frame` for analysis. Optionally,
#' the user can also request a DOI for a given download to facilitate citation
#' and re-use of specific data resources.
#'
#' @param request optional `data_request` object: generated by a call to
#' [galah_call()].
#' @param identify `data.frame`: generated by a call to
#' [galah_identify()].
#' @param filter `data.frame`: generated by a call to
#' [galah_filter()]
#' @param geolocate `string`: generated by a call to
#' [galah_geolocate()]
#' @param data_profile `string`: generated by a call to
#' [galah_apply_profile()]
#' @param select `data.frame`: generated by a call to
#' [galah_select()] 
#' @param mint_doi `logical`: by default no DOI will be generated. Set to
#' `TRUE` if you intend to use the data in a publication or similar
#' @param doi  
#'   `r lifecycle::badge("deprecated")` Use `collect_occurrences` instead. 
#'   
#'   `string`: this argument enables retrieval of occurrence records previously 
#'   downloaded from the ALA, using the DOI generated by the data.
#' @param refresh_cache `logical`: if set to `TRUE` and 
#' `galah_config(caching = TRUE)` then files cached from a previous query will 
#' be replaced by the current query
#' @details
#' Note that unless care is taken, some queries can be particularly large.
#' While most cases this will simply take a long time to process, if the number
#' of requested records is >50 million the call will not return any data. Users
#' can test whether this threshold will be reached by first calling
#' [atlas_counts()] using the same arguments that they intend to pass to
#' `atlas_occurrences`(). It may also be beneficial when requesting a large
#' number of records to show a progress bar by setting `verbose = TRUE` in
#' [galah_config()].
#' @return An object of class `tbl_df` and `data.frame` (aka a tibble) of 
#' occurrences, containing columns as specified by [galah_select()]. 
#' The `data.frame` object has the following attributes:
#' 
#' * a listing of the user-supplied arguments of the `data_request` 
#' (i.e., identify, filter, geolocate, select)
#' * a `doi` of the data download
#' * the `search_url` of the query to ALA API
#' 
#' @section Examples:
#' ```{r, child = "man/rmd/setup.Rmd"}
#' ```
#' 
#' Search for occurrences matching a taxon identifier
#' 
#' ```{r, comment = "#>", collapse = TRUE, results = "hide", eval = FALSE}
#' galah_config(email = "your-email@email.com")
#' galah_call() |>
#'   galah_identify("Reptilia") |>
#'   atlas_occurrences()
#' ```
#'
#' Search for occurrences in a year range
#' 
#' ```{r, comment = "#>", collapse = TRUE, results = "hide", eval = FALSE}
#' galah_call() |>
#'   galah_filter(year >= 2010, year <= 2020) |>
#'   atlas_occurrences()
#' ```
#'
#' Search for occurrences in a WKT-specified area
#' 
#' ```{r, comment = "#>", collapse = TRUE, results = "hide", eval = FALSE}
#' polygon <- "POLYGON((146.24960 -34.05930,146.37045 -34.05930,146.37045 -34.152549,146.24960 -34.15254,146.24960 -34.05930))"
#' galah_call() |> 
#'   galah_geolocate(polygon) |>
#'   atlas_occurrences()
#' ```
#' 
#' You can also download occurrence records by piping with `%>%` if you prefer.
#' 
#' ```{r, comment = "#>", collapse = TRUE, results = "hide", eval = FALSE}
#' galah_call() %>%
#'   galah_identify("Reptilia") %>%
#'   galah_filter(year >= 2010) %>%
#'   galah_geolocate(polygon) %>%
#'   atlas_occurrences()
#' ```
#' 
#' @export
atlas_occurrences <- function(request = NULL, 
                              identify = NULL, 
                              filter = NULL, 
                              geolocate = NULL,
                              data_profile = NULL,
                              select = NULL,
                              mint_doi = FALSE, 
                              doi = NULL, # check missingness code
                              refresh_cache = FALSE
                              ) {
  if(!is.null(request)){
    check_data_request(request)
    current_call <- update_galah_call(request, 
      identify = identify,
      filter = filter,
      geolocate = geolocate,
      data_profile = data_profile,
      select = select,
      mint_doi = mint_doi, # NOTE: check behaviour of update_galah_call here
      doi = doi,
      refresh_cache = refresh_cache
    ) 
          
  }else{
    current_call <- galah_call(
      identify = identify,
      filter = filter,
      geolocate = geolocate,
      data_profile = data_profile,
      select = select,
      mint_doi = mint_doi,
      doi = doi,
      refresh_cache = refresh_cache         
    )
  }

  # subset to available arguments
  custom_call <- current_call[
    names(current_call) %in% names(formals(atlas_occurrences_internal))]
  if(!is.null(doi)){
    custom_call <- custom_call["doi"]
  }
  class(custom_call) <- "data_request"
       
  # check for caching
  caching <- getOption("galah_config")$caching
  cache_file <- cache_filename("occurrences", unlist(custom_call))
  if (caching && file.exists(cache_file) && !refresh_cache) {
    return(read_cache_file(cache_file))
  }
       
  # run function using do.call
  result <- do.call(atlas_occurrences_internal, custom_call)
  attr(result, "data_request") <- custom_call

  # if caching requested, save
  if (caching) {
    write_cache_file(object = result, 
                     data_type = "occurrences",
                     cache_file = cache_file)
  }

  result                                
}


# internal workhorse function
atlas_occurrences_internal <- function(identify = NULL,
                                       filter = NULL, 
                                       geolocate = NULL,
                                       data_profile = NULL,
                                       select = NULL,
                                       mint_doi = FALSE, 
                                       doi = NULL, 
                                       refresh_cache = FALSE) {
                                         
  # check whether API exists
  occurrences_url <- atlas_url("records_occurrences")

  verbose <- getOption("galah_config")$verbose
  assert_that(is.logical(mint_doi))
  if(!is.null(doi)){
    abort("Argument `doi` is deprecated; use `collect_occurrences()` instead")
  }
  
  # If no filters are specified, reject
  if(
    all(unlist(lapply(list(identify, filter, geolocate, doi), is.null)))
  ){
    bullets <- c(
      "Your data request was too large.",
      i = "A maximum of 50 million records can be retrieved at once.",
      i = "Please narrow the query and try again."
    )
    abort(bullets, call = caller_env())
  }
  
  # set default columns
  if(is.null(select)){
    select <- galah_select(group = "basic")
  }
  
  # ensure profile works from galah_filter as well as galah_profile  
  if(is.null(data_profile)){
    if(is.null(filter)){
      profile <- NULL
    }else{
      profile <- extract_profile(filter)
    }
  }else{
    profile <- data_profile$data_profile
  }
  
  query <- build_query(identify, filter, geolocate, select, profile)

  # Check record count
  if (getOption("galah_config")$run_checks) {
    count <- record_count(query)
    if (is.null(count)){
      bullets <- c(
        "Calling the API failed for `atlas_occurrences`.",
        i = "This might mean that the selected system is down. Double check that your query is correct.",
        i = "If you continue to see this message, please email support@ala.org.au."
      )
      inform(bullets)
      return(tibble())
    }else{
      check_count(count) # aborts under selected circumstances
    }
  }
  
  assertion_select <- select[select$type == "assertions", ]

  query <- c(query, 
    fields = build_columns(select[select$type != "assertions", ]),
    qa = build_assertion_columns(assertion_select),
    emailNotify = email_notify(),
    sourceTypeId = 2004,
    reasonTypeId = getOption("galah_config")$download_reason_id,
    email = user_email(), 
    dwcHeaders = "true")

  if (mint_doi & getOption("galah_config")$atlas == "Australia") {
    query$mintDoi <- "true"
  }
    
  # Get data
  tmp <- tempfile()
  download_resp <- wait_for_download(query)
  if(is.null(download_resp)){
    inform("Calling the API failed for `atlas_occurrences`")
    return(tibble())
  }
  
  # download from url
  result_df <- url_download(download_resp)

  result_df
}


get_doi <- function(mint_doi, data_path) {
  doi <- NA
  if (as.logical(mint_doi)) {
    tryCatch(
      doi <- as.character(
        read.table(unz(data_path, "doi.txt"))$V1),
      warning = function(e) {
        e$message <- "No DOI was generated for this download. The DOI server may
        be down or, if this is a cached result, may not have been generated for
        the original download."
      })
  }
  return(doi)
}

wait_for_download <- function(query) {

  url <- atlas_url("records_occurrences")
  status_initial <- atlas_GET(url, params = query, on_error = occ_error_handler)
    
  if(is.null(status_initial)){
    return(NULL)
  }

  # check running status, with rate limiting
  status_post_queue <- check_queue(status_initial)
  if(!is.null(status_post_queue$status)){
    if(status_post_queue$status != "finished"){
      status <- check_running(status_post_queue)
    }else{
      status <- status_post_queue
    }
  }else{
    status <- NULL
  }

  return(status$downloadUrl)
}

api_intervals <- function(){
  c(
    rep(0.5, 4),
    rep(1, 3),
    rep(2, 5),
    rep(5, 8),
    rep(10, 10),
    rep(30, 10))
}

# check queue status, with rate limiting
check_queue <- function(status_initial){
  interval_times <- api_intervals()
  n_intervals <- length(interval_times)
  status <- status_initial
  iter <- 1
  queue_size <- status$queueSize
  
  verbose <- getOption("galah_config")$verbose
  if(verbose){
    cat(paste0("Checking queue\nCurrent queue size: ", queue_size))
  }
  
  while(status$status == "inQueue"){
    if(verbose){
      if(is.null(status$queueSize)){status$queueSize <- 0}
      if(status$queueSize < queue_size){
        queue_size <- status$queueSize
        cat(paste0(" ", queue_size, " "))
      }else{
        cat(".")
      }
    }
    status <- atlas_GET(status$statusUrl)
    if(is.null(status$statusUrl)){
      break()
    }else{
      if(iter <= n_intervals){
        lag <- interval_times[iter]
      }else{
        lag <- 60
      }
      Sys.sleep(lag)
    }
    iter <- iter + 1
  }
  return(status)
}

check_running <- function(status){
  interval_times <- api_intervals()
  n_intervals <- length(interval_times)
  iter <- 1
  
  verbose <- getOption("galah_config")$verbose
  if(verbose){
    cat("\nRunning query on selected atlas\n")
    pb <- txtProgressBar(max = 1, style = 3)
  }
  
  while(status$status == "running"){
    if(verbose){
      if(is.null(status$records)){status$records <- 0}
      if(is.null(status$totalRecords)){status$totalRecords <- 1}
      setTxtProgressBar(pb, status$records/status$totalRecords)
    }
    status <- atlas_GET(status$statusUrl)
    if(is.null(status$statusUrl)){
      break()
    }else{
      if(iter <= n_intervals){
        lag <- interval_times[iter]
      }else{
        lag <- 60
      }
      Sys.sleep(lag)
    }
    iter <- iter + 1
  }
  
  if(verbose){
    setTxtProgressBar(pb, value = 1)
    close(pb)
  }
  return(status)
}

check_count <- function(count, error_call = caller_env()) {
  if (count == 0) {
    abort("This query does not match any records.", call = error_call)
  } else if (count > 50000000) {
    bullets <- c(
      "Your data request was too large.",
      i = "A maximum of 50 million records can be retrieved at once.",
      i = "Please narrow the query and try again."
    )
    abort(bullets, call = error_call)
  } else {
    if (getOption("galah_config")$verbose) {
      inform(glue("This query will return {count} records"))
      }
  }
}

email_notify <- function() {
  notify <- as.logical(getOption("galah_config")$send_email)
  if (is.na(notify)) {
    notify <- FALSE
  }
  # ala api requires lowercase
  ifelse(notify, "true", "false")
}

user_email <- function(error_call = rlang::caller_env()) {
  email <- getOption("galah_config")$email
  if (email == "") {
    email <- Sys.getenv("email")
  }
  if (email == "") {
    bullets <- c(
      "No user email was found.",
      i = glue("To download occurrence records you must provide a valid email ",
                     "address registered with the selected atlas using `galah_config(email = )`")
    )
    abort(bullets, call = error_call)
  }
  email
}

occ_error_handler <- function(code, error_call = rlang::caller_env()) {
  if (code == 403) {
    bullets <- c(
      "Status code 403 was returned.",
      i = glue("Is the email you provided to `galah_config()` registered with the selected atlas?")
    )
    inform(bullets)
  }
  if (code == 504) {
    bullets <- c(
      "Status code 504 was returned.",
      i = "This usually means that the selected API is down.",
      i = "If you continue to receive this error, please email support@ala.org.au"
    )
    inform(bullets)
  }
}
