% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/futility.R
\name{completeTrial.byArm}
\alias{completeTrial.byArm}
\title{Treatment Arm-Specific Simulation-Based Completion of a Randomized Efficacy Trial with a Time-to-Event Endpoint and Fixed Follow-up Using an Interim Data-set}
\usage{
completeTrial.byArm(interimData, nTrials, trtNames, N, enrollRate = NULL,
  enrollRatePeriod, eventPriorWeight, eventPriorRate, missVaccProb = NULL,
  fuTime, visitSchedule, visitSchedule2 = NULL, saveFile = NULL,
  saveDir = NULL, randomSeed = NULL)
}
\arguments{
\item{interimData}{a data frame capturing observed data at an interim timepoint that contains one row per enrolled participant and the following variables: \code{arm} (treatment arm), \code{schedule2} (an indicator that a participant follows the \code{visitSchedule2} schedule, e.g., participants who discontinue study product administration may remain in primary follow-up on a different schedule), \code{entry} (number of weeks since the reference date until the enrollment date), \code{exit} (number of weeks since the reference date until the trial exit date defined as the date of either infection diagnosis, dropout, or primary follow-up completion, whichever occurs first; \code{NA} for participants still in primary follow-up), \code{last_visit_dt} (number of weeks since the reference date until the last visit date), \code{event} (event indicator), \code{dropout} (dropout indicator), \code{complete} (indicator of completed follow-up), \code{followup} (indicator of being in primary follow-up). The reference date is defined as the enrollment date of the first participant. The variables \code{entry}, \code{exit}, and \code{last_visit_dt} use week as the unit of time. Month is defined as 52/12 weeks.}

\item{nTrials}{the number of trials to be simulated}

\item{trtNames}{a character vector of treatment labels as specified in \code{interimData$arm} determining the order of treatment arms in other input arguments}

\item{N}{a numeric vector specifying the target number of enrolled participants in each treatment arm, with the arms in the same order as in \code{trtNames}}

\item{enrollRate}{a treatment arm-pooled weekly enrollment rate used for completing enrollment if \code{interimData}'s enrollment is incomplete. If \code{NULL} (default), the rate is calculated as the average over the last \code{enrollRatePeriod} weeks of enrollment in \code{interimData}. If equal to a numeric value, then \code{enrollRatePeriod} is ignored.}

\item{enrollRatePeriod}{the length (in weeks) of the time period preceding the time of the last enrolled participant in \code{interimData} that the average weekly enrollment rate will be based on and used for completing enrollment. If \code{NULL} (default), then \code{enrollRate} must be specified.}

\item{eventPriorWeight}{a numeric value in \eqn{[0,1]} representing a weight assigned to the prior gamma distribution of the treatment arm-specific event rates at the time when 50\% of the estimated person-time at risk in each arm has been accumulated (see the vignette)}

\item{eventPriorRate}{a numeric vector of treatment arm-specific prior mean incidence rates for the endpoint, expressed as numbers of events per person-year at risk, with the arms in the same order as in \code{trtNames}}

\item{missVaccProb}{a probability of being excluded from the per-protocol cohort. If \code{NULL} (default), no per-protocol indicator is generated; if specified, the indicator is sampled from the Bernoulli distribution with probability \code{missVaccProb}.}

\item{fuTime}{a follow-up time (in weeks) of each participant}

\item{visitSchedule}{a numeric vector of visit weeks at which testing for the endpoint is conducted}

\item{visitSchedule2}{a numeric vector of visit weeks at which testing for the endpoint is conducted in a subset of participants (e.g., those who discontinue administration of the study product but remain in follow-up). If \code{NULL} (default), everyone is assumed to follow \code{visitSchedule}.}

\item{saveFile}{a character string specifying an \code{.RData} file storing the output list. If \code{NULL} and \code{saveDir} is specified, the file name will be generated. If, in turn, \code{saveFile} is specified but \code{saveDir} equals \code{NULL}, then \code{saveFile} is ignored, and the output list will be returned.}

\item{saveDir}{a character string specifying a path for the output directory. If supplied, the output is saved as an \code{.RData} file in the directory; otherwise the output is returned as a list.}

\item{randomSeed}{seed of the random number generator for simulation reproducibility}
}
\value{
If \code{saveDir} is specified, the output list (named \code{trialObj}) is saved as an \code{.RData} file; otherwise it is returned. The output object is a list with the following components:
\itemize{
\item \code{trialData}: a list with \code{nTrials} components each of which is a \code{data.frame} with the variables \code{arm}, \code{entry}, \code{exit}, \code{event}, and \code{dropout} storing the treatment assignments, enrollment times, study exit times, event indicators, and dropout indicators, respectively. The observed follow-up times can be recovered as \code{exit} - \code{entry}. Indicators of belonging to the per-protocol cohort (named \code{pp}) are included if \code{missVaccProb} is specified.
\item \code{nTrials}: the number of simulated trials
\item \code{N}: the total number of enrolled trial participants
\item \code{rates}: a list with three components:
\itemize{
\item \code{enrollRate}: the treatment arm-pooled \emph{weekly} enrollment rate
\item \code{dropRate}: the average \emph{annual} treatment arm-pooled dropout rate in \code{interimData}
\item \code{eventPostRate}: a list with \code{length(trtNames)} components (labeled by the levels of the \code{arm} variable in \code{interimData}) each of which is a numeric vector of length \code{nTrials} of the sampled treatment arm-specific posterior \emph{annual} event rates
}
\item \code{randomSeed}: seed of the random number generator for simulation reproducibility
}
}
\description{
Considers data collected through an interim timepoint and generates independent time-to-event data-sets, by treatment arm, to assess the distribution of the number of treatment arm-specific endpoints
at the end of the follow-up period. A Bayesian model for treatment arm-specific endpoint rates is used for generating future data (see the vignette).
}
\examples{
arm <- rep(c("C3","T1","T2"), each=250)
schedule <- rbinom(length(arm), 1, 0.01)
entry <- rpois(length(arm), lambda=60)
entry <- entry - min(entry)
last_visit_dt <- entry + runif(length(arm), min=0, max=80)
event <- rbinom(length(arm), 1, 0.01)
dropout <- rbinom(length(arm), 1, 0.02)
dropout[event==1] <- 0
exit <- rep(NA, length(arm))
exit[event==1] <- last_visit_dt[event==1] + 5
exit[dropout==1] <- last_visit_dt[dropout==1] + 5
followup <- ifelse(event==1 | dropout==1, 0, 1)
interimData <- data.frame(arm=arm, schedule2=schedule, entry=entry, exit=exit,
last_visit_dt=last_visit_dt, event=event, dropout=dropout, complete=0, followup=followup)

completeData <- completeTrial.byArm(interimData=interimData, nTrials=5,
trtNames=c("C3","T1","T2"), N=c(500,500,500), enrollRatePeriod=24, eventPriorWeight=0.5,
eventPriorRate=c(0.001,0.0004,0.0004), fuTime=80, visitSchedule=seq(0, 80, by=4),
visitSchedule2=c(0,seq(from=8,to=80,by=12)), randomSeed=9)
### alternatively, to save the .RData output file (no '<-' needed):
completeTrial.byArm(interimData=interimData, nTrials=5, trtNames=c("C3","T1","T2"),
N=c(500,500,500), enrollRatePeriod=24, eventPriorWeight=0.5,
eventPriorRate=c(0.001,0.0004,0.0004), fuTime=80, visitSchedule=seq(0, 80, by=4),
visitSchedule2=c(0,seq(from=8,to=80,by=12)), saveDir="./", randomSeed=9)

}
\seealso{
\code{\link{completeTrial.pooledArms}}
}
