% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fdrTbl.R
\name{fdrTbl}
\alias{fdrTbl}
\title{FDR Estimate and Confidence Interval Sequence Table}
\usage{
fdrTbl(
  obs.vec,
  perm.list,
  pname,
  ntests,
  lowerbound,
  upperbound,
  incr = 0.1,
  cl = 0.95,
  c1 = NA,
  correct = "none"
)
}
\arguments{
\item{obs.vec}{observed vector of p-values.}

\item{perm.list}{list of dataframes that include a column of permutation
p-values (or statistics) in each. The length of the list permp = number of
permutations.}

\item{pname}{name of column in each list component dataframe that includes
p-values (or statistics).}

\item{ntests}{total number of observed tests, which is usually the same as
the length of obs.vec and the number of rows in each perm.list dataframe.
However, this may not be the case if results were filtered by a p-value
threshold or statistic threshold. If filtering was conducted then lowerbound
must be greater (more extreme) than the filtering criterion.}

\item{lowerbound}{lowerbound refers to the range of -log10(p-value) over
which fdr is computed for a sequence of thresholds}

\item{upperbound}{upperbound refers to the range of -log10(p-value) over
which fdr is computed for a sequence of thresholds}

\item{incr}{value by which to increment the sequence from lowerbound to
upperbound on a -log10(p-value) scale. Default is 0.1.}

\item{cl}{confidence level (default is .95).}

\item{c1}{overdispersion parameter to account for dependencies among tests. If 
all tests are known to be independent, then this parameter should be set to 1.}

\item{correct}{{"none", "BH"}, should confidence intervals be corrected for 
multiplicity using a modification of the Benjamini and Yekutieli (2005) approach 
for selecting and correcting intervals? (default is "none")}
}
\value{
A dataframe is returned where rows correspond to p-value thresholds
in the sequence from lowerbound to upperbound and columns are:
c("threshold","fdr","ll","ul","pi0","odp","S","Sp") \item{threshold
}{p-value threshold chosen to define positive tests} \item{fdr }{estimated
FDR at the chosen p-value threshold} \item{ll }{estimated lower 95\%
confidence bound for the FDR estimate} \item{ul }{estimated upper 95\%
confidence bound for the FDR estimate} \item{pi0 }{estimated percent of true
null hypotheses} \item{odp }{estimated over-dispersion parameter} \item{S
}{observed number of positive tests} \item{Sp }{total number of positive
tests summed across all permuted result sets}
}
\description{
Computes FDR estimates and confidence intervals for a sequence of potential
significance thresholds.
}
\details{
fdrTbl calls fdr_od for a series of discovery thresholds. 
Output from fdrTbl() can be used for FDRplot() input.

If correct = "BH", then confidence intervals will be corrected according to
the thresholds specified by lowerbound, upperbound, and incr. Thresholds will 
be selected if FDR is determined to be significantly different than 1. First
a Z-score test is conducted using the Millstein & Volfson standard error estimate.
Then BH FDR is computed according to the Benjamini and Yekutieli (2005) approach.
CIs for selected thresholds will be adjusted to account for multiple CI estimation.
For thresholds that are not selected, NA values are returned.
}
\examples{


n.row=100
n.col=100
X = as.data.frame(matrix(rnorm(n.row*n.col),nrow=n.row,ncol=n.col))
e = as.data.frame(matrix(rnorm(n.row*n.col),nrow=n.row,ncol=n.col))
Y = .1*X + e
nperm = 10

myanalysis = function(X,Y){
	ntests = ncol(X)
	rslts = as.data.frame(matrix(NA,nrow=ntests,ncol=2))
	names(rslts) = c("ID","pvalue")
	rslts[,"ID"] = 1:ntests
	for(i in 1:ntests){
		fit = cor.test(X[,i],Y[,i],na.action="na.exclude",
			alternative="two.sided",method="pearson")
		rslts[i,"pvalue"] = fit$p.value
	}
	return(rslts)
} # End myanalysis

## Generate observed results
obs = myanalysis(X,Y)

## Generate permuted results
perml = vector('list',nperm)
for(perm in 1:nperm){
	X1 = X[order(runif(n.col)),]
	perml[[perm]] = myanalysis(X1,Y)
}

## FDR results table
fdrTbl(obs$pvalue,perml,"pvalue",n.col,1,2)
fdrTbl(obs$pvalue,perml,"pvalue",n.col,1,2,correct="BH")

}
\references{
Millstein J, Volfson D. 2013. Computationally efficient
permutation-based confidence interval estimation for tail-area FDR.
Frontiers in Genetics | Statistical Genetics and Methodology 4(179):1-11.

Benjamini, Yoav, and Daniel Yekutieli. "False discovery rate adjusted multiple 
confidence intervals for selected parameters." Journal of the American Statistical 
Association 100.469 (2005): 71-81.
}
\author{
Joshua Millstein
}
\keyword{htest}
\keyword{nonparametric}
