# ehelp-defns.R
#  -- M.Ponce

#############################################################################################
#  eHelp: palette and output format definitions
#############################################################################################

# define color palettes for customizing the output
ehelp.palette <- function(palette="color") {
#' This function defines the "palettes" of colours to be used by eHelp()
#' There are two options right now: "color" and "bw"; default is set to "color"
#' @param palette  allows to select which color palette to use
#' @keywords internal

	# define special codes associated to the keywords
	codes <-  list("<<code_eHelp_@fnName>>","<<code_eHelp_@param>>","<<code_eHelp_@usage>>",
                        "<<code_eHelp_@author>>","<<code_eHelp_@email>>","<<code_eHelp_@repo>>","<<code_eHelp_@ref>>",
                        "<<code_eHelp_ALERT>>","<<code_eHelp_BLD>>","<<code_eHelp_UNDRLN>>")


	# color table with colors
        color1 <- list(list(crayon::blue$bgCyan$bold,crayon::reset), list(crayon::blue$italic,crayon::reset), list(crayon::bgWhite,crayon::reset),
                        list(crayon::bold,crayon::reset), list(crayon::bold,crayon::reset), list(crayon::bold,crayon::reset), list(crayon::underline,crayon::reset), list(crayon::underline,crayon::reset),
                        list(crayon::red,crayon::reset), list(crayon::bold,crayon::reset), list(crayon::underline,crayon::reset) )
	# color table with mostly BW
        color2 <- list(list(crayon::inverse$bold,crayon::reset), list(crayon::blue$italic,crayon::reset), list(crayon::inverse,crayon::reset), list(crayon::red,crayon::reset),
                        list(crayon::bold,crayon::reset), list(crayon::bold,crayon::reset), list(crayon::bold,crayon::reset), list(crayon::underline,crayon::reset), list(crayon::underline,crayon::reset),
                        list(crayon::bold,crayon::reset), list(crayon::bold,crayon::reset), list(crayon::underline,crayon::reset) )

	# codes for HTML output
	html.codes <- list(list("<h1>","</h1>"),list("<i>","</i>"),list("<h2>","</h2>"),
			list("<h3>","</h3>"),list("<b>","</b>"),list("<u>","</u>"),list("<b>","</b>"),
			list("<h4>","</h4>"),list("<b>","</b>"),list("<u>","</u>"))

	# codes for LaTeX output
	latex.codes <- list(list("\\section*{","}\n"),list("\\begin{itemize} \\item ","\\end{itemize}"),list("\\textcolor{blue}{\\tt ","}\n"),
                        list("\\textbf{","}"),list("\\emph{","}\n"),list("\\textbf{","}\n"),list("\\subsection{","}"),
                        list("\\emph{","}"),list("\\textbf{","}"),list("\\underline{","}"))

	# codes for markdown
	markdown.codes <- list(list("#",""), list("*","*"), list("```","```"),
				list("##",""), list("###",""), list("**","**"),list("*","*"),
				list("**","**"), list("**","**"), list("**","**"))

	# combine all of them...
        ecodes <- list(codes=codes, color=color1,bw=color2, html=html.codes, latex=latex.codes, markdown=markdown.codes)
        return(ecodes)
}


#######################


format.defns <- function(fmt,filename="",ending=NA){
#' Function to define specifics structures, symbols and 'codes'
#' @param  fmt   an specific file format; possible values are: txt, html, latex, markdown
#' @param  ending  optional argument for selecting when continuing saving into the file
#' @keywords internal

	# TXT or ASCII
	if ( (fmt == "txt") || (fmt == "ascii")) {
		ending <- ""
		sp.chars <- list()
		lines <- "----------------------------------------- \n"
		struct <- ""
		pre <- ""
		post <- ""
		eol <- ""
		listings <- list(begining="",ending="",eol='\n')
	# HTML
	} else if (fmt == "html") {
		ending <- "</P> \n </BODY> \n </HTML>"
		sp.chars <- list(list("\t","&emsp;&emsp;"),list("\n",""))
		lines <- "<hr> \n"
		struct <- c(
			paste0(	"<HEAD> \n",
				"<TITLE>Documentation for  <<<",gsub("-eHelp.html","",filename,ignore.case=T),">>> </TITLE> \n",
				"</HEAD> \n",
				"\n <!-- document generated by R's eHelp package -->",'\n\n',
				"<BODY> \n",
				"<P>"
			) )
		pre <- "<div align=\"right\"><small>"
		post <- "</small></div> <hr>"
		eol <- "<br> \n"
		listings <- list(begining="<p> <code>",ending="</code></p> \n </body> \n </HTML>",eol=eol)
	# LATEX
	} else if (fmt == "latex") {
		if (is.na(ending)) ending <- "\\end{document}"
		sp.chars <- list(list("\t","  "),list("#",""),list("%","\\%"))
		struct <-  c(
			paste0(  "% document generated by R's eHelp package",'\n\n',
                        	"\\documentclass{article}",'\n\n\n',
                        	"\\usepackage{xcolor}",'\n\n\n',
                        	"\\title{Documentation for  \\textbf{",gsub("-eHelp.tex","",filename,ignore.case=T),"} }",'\n\n\n',
                        	"\\begin{document}",'\n',
                        	'\n',"\\maketitle",'\n\n'
                        ) )
		lines <- ""
		#lines <- "\\hrule"
		pre <- "\\flushright \n \\fbox{ \\begin{minipage}{0.85\\textwidth} \n \\flushright  \\small "
		post <- paste(ending=paste("\\end{minipage} } \n \\flushleft \n\n",ending))
		eol <- "\n"
		listings <- list(begining="\\section*{Listing}\n\\label{listing} \n \\begin{minipage}{0.75\\textwidth}\n \\small \n \\begin{verbatim}" ,
				ending="\\end{verbatim} \n \\end{minipage} \n\n \\end{document}",
				eol='\n')
	# MARKDOWN
	} else if (fmt == "markdown") {
                ending <- ""
                sp.chars <- list()
                lines <- "--- \n"
                struct <- c(
			paste0(	"---", '\n',
				"title: \"Documentation for",gsub("-eHelp.md","",filename,ignore.case=T),"\"",'\n',
				"author: \"eHelp Package\" \n",
				"date: ",format(Sys.time(), "%a %b %d %X %Y"),'\n',
				"---", '\n'
			) )
                pre <- ""
                post <- ""
                eol <- ""
		listings <- list(begining="```",ending="```", eol="\n")
	# something else will default to 'nill'...
	} else {
		ending <- ""
		sp.chars <- list()
		lines <- ""
		struct <- ""
		pre <- ""
		post <- ""
		eol <- ""
		listings <- list(begining="",ending="",eol='')
	}

	return(list(fmt=fmt, ending=ending, sp.chars=sp.chars, lines=lines,
			struct=struct, pre=pre, post=post, eol=eol, lst=listings))
}



#################################################################################################


