if(interactive()){
  library(dynamichazard); library(testthat); library(survival)

  if(grepl("testthat$", getwd()))
    source("../../R/test_utils.R") else
      source("./R/test_utils.R")

  exp_model_names <- with(environment(ddhazard), exp_model_names)
}


# Had issues with win builder. Thus, these lines
test_name <- "UKF"
cat("\nRunning", test_name, "\n")


test_that("UKF on head_neck works with logit model", {
  # tmp <- file("tmp.txt")
  # sink(tmp)
  suppressMessages(result <- ddhazard(
    formula = survival::Surv(start, stop, event) ~ group,
    data = head_neck_cancer,
    by = 1, Q_0 = diag(1, 2), a_0 = c(-3, 0),
    Q = diag(1e-1, 2),
    control = list(est_Q_0 = F, n_max = 10^4, eps = 10^-3,
                   method = "UKF", save_data = F, save_risk_set = F,
                   beta = 0),
    max_T = 30,
    id = head_neck_cancer$id, order = 1,
    verbose = F,
    model = "logit"
  ))
  # sink()
  # close(tmp)
  # matplot(result$state_vecs, type = "l")
  # get_expect_equal(result, file = "tmp.txt")

  expect_equal(c(result$state_vecs),
               c(-3.6210434470696890585, -3.6219775909341636044, -3.1965255364948381711, -2.8822948351329835681, -2.6776156900575127651, -2.2095602359803558912, -2.3039025570373250495, -2.7036359117999517743, -2.8953691047058138786, -3.0279231524362413275, -3.1628588573884406365, -3.3904425619815969917, -3.5138861403774774850, -3.5567556811008818052, -3.3898744925660686533, -3.4585646333131219166, -3.5946313394922535522, -3.6965037242615896851, -3.6474672416224787064, -3.7300581555136917977, -3.6881032589664783927, -3.8180662106301541669, -3.9901818199601550852, -4.0988787019364556485, -4.1448073556898235026, -4.2615650315205364507, -4.3290075039660331768, -4.3505261677194129177, -4.4572955274518104574, -4.5274560988752776680, -4.5621022108524913463, 0.0437524209268248740, 0.0434488530566888853, 0.2434369545035404980, 0.3969322710753109273,  0.4240346032522276376, 0.6773390287582086167, 0.6451839996937245303, 0.4537886436474507512, 0.4583136798526994404,  0.4345211638159840462, 0.3960674102490738813, 0.2955817385491453231, 0.2705570910979426813, 0.3096639084387750596,  0.4470865166245729938, 0.4000008635825747505, 0.3212092861022941559, 0.2816445842895028973, 0.3106255364430759447,  0.2768430914252567598, 0.2992739675623217188, 0.1908583206996150583, 0.0829235578956353248, 0.0007578594815966233, -0.0555913462040324990, -0.1227738310103664021, -0.1696660035920780352, -0.1974353476942747043, -0.2421429545513467396, -0.2713445719050950333, -0.2856785819140564842 ))

  expect_equal(c(result$state_vars),
               c( 0.27586130139258525, -0.01729696648038014, -0.01729696648038018, 0.25673566962837213, 0.19204978899311037, -0.04652972542539296, -0.04652972542539302, 0.21250496603901070, 0.17969437455520307, -0.04149160607874319, -0.04149160607874328, 0.19263335842713530,  0.15127185438701288, -0.04590055557720557, -0.04590055557720563, 0.16646399084541957, 0.12408621590335175, -0.05264837752236441, -0.05264837752236444, 0.14021440808669886, 0.11181404217186226, -0.04597460917218005, -0.04597460917218010, 0.12930818038659231,  0.08969222661530118, -0.04965701541242944, -0.04965701541242947, 0.11483060025883877, 0.09296408991344228, -0.04606189072130398, -0.04606189072130403, 0.11692307001859213, 0.10838201464451167, -0.03966349773915480, -0.03966349773915485, 0.12981193907805577,  0.12097949468442352, -0.03851290694902363, -0.03851290694902369, 0.14206446783109883, 0.13211592509083336, -0.03874511949845088, -0.03874511949845093, 0.15539114693965705, 0.14384890041396725, -0.03855383544648393, -0.03855383544648397, 0.17079679789240104,  0.16264830505235467, -0.03457106391527271, -0.03457106391527275, 0.19045525247100575, 0.17912420788215988, -0.03264849154946278, -0.03264849154946281, 0.20869824116994468, 0.19170844991569747, -0.03405708296339675, -0.03405708296339679, 0.22394393261604215,  0.18681363202722745, -0.04542000114778499, -0.04542000114778502, 0.23139642142144787, 0.19283005720931323, -0.05054236761614715, -0.05054236761614718, 0.24501581354478283, 0.20818936050880404, -0.05078982491866487, -0.05078982491866491, 0.26471653072460333,  0.22639191934206793, -0.05080994943479061, -0.05080994943479065, 0.28653105710720650, 0.23654094916899326, -0.05606446938229647, -0.05606446938229650, 0.30597282151091842, 0.25064078106728011, -0.06050423110661661, -0.06050423110661664, 0.32802302101753855,  0.25575159167104355, -0.07080863826708850, -0.07080863826708853, 0.34775129645080316, 0.27284865609263720, -0.07508608685919037, -0.07508608685919041, 0.37559466447733608, 0.29861404981717721, -0.07541670702878020, -0.07541670702878023, 0.41013515123571959,  0.32594704948607256, -0.07566214415849024, -0.07566214415849026, 0.44792698017987587, 0.35200387652970888, -0.07740493704447665, -0.07740493704447668, 0.48739743860021112, 0.38891852074022082, -0.07592996632165271, -0.07592996632165273, 0.53185859068309049,  0.43234651088258613, -0.07262126182518183, -0.07262126182518186, 0.57999557271232094, 0.48292910117995069, -0.06748916445709460, -0.06748916445709463, 0.63130448681350515, 0.56054019723716186, -0.05250657840787097, -0.05250657840787101, 0.69090884779240913,  0.66854326851757739, -0.02655958573405968, -0.02655958573405972, 0.75856729979961257 ))

  expect_equal(c(result$lag_one_cov),
               c(NULL ))

  expect_equal(c(result$fixed_effects),
               c(numeric(0) ))

  expect_equal(c(result$n_iter),
               c(14 ))

  expect_equal(c(result$Q),
               c(0.1479302025208791, 0.0414058984636589, 0.0414058984636589, 0.0752060922218248 ))

  expect_equal(c(result$Q_0),
               c(1, 0, 0, 1 ))

  expect_equal(c(result$n_risk),
               c(96, 95, 91, 84, 78, 65, 53, 49, 46, 43, 39, 37, 36, 36, 33, 31, 30, 30, 25, 25, 23, 21, 21, 21, 20, 19, 19, 18, 18, 18 ))

  expect_equal(c(result$times),
               c( 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30 ))

  expect_equal(c(result$risk_set),
               c(NULL ))

  expect_equal(c(result$data),
               c(NULL ))

  expect_equal(c(result$order),
               c(1 ))

  expect_equal(c(result$F_),
               c(1, 0, 0, 1 ))

  expect_equal(c(result$method),
               c("UKF" ))

  expect_equal(c(result$model),
               c("logit" ))

  expect_equal(c(result$est_Q_0),
               c(FALSE ))

  expect_equal(c(result$LR),
               c(1 ))
})


set.seed(2972)
sims <- test_sim_func_logit(n_series = 5e2, n_vars = 3, t_0 = 0, t_max = 10,
                            x_range = 1, x_mean = -.5, re_draw = T, beta_start = 1,
                            intercept_start = -1, sds = c(.1, rep(1, 3)))
sum(sims$res$event)

test_that("UKF does not fail and both methods give the same",{
  res_new <- ddhazard(formula = survival::Surv(tstart, tstop, event) ~ . - tstart - tstop - event - id,
                  by = 1,
                  data = sims$res,
                  a_0 = rep(0, ncol(sims$res) + 1 - 4),
                  Q_0 = diag(rep(1, ncol(sims$res) + 1 - 4)),
                  verbose = F,
                  control = list(eps = 1e-2, est_Q_0 = F, method = "UKF",
                                 kappa = 0, alpha = 1, beta = 0),
                  id = sims$res$id,
                  max_T = 10)


  res_old <- ddhazard(formula = survival::Surv(tstart, tstop, event) ~ . - tstart - tstop - event - id,
                      by = 1,
                      data = sims$res,
                      a_0 = rep(0, ncol(sims$res) + 1 - 4),
                      Q_0 = diag(rep(1, ncol(sims$res) + 1 - 4)),
                      control = list(est_Q_0 = F, kappa = 0, alpha = 1, beta = 0, eps = 1e-2, method = "UKF_org"),
                      id = sims$res$id,
                      max_T = 10)

  expect_equal(res_new$state_vecs, res_old$state_vecs)
  expect_equal(res_new$state_vars, res_old$state_vars)
})

test_that("Chaning time scale in UKF does no change results when other parems are changed accoridngly",{
  arg_list <- list(formula = survival::Surv(tstart, tstop, event) ~ . - tstart - tstop - event - id,
                   by = 1,
                   data = sims$res,
                   a_0 = rep(0, ncol(sims$res) + 1 - 4),
                   Q_0 = diag(rep(1, ncol(sims$res) + 1 - 4)),
                   Q = diag(rep(1e-2, ncol(sims$res) + 1 - 4)),
                   control = list(kappa = 0, alpha = 1, beta = 0,
                                  est_Q_0 = F, method = "UKF", eps = 1e-2),
                   id = sims$res$id,
                   verbose = F,
                   max_T = 10)

  res <- do.call(ddhazard, arg_list)

  sim_tmp <- sims
  t_mult <- 2.5
  sim_tmp$res$tstart <- sim_tmp$res$tstart * t_mult
  sim_tmp$res$tstop <- sim_tmp$res$tstop * t_mult
  arg_list$data <- sim_tmp$res
  arg_list$by <- t_mult
  arg_list$max_T <- arg_list$max_T * t_mult
  arg_list$Q <- arg_list$Q / t_mult

  res_new_time <- do.call(ddhazard, arg_list)

  expect_equal(res$state_vecs, res_new_time$state_vecs)
  expect_equal(res$state_vars, res_new_time$state_vars)
  expect_equal(res$Q, res_new_time$Q * t_mult)
})

test_that("Testing UKF against prev computed values",{
  arg_list <- list(formula = survival::Surv(tstart, tstop, event) ~ . - tstart - tstop - event - id,
                   by = 1,
                   data = sims$res,
                   a_0 = rep(0, ncol(sims$res) + 1 - 4),
                   Q_0 = diag(rep(1, ncol(sims$res) + 1 - 4)),
                   Q = diag(rep(1e-2, ncol(sims$res) + 1 - 4)),
                   control = list(kappa = 0, alpha = 1, beta = 0,
                                  est_Q_0 = F, method = "UKF", eps = 1e-2,
                                  save_data = F, save_risk_set = F),
                   id = sims$res$id,
                   verbose = F,
                   max_T = 10)

  res <- do.call(ddhazard, arg_list)

  # matplot(sims$betas, type = "l")
  # matplot(res$state_vecs, add = T, type = "l")
  # get_expect_equal(res, file = "tmp.txt")

  expect_equal(c(res$state_vecs),
               c(-1.586377228122479943, -1.586630033484746516, -1.428376221113095479, -0.784370178711367605, -0.641999351480161407, -0.518433459927804763, -0.727906112606297895, -0.559484221494314204, -0.731705419333232077, -0.248707136761067260, -0.367935180117968219, 2.198771692817839352, 2.175231794247460382, 2.045516835180890780, 1.933855355358886641,  1.603907220226844288, 1.196301311713791060, 1.818038819570645392, 3.400829273222566673, 3.299842637531596790,  2.550263998669416221, 2.388201262064851615, 0.920991063786816277, 0.886046736214297526, 0.953588306031064148,  1.175089262870585793, 0.791708045469890154, 0.593000224938449838, 1.215726149479578133, 2.375444389485041441,  1.464487580941969957, -0.335690612684172829, -0.806051122637007067, 1.425111865038881431, 1.480868651999746044,  1.253247745933509183, 0.066811905544865408, 0.497641457837281531, 0.801002033089991405, -0.008536873968637337, -2.724247674919297868, -1.405637265305051731, 0.315610362649203113, 1.135735482814673514 ))

  expect_equal(c(res$state_vars),
               c( 0.1824210226547152702, -0.0280567323423201904, -0.0510051459995278234, -0.0495654523271920333, -0.0280567323423278475,  0.4305693732570275989, -0.0013574119131351814, -0.1907928477608626938, -0.0510051459995305434, -0.0013574119131365258,  0.4189410607259385078, -0.2424785841118443086, -0.0495654523271992775, -0.1907928477608632212, -0.2424785841118429763,  0.6024630917946190989, 0.1013657754692213198, 0.0156306337058410849, 0.0073509843173614753, -0.0268884278373829472,  0.0156306337058329664, 0.4064563825696114185, 0.0685159081505088974, -0.3226725501523874939, 0.0073509843173616973,  0.0685159081505069267, 0.4141679079403943753, -0.3907653399689948737, -0.0268884278373968250, -0.3226725501523870498, -0.3907653399689898222, 0.8257138664824452690, 0.0663798238546949304, 0.0262599432837489349, 0.0246249497727718952, -0.0076471942063543047, 0.0262599432837406221, 0.3507836336884601613, 0.0690548826900880819, -0.2996261151965867175,  0.0246249497727720479, 0.0690548826900864443, 0.3610801623583721476, -0.3651998385274937697, -0.0076471942063682935, -0.2996261151965861069, -0.3651998385274876635, 0.7461269585768464108, 0.0592165548797238539, 0.0220032502799559909,  0.0216019785463589797, 0.0028276012260004812, 0.0220032502799478030, 0.2790894576969081275, 0.0599830146087604044, -0.2321988193208725793, 0.0216019785463590352, 0.0599830146087596966, 0.3062152034547571677, -0.3036780373732752558,  0.0028276012259867561, -0.2321988193208721074, -0.3036780373732677618, 0.6153341528879432420, 0.0464677116509729787,  0.0149739483771787751, 0.0090077371435139594, 0.0396265514245370959, 0.0149739483771707330, 0.1510173226968049565,  0.0044512076679463844, -0.0772574714336502488, 0.0090077371435133557, 0.0044512076679491721, 0.1649460466541940729, -0.1094233633948897277, 0.0396265514245242034, -0.0772574714336518586, -0.1094233633948788614, 0.2477618406607089707,  0.0457933278824830189, 0.0189979885556992474, 0.0181719065278271100, 0.0271404629246799478, 0.0189979885556913405,  0.1504277543321448363, 0.0100330729069115221, -0.0853888505698948558, 0.0181719065278262183, 0.0100330729069145318,  0.1581817287805460381, -0.1140592962951975375, 0.0271404629246679817, -0.0853888505698969930, -0.1140592962951878786,  0.2621288086480622059, 0.0461191460035962758, 0.0238808575706914543, 0.0195345526366445185, 0.0221014143085921586,  0.0238808575706839291, 0.1611883120170764239, 0.0170997395987705567, -0.1024694511914471529, 0.0195345526366442548,  0.0170997395987733392, 0.1582661013906956349, -0.1154217804871662256, 0.0221014143085789608, -0.1024694511914489292, -0.1154217804871567055, 0.2747418086179178598, 0.0505433907259783405, 0.0208193912344890042, 0.0124159188930033776,  0.0268482653090398246, 0.0208193912344817877, 0.1958486607521109923, 0.0334458343199537586, -0.1318413432886212655,  0.0124159188930035910, 0.0334458343199564301, 0.2120134817275364236, -0.1613327444182467707, 0.0268482653090259954, -0.1318413432886225978, -0.1613327444182368897, 0.3281275258530819139, 0.0552289918339923941, 0.0124368321962026462,  0.0005551672977831268, 0.0462487962034608333, 0.0124368321961957559, 0.2499891923201728594, 0.0646110960888437169, -0.1816810466370180344, 0.0005551672977833272, 0.0646110960888461872, 0.2401066921843042490, -0.1850146587074111282,  0.0462487962034465044, -0.1816810466370191446, -0.1850146587074022742, 0.3468552081735357673, 0.0604163269531132405,  0.0067712707394571442, -0.0015307105494844556, 0.0489851583859482326, 0.0067712707394502287, 0.2990597506576424536,  0.0634040541957961862, -0.1983717136626504951, -0.0015307105494842595, 0.0634040541957986842, 0.2699057656205353095, -0.2079062532167276489, 0.0489851583859339454, -0.1983717136626514388, -0.2079062532167189892, 0.3841578936288893575,  0.0602019128638456857, 0.0305336369372033664, 0.0309621279864029514, 0.0110088952405827216, 0.0305336369371969757,  0.3205268793431931273, 0.0292230528586805427, -0.2043880476042079231, 0.0309621279864037285, 0.0292230528586820970,  0.2681554241099355362, -0.2349343460011865670, 0.0110088952405680424, -0.2043880476042079231, -0.2349343460011785734,  0.4784499873473575526 ))

  expect_equal(c(res$lag_one_cov),
               c(NULL ))

  expect_equal(c(res$fixed_effects),
               c(numeric(0) ))

  expect_equal(c(res$n_iter),
               c(32 ))

  expect_equal(c(res$Q),
               c( 0.12160536293782964, -0.05097094130727473, -0.07849865845916659, -0.06184413844291841, -0.05097094130727473,  0.66720696425256443, 0.50251010237681148, -0.95634626918935339, -0.07849865845916659, 0.50251010237681148,  0.95470908871885174, -1.27004602490409324, -0.06184413844291841, -0.95634626918935339, -1.27004602490409324,  2.19193807022204634 ))

  expect_equal(c(res$Q_0),
               c(1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1 ))

  expect_equal(c(res$n_risk),
               c(500, 476, 458, 421, 359, 295, 259, 214, 194, 157 ))

  expect_equal(c(res$times),
               c( 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10 ))

  expect_equal(c(res$risk_set),
               c(NULL ))

  expect_equal(c(res$data),
               c(NULL ))

  expect_equal(c(res$order),
               c(1 ))

  expect_equal(c(res$F_),
               c(1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1 ))

  expect_equal(c(res$method),
               c("UKF" ))

  expect_equal(c(res$model),
               c("logit" ))

  expect_equal(c(res$est_Q_0),
               c(FALSE ))

  expect_equal(c(res$LR),
               c(1 ))
})

test_that("Altering UKF alpha, beta and kappa change the results",{
  arg_list <- list(formula = survival::Surv(tstart, tstop, event) ~ . - tstart - tstop - event - id,
                   by = 1,
                   data = sims$res,
                   Q_0 = diag(rep(1, ncol(sims$res) + 1 - 4)),
                   Q = diag(rep(1e-1, ncol(sims$res) + 1 - 4)),
                   control = list(kappa = 0, alpha = 1, beta = 0,
                                  est_Q_0 = F, method = "UKF", eps = 1e-2),
                   id = sims$res$id,
                   verbose = F,
                   max_T = 10)
  suppressMessages(m1 <- do.call(ddhazard, arg_list))

  arg_list$control$beta <- 2

  suppressMessages(m2 <- do.call(ddhazard, arg_list))

  expect_true(class(all.equal(m1$state_vecs, m2$state_vecs)) == "character")
  expect_true(class(all.equal(m1$state_vars, m2$state_vars)) == "character")

  arg_list$control$alpha <- .1
  arg_list$control$kappa <- 4 / (.1)^2 + 4

  suppressMessages(m3 <- do.call(ddhazard, arg_list))

  expect_true(class(all.equal(m2$state_vecs, m3$state_vecs)) == "character")
  expect_true(class(all.equal(m2$state_vars, m3$state_vars)) == "character")

  arg_list$control$beta <- 0
  arg_list$control$alpha <- 1
  arg_list$control$kappa <- .5

  suppressMessages(m4 <- do.call(ddhazard, arg_list))

  expect_true(class(all.equal(m3$state_vecs, m4$state_vecs)) == "character")
  expect_true(class(all.equal(m3$state_vars, m4$state_vars)) == "character")
})

arg_list <- list(formula = survival::Surv(tstart, tstop, event) ~ . - tstart - tstop - event - id,
                 by = 1,
                 data = NULL,
                 Q_0 = diag(rep(1, ncol(sims$res) + 1 - 4)),
                 Q = diag(rep(1e-1, ncol(sims$res) + 1 - 4)),
                 control = list(kappa = 0, alpha = 1, beta = 0,
                                est_Q_0 = F, method = "UKF", eps = 1e-2),
                 id = NULL,
                 verbose = F,
                 max_T = 10)

set.seed(2972)
sims <- test_sim_func_logit(n_series = 2e3, n_vars = 3, t_0 = 0, t_max = 10,
                            x_range = 1, x_mean = -.5, re_draw = T, beta_start = 1,
                            intercept_start = -2, sds = c(.1, rep(1, 3)))
sum(sims$res$event)

arg_list$data <- sims$res
arg_list$id <- sims$res$id

arg_list$control$beta <- 0
suppressMessages(fit1 <- do.call(ddhazard, arg_list))

arg_list$control$beta = 2
suppressMessages(fit2 <- do.call(ddhazard, arg_list))

# matplot(sims$betas, type = "l", lty = 1, ylim = range(sims$betas, fit1$state_vecs, fit2$state_vecs))
# matplot(fit1$state_vecs, type = "l", lty = 2, add = T)
# matplot(fit2$state_vecs, type = "l", lty = 4, add = T)

test_that("UKF on head_neck works with exponential model", {
  # Change by argument
  # tmp <- file("tmp.txt")
  # sink(tmp)
  suppressMessages(result_exp <- ddhazard(
    formula = survival::Surv(start, stop, event) ~ group,
    data = head_neck_cancer,
    by = 1, Q_0 = diag(1, 2), a_0 = c(-3, 0),
    Q = diag(1e-1, 2),
    control = list(est_Q_0 = F, n_max = 10^4, eps = 10^-3, beta = 0,
                   method = "UKF", save_data = F, save_risk_set = F),
    max_T = 30,
    id = head_neck_cancer$id, order = 1,
    verbose = F,
    model = "exp_combined"
  ))
  # sink()
  # close(tmp)
  # matplot(result_exp$state_vecs, type = "l")
  # get_expect_equal(result_exp, file = "tmp.txt")

  expect_equal(c(result_exp$state_vecs),
               c(-3.62169912365052138, -3.62246078452342424, -3.22364359257399835, -2.92217977205102253, -2.72907999003154789, -2.31190644378908861, -2.42049669642081922, -2.78394368630350941, -2.96084539618035114, -3.08533315953903431, -3.21320088682824023, -3.42328511524642787, -3.53914011221686220, -3.58347883705927117, -3.44131620120169446, -3.50485901032138036, -3.62983767601771934, -3.72502283544137036, -3.68571947451093873, -3.76393606572367734, -3.73198475887238157, -3.85137633806533186, -4.00968934620624928, -4.11071598748750056, -4.15544703206457289, -4.26352440239257380, -4.32694584038677466, -4.34893991704142202, -4.44666450998967910, -4.51086590148823863, -4.54257172268638154, 0.03512153461984305, 0.03499222987309403, 0.21904752732582489, 0.36349610863524906, 0.37975430073260097,  0.60372085835324862, 0.56248876424764083, 0.39490258987221666, 0.41325691765382555, 0.39761739711562594, 0.36622107890507077,  0.27804691138392745, 0.26016928296526448, 0.30176607029358010, 0.42852938902956261, 0.38505456407735117, 0.31329924797493830,  0.27896343070447388, 0.30391561557526409, 0.27386171379931751, 0.29203485932130385, 0.18924291566550427, 0.08933460325358744,  0.01157613939984603, -0.04414995777165318, -0.10612942803044154, -0.15059962381984210, -0.17859069537436190, -0.21782376721781033, -0.24342466059260034, -0.25598554356591635 ))

  expect_equal(c(result_exp$state_vars),
               c( 0.26179350713741312, -0.02518477240080714, -0.02518477240080657, 0.25041835838325355, 0.18393691970255002, -0.05025960835094892, -0.05025960835094823, 0.20774054207588433, 0.17137008347534013, -0.04496823747370379, -0.04496823747370315, 0.18771414842964934,  0.14516715664778784, -0.04769253323669073, -0.04769253323669005, 0.16194072493710138, 0.11889796047033377, -0.05342280266159982, -0.05342280266159918, 0.13530799475546648, 0.10582589814661759, -0.04658165898077433, -0.04658165898077363, 0.12344861967201867,  0.08313613429058933, -0.05034845851381021, -0.05034845851380952, 0.10738477237551117, 0.08700514565472872, -0.04631427160359920, -0.04631427160359852, 0.11025031850038432, 0.10219195715158980, -0.04000705463604366, -0.04000705463604295, 0.12422177899873321,  0.11437522628233976, -0.03952595961941931, -0.03952595961941859, 0.13672962334571048, 0.12508816746533474, -0.04040307339437286, -0.04040307339437216, 0.15002705609780154, 0.13630789900930673, -0.04089970967323204, -0.04089970967323135, 0.16517439744018070,  0.15385886149528005, -0.03799173775194831, -0.03799173775194760, 0.18428623361581675, 0.16921157576514878, -0.03702988940581356, -0.03702988940581282, 0.20190492266278764, 0.18104762414551845, -0.03913483944086422, -0.03913483944086346, 0.21657512006117122,  0.17805207668510784, -0.04955804999934899, -0.04955804999934817, 0.22427712705464720, 0.18410058923231776, -0.05472902877865486, -0.05472902877865403, 0.23752059866658176, 0.19840204349199297, -0.05565523798676760, -0.05565523798676678, 0.25648493655242899,  0.21525804099994120, -0.05659656554308166, -0.05659656554308083, 0.27737253346425306, 0.22543035980169196, -0.06198157350380359, -0.06198157350380275, 0.29627505072024507, 0.23888290615416641, -0.06699283747087292, -0.06699283747087208, 0.31750358925872424,  0.24494004212485823, -0.07699086971009064, -0.07699086971008978, 0.33685884702696189, 0.26140825476372442, -0.08177119028289324, -0.08177119028289238, 0.36399049173075732, 0.28558403983407588, -0.08332215609003478, -0.08332215609003391, 0.39744323778357521,  0.31146988351280047, -0.08484249930210493, -0.08484249930210402, 0.43413380789258649, 0.33681863620703600, -0.08753868007063045, -0.08753868007062955, 0.47274077770520945, 0.37190976148156496, -0.08777888706176394, -0.08777888706176304, 0.51605618435243505,  0.41325624916857340, -0.08646971224126070, -0.08646971224125981, 0.56302663516032425, 0.46161411173626238, -0.08351269871312506, -0.08351269871312415, 0.61326724927858622, 0.53351538087394712, -0.07276018186437519, -0.07276018186437430, 0.67097219462493185,  0.63194503520661316, -0.05320906518387750, -0.05320906518387660, 0.73592908613487684 ))

  expect_equal(c(result_exp$lag_one_cov),
               c(NULL ))

  expect_equal(c(result_exp$fixed_effects),
               c(numeric(0) ))

  expect_equal(c(result_exp$n_iter),
               c(13 ))

  expect_equal(c(result_exp$Q),
               c(0.13282206634683452, 0.03202721699929914, 0.03202721699929914, 0.07120662065543586 ))

  expect_equal(c(result_exp$Q_0),
               c(1, 0, 0, 1 ))

  expect_equal(c(result_exp$n_risk),
               c(96, 95, 91, 84, 78, 65, 53, 49, 46, 43, 39, 37, 36, 36, 33, 31, 30, 30, 25, 25, 23, 21, 21, 21, 20, 19, 19, 18, 18, 18 ))

  expect_equal(c(result_exp$times),
               c( 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30 ))

  expect_equal(c(result_exp$risk_set),
               c(NULL ))

  expect_equal(c(result_exp$data),
               c(NULL ))

  expect_equal(c(result_exp$order),
               c(1 ))

  expect_equal(c(result_exp$F_),
               c(1, 0, 0, 1 ))

  expect_equal(c(result_exp$method),
               c("UKF" ))

  expect_equal(c(result_exp$model),
               c("exp_combined" ))

  expect_equal(c(result_exp$est_Q_0),
               c(FALSE ))

  expect_equal(c(result_exp$LR),
               c(1 ))
})

test_that("UKF on simulated data works with exponential model where both variables are used", {
  set.seed(9997)
  sims <- test_sim_func_exp(n_series = 1e3, n_vars = 10, t_0 = 0, t_max = 10,
                            x_range = 1, x_mean = 0, re_draw = T, beta_start = (1:10 - 5) / 2.5,
                            intercept_start = -5, sds = c(.1, rep(1, 10)))
  # tmp <- file("tmp.txt")
  # sink(tmp)
  suppressMessages(result_exp <- ddhazard(
    formula = survival::Surv(tstart, tstop, event) ~ . - id - tstart - tstop - event,
    data = sims$res,
    by = (by_ <- 1),
    Q_0 = diag(1, 11),
    Q = diag(1e-1, 11),
    control = list(est_Q_0 = F, eps = 10^-2, n_max = 10^3, method = "UKF",
                   debug = F, beta = 0, save_data = F, save_risk_set = F),
    max_T = 10,
    id = sims$res$id, order = 1,
    verbose = F,
    model = "exp_combined"))
  # sink()
  # close(tmp)

  # sum(sims$res$event)
  # diag(result_exp$Q)
  # result_exp$LR
  # matplot(sims$betas, type = "l", lty = 1)
  # matplot(result_exp$state_vecs, type = "l", lty = 2, add = T)
  # get_expect_equal(result_exp, file = "tmp.txt")

  expect_equal(c(result_exp$state_vecs),
               c(-5.296985751931666542, -5.294997720366526828, -4.987931290161729869, -4.790365169112824617, -4.719816425838490481, -4.632238719804252725, -4.788674354517598708, -4.740386195760116550, -4.613032780318246928, -4.201210052042589460, -3.803262785212128350, -2.072226320663728583, -2.056332478790297902, -2.554356756714668553, -2.332029520834581149, -1.836895545678870256, -1.802448703526633000, -1.996368704429387853, -2.262155454794579246, -2.385241061073536173, -1.821641600602017741, -1.593990021324609208, -0.201978209091266597, -0.103105335987053681, -1.407690323835784074, -0.580716240638760040, 0.994498562080960102, 0.477382287548883288, 0.592517572595406827, -0.627086589251302451, -1.945897379530036719, -0.870211508056799476, 0.731963694100878026, 0.005299053452932317, -0.049477931190040791,  0.582038060750140973, 0.286153973223754576, -0.608556434282100378, -0.214685842952759109, -0.023814536985214046,  0.558859729567239505, 1.252541499128918989, 0.465284923411283802, -0.530473641569741527, -0.215368006986644267, -0.221136530877780929, -0.521633778771008916, -0.267148540153603120, -0.156355884721407046, 0.380171675657571373,  0.640726525403425784, 0.904449362439669891, 0.991961671915229548, 1.229680129445827941, 1.331819024479016678,  0.569587491755787045, 0.580074070396911168, -0.797388641227575135, 0.457003793248005219, 0.844866285461513544,  1.942221233568787842, 2.925982593758558625, 2.883705544000623622, 2.416805095089312516, 2.122318571208535953,  2.303653227172255402, -1.210546063847233622, -1.127402057267626212, -1.138676420221215402, -2.452475696615405543, -1.478568819371093745, -3.794811689160051849, -5.079974584064555820, -6.325221676979642105, -7.138817795434787783, -6.178388872773273199, -5.322892858271043082, -0.106228603475141653, -0.195160335623876219, -1.201069324059879095,  1.007593281935528440, 1.233108998269667245, 3.653397130810078774, 3.155402596108486168, 3.536763571301665188,  4.097296930611038945, 3.211786440832852030, 1.835422847785916023, 1.374969911906207543, 1.413743903197165475,  1.566712523598329243, 1.817345504919710253, 1.962554546233724828, 2.115986386795585261, 3.211074244777490705,  3.311417917604556838, 3.155523584764688749, 3.909825978564392734, 5.184385936669507089, 2.010419730604675159,  2.014380151220583226, 4.405422284852015480, 2.606537459372088250, 1.739768264372042905, 0.174903203177358746, -0.492264806495917639, -0.226825404336185588, 0.352348400806409834, 0.847477290027751962, 1.382959172463448461,  2.363675420334485189, 2.312563813615891917, 3.167160018567376500, 3.929891885048522049, 3.613724621996472486,  4.217355055640840789, 3.198738591417742860, 3.301241479116225896, 3.714410848720285596, 3.127782296801913553,  2.590662278064039814 ))

  expect_equal(c(result_exp$state_vars),
               c( 1.442959460496754e-01, 5.160445035091613e-02, 4.211332909911270e-02, -4.193835803731979e-02, 3.903754432397725e-02, -3.385441773702751e-02, -6.215211887600101e-03, 2.845211779683128e-02, 6.907179845259179e-02, 5.744242623014516e-02,  3.094847885527608e-02, 5.160445035091461e-02, 2.135134103898079e-01, 7.426403950254132e-02, -5.067016213329337e-02,  4.631182505184175e-02, -1.425848508489183e-02, 5.086333805562290e-02, 4.706821646579924e-02, 1.589296520950823e-02, -3.459136465971743e-02, -3.985481024913286e-02, 4.211332909910802e-02, 7.426403950254200e-02, 4.627099922123573e-01, -1.884332692097576e-01, -1.216891399014912e-02, 1.026822063160919e-01, 1.526448800873511e-01, -3.023047896977525e-02,  1.076611626529131e-01, -1.017970127860425e-01, -1.096564714613831e-02, -4.193835803731404e-02, -5.067016213329385e-02, -1.884332692097578e-01, 2.681339075714266e-01, -3.731250219879080e-02, -2.735903687031177e-02, -1.021581123913636e-01,  1.800281356097098e-02, -4.731977165540389e-02, 2.652810124466400e-02, 4.881208670995308e-03, 3.903754432397463e-02,  4.631182505184123e-02, -1.216891399014888e-02, -3.731250219879086e-02, 2.017357681676648e-01, 2.648704376643518e-02, -5.749981336663351e-02, 4.166109479630123e-02, 6.046016517035593e-02, -1.534948338758168e-02, -6.009983081874426e-02, -3.385441773702309e-02, -1.425848508489224e-02, 1.026822063160920e-01, -2.735903687031132e-02, 2.648704376643466e-02,  2.859702905773561e-01, -1.209568042346074e-01, 5.786739319787282e-02, 6.301324540318760e-02, -1.898280375953029e-01, -4.666311242964261e-02, -6.215211887602735e-03, 5.086333805562254e-02, 1.526448800873511e-01, -1.021581123913633e-01, -5.749981336663334e-02, -1.209568042346070e-01, 4.952873805249908e-01, -1.329405709219308e-01, -1.068963310495554e-01,  6.647420248371823e-02, -1.738206946067629e-02, 2.845211779683064e-02, 4.706821646579944e-02, -3.023047896977563e-02,  1.800281356097048e-02, 4.166109479630120e-02, 5.786739319787324e-02, -1.329405709219308e-01, 5.029648087380483e-01, -8.739476804560356e-02, -1.767197062952541e-01, 1.486063502168543e-01, 6.907179845259523e-02, 1.589296520950774e-02,  1.076611626529134e-01, -4.731977165540441e-02, 6.046016517035642e-02, 6.301324540318722e-02, -1.068963310495558e-01, -8.739476804560337e-02, 3.958974753400147e-01, 5.404812374174634e-02, -6.713535647210322e-02, 5.744242623014502e-02, -3.459136465971737e-02, -1.017970127860425e-01, 2.652810124466324e-02, -1.534948338758165e-02, -1.898280375953029e-01,  6.647420248371824e-02, -1.767197062952547e-01, 5.404812374174535e-02, 4.706643434109500e-01, 6.454544599923850e-02,  3.094847885527874e-02, -3.985481024913300e-02, -1.096564714613787e-02, 4.881208670995455e-03, -6.009983081874325e-02, -4.666311242964283e-02, -1.738206946067639e-02, 1.486063502168543e-01, -6.713535647210428e-02, 6.454544599923864e-02,  4.240343861640425e-01, 1.072147398468091e-01, 4.938901193257121e-02, 5.111778721467519e-02, -4.267796344882502e-02,  3.055191571342383e-02, -2.121421779865385e-02, 1.721689730618654e-02, 1.128132405377519e-02, 5.145277991022046e-02,  3.140297749914291e-02, 5.407703444717457e-03, 4.938901193256613e-02, 1.800846678364515e-01, 7.308448378548453e-02, -5.044408990229277e-02, 4.054587361662376e-02, -3.212240530851493e-03, 3.713005867575258e-02, 3.523422405796319e-02,  2.875476785940101e-02, -2.996496107467870e-02, -3.279302217023847e-02, 5.111778721465709e-02, 7.308448378548565e-02,  3.278045618396414e-01, -1.454369018943343e-01, 3.193078162150361e-03, 6.037935410571375e-02, 1.065485607181728e-01, -2.436015516607696e-02, 1.006956431645236e-01, -5.425100808750180e-02, -1.309337603066706e-02, -4.267796344881047e-02, -5.044408990229376e-02, -1.454369018943357e-01, 2.276185335867452e-01, -3.655786123651856e-02, -1.553808910184488e-02, -8.315702740611575e-02, 2.132208422725577e-02, -5.197306411840578e-02, 1.034631717379929e-02, 1.026835962844764e-02,  3.055191571342181e-02, 4.054587361662328e-02, 3.193078162151084e-03, -3.655786123651887e-02, 1.676126229390192e-01,  2.782616449364055e-02, -4.265665906781760e-02, 3.503219780087138e-02, 5.356238569224849e-02, -2.421428510146631e-02, -5.401569118081161e-02, -2.121421779864685e-02, -3.212240530852138e-03, 6.037935410571250e-02, -1.553808910184309e-02,  2.782616449363948e-02, 1.945717135038873e-01, -8.639898430809413e-02, 4.861558209373115e-02, 4.417252925986209e-02, -1.371171499058456e-01, -3.496812143313740e-02, 1.721689730616829e-02, 3.713005867575257e-02, 1.065485607181734e-01, -8.315702740611465e-02, -4.265665906781725e-02, -8.639898430809173e-02, 3.660917978350471e-01, -1.134194537295263e-01, -5.228084885863869e-02, 8.777010703179150e-02, -8.123507318534492e-04, 1.128132405378373e-02, 3.523422405796311e-02, -2.436015516607952e-02, 2.132208422725625e-02, 3.503219780087127e-02, 4.861558209373128e-02, -1.134194537295282e-01,  4.047691442451877e-01, -7.825211449249930e-02, -1.665709196181848e-01, 1.212138349106529e-01, 5.145277991022316e-02,  2.875476785940087e-02, 1.006956431645258e-01, -5.197306411840791e-02, 5.356238569224887e-02, 4.417252925985980e-02, -5.228084885863690e-02, -7.825211449250052e-02, 3.244391411346935e-01, 3.531163668422253e-02, -8.159280685404315e-02,  3.140297749914084e-02, -2.996496107467817e-02, -5.425100808749850e-02, 1.034631717379525e-02, -2.421428510146566e-02, -1.371171499058475e-01, 8.777010703179640e-02, -1.665709196181884e-01, 3.531163668421938e-02, 3.934517907851997e-01,  3.514487134346666e-02, 5.407703444724479e-03, -3.279302217023823e-02, -1.309337603066600e-02, 1.026835962844668e-02, -5.401569118080999e-02, -3.496812143313850e-02, -8.123507318524154e-04, 1.212138349106524e-01, -8.159280685404466e-02,  3.514487134346805e-02, 3.547339403822855e-01, 8.171176096460897e-02, 4.538135085008629e-02, 6.133670558753212e-02, -4.390297112190376e-02, 2.318214683174692e-02, -1.583380502672094e-02, 2.796647241270560e-02, 3.754792729205765e-04,  4.703731031942416e-02, 2.219235143718046e-02, -1.882516166020858e-03, 4.538135085008103e-02, 1.639794594015089e-01,  1.056396875706231e-01, -6.766373150761112e-02, 3.604693255783164e-02, 9.737229146445807e-03, 5.778335415452213e-02,  2.629781228207055e-02, 3.016215218535812e-02, -4.478813926105686e-02, -3.789996094153091e-02, 6.133670558751405e-02,  1.056396875706250e-01, 3.970302405953996e-01, -1.996120996585702e-01, 1.219054391223120e-02, 6.975569092721312e-02,  1.719546349735608e-01, -6.450953848111093e-02, 1.266285871314356e-01, -6.727345289259330e-02, -4.806237907435786e-02, -4.390297112188937e-02, -6.766373150761229e-02, -1.996120996585712e-01, 2.378235364892681e-01, -3.216803445548669e-02, -1.839360908397079e-02, -1.310632184361520e-01, 5.580304265270060e-02, -6.760701679097658e-02, 1.465370194403191e-02,  3.250642483314026e-02, 2.318214683174451e-02, 3.604693255783104e-02, 1.219054391223142e-02, -3.216803445548686e-02,  1.419707970038930e-01, 4.010143368536914e-02, -5.042499740852557e-02, 4.017030370811858e-02, 5.216279113173972e-02, -4.014668089418783e-02, -5.003991864601273e-02, -1.583380502671543e-02, 9.737229146445300e-03, 6.975569092721164e-02, -1.839360908396905e-02, 4.010143368536839e-02, 1.942915646396777e-01, -1.107106505560581e-01, 8.718996602574111e-02,  4.770001951469588e-02, -1.695430317855846e-01,
                  -3.437313441099472e-02, 2.796647241268887e-02, 5.778335415452220e-02,  1.719546349735617e-01, -1.310632184361510e-01, -5.042499740852544e-02, -1.107106505560552e-01, 4.617931185208000e-01, -2.044585228399286e-01, -4.270742465647900e-02, 1.252021580913351e-01, -3.675102203493297e-02, 3.754792729270470e-04,  2.629781228207054e-02, -6.450953848111436e-02, 5.580304265270110e-02, 4.017030370811887e-02, 8.718996602574083e-02, -2.044585228399310e-01, 5.076140374564312e-01, -1.172044889252181e-01, -2.443659535181436e-01, 1.658669091227874e-01,  4.703731031942696e-02, 3.016215218535812e-02, 1.266285871314373e-01, -6.760701679097844e-02, 5.216279113174020e-02,  4.770001951469390e-02, -4.270742465647774e-02, -1.172044889252185e-01, 3.308714906616736e-01, 4.788970361033945e-02, -9.677024625911193e-02, 2.219235143718090e-02, -4.478813926105639e-02, -6.727345289258922e-02, 1.465370194402799e-02, -4.014668089418780e-02, -1.695430317855866e-01, 1.252021580913406e-01, -2.443659535181486e-01, 4.788970361033748e-02,  4.412633633110921e-01, 2.102777386401778e-02, -1.882516166013634e-03, -3.789996094153054e-02, -4.806237907435630e-02,  3.250642483313913e-02, -5.003991864601112e-02, -3.437313441099570e-02, -3.675102203493218e-02, 1.658669091227867e-01, -9.677024625911335e-02, 2.102777386401938e-02, 3.528092000246664e-01, 5.597995104300475e-02, 4.377994417395787e-02,  6.310285872530511e-02, -3.835985110182168e-02, 2.072677338411147e-02, 3.488844589337444e-03, 2.342988454853408e-02,  1.478629698043888e-02, 2.785747152707718e-02, -1.974084541837387e-02, -1.684288695522457e-02, 4.377994417395135e-02,  1.263617587513195e-01, 8.671548987701248e-02, -5.264081898699502e-02, 2.209619873270708e-02, -5.954123444288407e-03,  5.733003022896989e-02, -2.562921179300494e-04, 2.818707305056932e-02, -5.216210681675836e-03, -2.080465519278853e-02,  6.310285872528187e-02, 8.671548987701579e-02, 2.832422056726625e-01, -1.372089311135710e-01, 1.153046181242135e-02,  2.748500264075120e-02, 1.236453119508758e-01, -5.774836605528541e-02, 9.883774723275246e-02, -4.457098843223234e-03, -2.116513915934396e-02, -3.835985110180574e-02, -5.264081898699643e-02, -1.372089311135719e-01, 1.709956571520206e-01, -2.574076563927348e-02, -2.255419695772328e-03, -8.584076442003205e-02, 4.398032404882100e-02, -5.329430590008079e-02, -8.411442423823585e-03, 1.686924522839861e-02, 2.072677338410685e-02, 2.209619873270684e-02, 1.153046181242175e-02, -2.574076563927381e-02, 9.846522428367610e-02, 2.111595223227914e-02, -1.717001983800003e-02, 1.567258525420226e-02,  3.095928660913456e-02, -1.141976131912539e-02, -2.565413402843326e-02, 3.488844589332143e-03, -5.954123444286245e-03,  2.748500264075169e-02, -2.255419695771465e-03, 2.111595223227915e-02, 1.211383923834642e-01, -7.350043398939682e-02,  3.948873700801118e-02, 3.499404180056619e-02, -6.478548991394698e-02, 2.588339853622917e-03, 2.342988454852589e-02,  5.733003022896693e-02, 1.236453119508715e-01, -8.584076442002896e-02, -1.717001983800212e-02, -7.350043398940057e-02,  3.005981444189226e-01, -1.206536584402046e-01, -1.174835411752737e-02, 7.396440039008628e-02, -3.401992480091360e-02,  1.478629698043412e-02, -2.562921179274577e-04, -5.774836605529022e-02, 4.398032404882073e-02, 1.567258525420224e-02,  3.948873700800441e-02, -1.206536584401958e-01, 2.799931581889871e-01, -5.276594693058725e-02, -9.444973468960864e-02,  1.091196237725575e-01, 2.785747152707787e-02, 2.818707305057031e-02, 9.883774723275780e-02, -5.329430590008279e-02,  3.095928660913700e-02, 3.499404180057382e-02, -1.174835411753321e-02, -5.276594693057539e-02, 2.106586992434445e-01,  9.617090101862243e-03, -5.331362511354099e-02, -1.974084541835831e-02, -5.216210681677911e-03, -4.457098843217478e-03, -8.411442423826992e-03, -1.141976131912433e-02, -6.478548991393952e-02, 7.396440039007479e-02, -9.444973468961283e-02,  9.617090101876117e-03, 1.937817869355041e-01, -2.417474484117984e-02, -1.684288695521660e-02, -2.080465519278744e-02, -2.116513915934042e-02, 1.686924522839628e-02, -2.565413402843112e-02, 2.588339853624907e-03, -3.401992480091579e-02,  1.091196237725588e-01, -5.331362511354108e-02, -2.417474484118120e-02, 2.251510540766790e-01, 4.358836445264071e-02,  3.402486367601027e-02, 4.619646487578237e-02, -3.169810120866215e-02, 1.441911338367883e-02, -6.462344942766266e-03,  3.583080426369201e-02, -8.952298343405839e-03, 1.912078776823290e-02, -3.519840719594332e-03, -2.452937104564646e-02,  3.402486367600325e-02, 1.046580912277660e-01, 8.352934693937397e-02, -4.718383149599968e-02, 2.060171941752949e-02,  5.941916508247196e-03, 4.036806370027327e-02, 1.402112646301633e-02, 2.578443727160530e-02, -2.323950322455957e-02, -1.821482178209995e-02, 4.619646487575685e-02, 8.352934693937887e-02, 2.572751940225726e-01, -1.211348249647752e-01,  1.532027435494488e-02, 4.411491748646502e-02, 9.702641231346414e-02, -2.565553602946294e-02, 8.635353142832999e-02, -3.857920191909407e-02, -2.631827229622479e-02, -3.169810120864474e-02, -4.718383149600190e-02, -1.211348249647763e-01,  1.459324627081435e-01, -2.607978798419956e-02, -1.059194597243695e-02, -6.349039850358498e-02, 2.447180938246818e-02, -5.183682642311927e-02, 4.819702647661366e-03, 1.326583359991242e-02, 1.441911338367409e-02, 2.060171941752948e-02,  1.532027435494626e-02, -2.607978798420045e-02, 8.281113750281643e-02, 2.109109107005740e-02, -1.294113862884143e-02,  1.204553885038906e-02, 2.693844496719443e-02, -1.841256766785564e-02, -3.150156498877377e-02, -6.462344942771927e-03,  5.941916508250525e-03, 4.411491748646848e-02, -1.059194597243744e-02, 2.109109107005688e-02, 8.513445259977480e-02, -3.511347617297018e-02, 1.673923826171445e-02, 2.728767178422082e-02, -5.037701183684795e-02, -1.386474389153495e-02,  3.583080426368213e-02, 4.036806370026988e-02, 9.702641231345295e-02, -6.349039850357802e-02, -1.294113862884373e-02, -3.511347617297674e-02, 2.215447660647301e-01, -6.249333926517806e-02, 2.824959955661614e-03, 5.027042861878115e-02,  4.379922849321779e-03, -8.952298343409403e-03, 1.402112646301956e-02, -2.565553602946332e-02, 2.447180938246536e-02,  1.204553885038883e-02, 1.673923826170859e-02, -6.249333926516864e-02, 2.029638792942333e-01, -5.888687931243666e-02, -8.611573383397925e-02, 5.213264297034235e-02, 1.912078776823320e-02, 2.578443727160726e-02, 8.635353142833919e-02, -5.183682642312335e-02, 2.693844496719657e-02, 2.728767178422799e-02, 2.824959955660483e-03, -5.888687931242811e-02,  1.834507876854514e-01, 7.593974870537226e-03, -6.177463316282338e-02, -3.519840719578057e-03, -2.323950322456281e-02, -3.857920191908946e-02, 4.819702647658129e-03, -1.841256766785380e-02, -5.037701183683992e-02, 5.027042861876822e-02, -8.611573383398410e-02, 7.593974870551394e-03, 1.852704286929950e-01, 2.089206147909695e-04, -2.452937104563800e-02, -1.821482178209895e-02, -2.631827229621926e-02, 1.326583359990897e-02, -3.150156498877155e-02, -1.386474389153188e-02,  4.379922849319560e-03, 5.213264297034344e-02, -6.177463316282199e-02, 2.089206147890127e-04, 1.790857363514045e-01,  3.868115746490976e-02, 2.580276341331452e-02, 3.321201972357997e-02, -2.421978215335279e-02, 1.225656086454226e-02, -1.239796191552179e-02, 3.068347654348569e-02, -1.564048857624480e-02, 1.821433164694879e-02,
                  9.782499817305777e-03, -2.108143081031552e-02, 2.580276341330766e-02, 8.886301104285542e-02, 7.361321658294918e-02, -4.354605828000414e-02,  1.682461646524194e-02, 1.742273082058427e-03, 4.543152493593675e-02, 4.119945582876983e-03, 1.977677145513970e-02, -2.137687306743102e-02, -2.471266563678070e-02, 3.321201972355595e-02, 7.361321658295400e-02, 2.272124740382112e-01, -1.109947118410505e-01, 7.714227448961431e-03, 3.635442551648015e-02, 1.009481042851409e-01, -3.808657459030512e-02,  7.033227194750426e-02, -3.578186621273490e-02, -3.751802067294242e-02, -2.421978215333608e-02, -4.354605828000623e-02, -1.109947118410512e-01, 1.338181752843074e-01, -1.911601938271160e-02, -9.721735009170594e-04, -7.545328145381859e-02,  3.694446619415361e-02, -4.121847363530302e-02, -1.617262240759011e-04, 2.136891331915738e-02, 1.225656086453729e-02,  1.682461646524188e-02, 7.714227448962474e-03, -1.911601938271224e-02, 7.116402305170910e-02, 1.698167952038005e-02, -1.367205607900992e-02, 1.097910404444170e-02, 2.409125571986115e-02, -1.436104555617713e-02, -3.028729595154638e-02, -1.239796191552813e-02, 1.742273082061246e-03, 3.635442551648382e-02, -9.721735009177133e-04, 1.698167952037814e-02,  9.675553011261737e-02, -5.002535468304838e-02, 3.019054798717558e-02, 2.298425475444396e-02, -7.310038353238275e-02, -1.897711291132146e-02, 3.068347654347774e-02, 4.543152493593394e-02, 1.009481042851306e-01, -7.545328145381244e-02, -1.367205607901065e-02, -5.002535468305688e-02, 2.467271355517321e-01, -8.278983036724538e-02, -8.640861183071284e-03,  5.565652401733917e-02, -7.928831261802161e-03, -1.564048857625069e-02, 4.119945582879473e-03, -3.808657459030627e-02,  3.694446619415161e-02, 1.097910404443933e-02, 3.019054798717123e-02, -8.278983036723485e-02, 2.334280677057330e-01, -7.370264727098735e-02, -1.083381034689119e-01, 6.698538649050070e-02, 1.821433164694967e-02, 1.977677145514146e-02,  7.033227194751183e-02, -4.121847363530642e-02, 2.409125571986363e-02, 2.298425475445189e-02, -8.640861183075110e-03, -7.370264727097624e-02, 1.831615267990975e-01, 2.221250439537571e-02, -6.826261141304321e-02, 9.782499817323455e-03, -2.137687306743340e-02, -3.578186621273029e-02, -1.617262240792179e-04, -1.436104555617308e-02, -7.310038353237461e-02,  5.565652401732330e-02, -1.083381034689142e-01, 2.221250439539137e-02, 2.134292431471407e-01, 9.262850752182267e-03, -2.108143081030805e-02, -2.471266563678003e-02, -3.751802067293790e-02, 2.136891331915480e-02, -3.028729595154458e-02, -1.897711291131757e-02, -7.928831261805158e-03, 6.698538649050254e-02, -6.826261141304296e-02, 9.262850752179086e-03,  1.870805125230552e-01, 3.318987171900897e-02, 1.573588515350322e-02, 1.830164531699434e-02, -1.389874223014401e-02,  5.281476897328954e-03, -2.305742770788402e-02, 3.696167785773608e-02, -3.473499473409056e-02, 1.127646704078566e-02,  2.672487996944445e-02, -2.161013446803004e-02, 1.573588515349632e-02, 5.982718786386711e-02, 2.889549700991120e-02, -1.527283975997846e-02, 8.503456283031105e-03, -8.822344723774633e-03, 2.883430162169626e-02, -3.161705817088455e-03,  4.631517333195998e-03, -1.748457146074737e-03, -1.622286322401465e-02, 1.830164531696984e-02, 2.889549700991534e-02,  1.205838548747314e-01, -5.009942397675712e-02, -2.425707639280372e-03, 1.031705594531601e-02, 6.253234113059933e-02, -4.655604684993610e-02, 3.465297574496976e-02, 8.136157032500415e-03, -2.144213933361742e-02, -1.389874223012677e-02, -1.527283975997989e-02, -5.009942397675680e-02, 8.115642563271244e-02, -1.065081167976757e-02, 7.709001501744766e-03, -3.768504219138171e-02, 2.733453843052012e-02, -2.048557196192498e-02, -1.286827239517619e-02, 6.904052277297216e-03,  5.281476897324328e-03, 8.503456283032052e-03, -2.425707639276064e-03, -1.065081167976994e-02, 5.530303149392082e-02,  1.026955353988607e-02, -3.868042589463869e-03, -1.636817683894416e-03, 1.197921644262416e-02, -3.647036152179432e-03, -2.713274444968947e-02, -2.305742770789087e-02, -8.822344723771015e-03, 1.031705594532343e-02, 7.709001501741649e-03,  1.026955353988183e-02, 1.054337707754620e-01, -5.892049808275534e-02, 4.284202502728306e-02, 4.084505090918813e-03, -8.184693062739849e-02, -1.616825852228407e-02, 3.696167785772824e-02, 2.883430162169133e-02, 6.253234113058206e-02, -3.768504219137209e-02, -3.868042589464472e-03, -5.892049808277076e-02, 1.802870717025704e-01, -5.326137201700345e-02, -1.595089757327225e-03, 6.203680472657829e-02, 1.486458798768099e-02, -3.473499473409776e-02, -3.161705817084311e-03, -4.655604684992983e-02, 2.733453843051370e-02, -1.636817683900954e-03, 4.284202502727930e-02, -5.326137201698116e-02,  1.896034600748784e-01, -7.544245954811307e-02, -1.055783663506612e-01, 3.161289476233643e-02, 1.127646704078684e-02,  4.631517333198147e-03, 3.465297574497882e-02, -2.048557196192938e-02, 1.197921644262679e-02, 4.084505090929306e-03, -1.595089757333214e-03, -7.544245954809727e-02, 1.309681561232047e-01, 3.773462398706372e-02, -4.599742685796118e-02,  2.672487996946301e-02, -1.748457146078530e-03, 8.136157032498578e-03, -1.286827239517549e-02, -3.647036152171350e-03, -8.184693062738932e-02, 6.203680472654990e-02, -1.055783663506595e-01, 3.773462398708298e-02, 1.889242116510925e-01,  8.143154387076353e-03, -2.161013446802285e-02, -1.622286322401371e-02, -2.144213933361070e-02, 6.904052277293102e-03, -2.713274444968889e-02, -1.616825852228058e-02, 1.486458798768158e-02, 3.161289476233592e-02, -4.599742685796108e-02,  8.143154387073331e-03, 1.324443301483917e-01, 3.518243877246465e-02, 1.578462041977440e-02, 1.662078117709618e-02, -1.388295091298936e-02, 3.406231763602240e-03, -3.153696139884019e-02, 4.543837903679795e-02, -2.987090705770241e-02,  4.015668752444600e-03, 3.295306385561587e-02, -1.238147987514632e-02, 1.578462041976795e-02, 5.928630347594697e-02,  3.194477144433685e-02, -1.675255055784405e-02, 1.087831627643816e-02, -3.696044264544369e-03, 2.440310772342551e-02,  5.303185860882718e-03, 4.812483609790419e-03, -9.558064307341067e-03, -1.562275406065793e-02, 1.662078117707319e-02,  3.194477144434087e-02, 1.256380958657254e-01, -5.302905574782560e-02, -4.680540150211565e-05, 1.753795153691843e-02,  5.788593257069304e-02, -2.558424213655533e-02, 3.221969963201142e-02, -1.304189874573788e-02, -2.351912499644103e-02, -1.388295091297284e-02, -1.675255055784575e-02, -5.302905574782683e-02, 7.908862164947353e-02, -1.217173193969098e-02,  6.806661438998911e-03, -3.655183700681942e-02, 1.644095415214933e-02, -1.614750741119300e-02, -2.459734119803937e-03,  8.411099767472840e-03, 3.406231763597522e-03, 1.087831627643783e-02, -4.680540150245327e-05, -1.217173193969077e-02,  5.666084089062712e-02, 1.497442491974560e-02, -6.256621778438850e-03, 7.170366153079234e-04, 1.391476010828070e-02, -1.073543888066659e-02, -2.960718002160501e-02, -3.153696139884681e-02, -3.696044264544358e-03, 1.753795153691225e-02,  6.806661439002900e-03, 1.497442491974165e-02, 1.272196975024444e-01, -7.253047279078566e-02, 5.830708558264221e-02,  8.436374362126112e-03, -1.141994793126626e-01, -2.238315090188197e-02, 4.543837903679120e-02, 2.440310772342593e-02,  5.788593257069554e-02, -3.655183700681845e-02, -6.256621778435469e-03, -7.253047279078796e-02, 1.818134663109345e-01, -5.905273456424017e-02,
                  -2.868637734189749e-03, 7.619577397976175e-02, 8.315923867422735e-03, -2.987090705771047e-02,  5.303185860881099e-03, -2.558424213656882e-02, 1.644095415215179e-02, 7.170366152982350e-04, 5.830708558262652e-02, -5.905273456422034e-02, 1.925353968536600e-01, -6.043016591304433e-02, -1.180048933895161e-01, 3.836872344149943e-02,  4.015668752446248e-03, 4.812483609791690e-03, 3.221969963201678e-02, -1.614750741119492e-02, 1.391476010828468e-02,  8.436374362141192e-03, -2.868637734205199e-03, -6.043016591301936e-02, 1.198692152602268e-01, 2.618391769195668e-02, -3.900557126543813e-02, 3.295306385563435e-02, -9.558064307340207e-03, -1.304189874572291e-02, -2.459734119811965e-03, -1.073543888065876e-02, -1.141994793126526e-01, 7.619577397974894e-02, -1.180048933895286e-01, 2.618391769198181e-02,  2.229935111566278e-01, 1.645705868654344e-02, -1.238147987513953e-02, -1.562275406065810e-02, -2.351912499643751e-02,  8.411099767469215e-03, -2.960718002160677e-02, -2.238315090188654e-02, 8.315923867431744e-03, 3.836872344148952e-02, -3.900557126544078e-02, 1.645705868655068e-02, 1.328682327451591e-01, 3.714118390294955e-02, 1.780094898790507e-02,  2.361922740166334e-02, -1.786256105658001e-02, 2.800826959592741e-03, -3.005672004822169e-02, 5.061028740232701e-02, -2.558450321542670e-02, 9.375488215463690e-04, 2.710143472341046e-02, -8.938375236574529e-03, 1.780094898789887e-02,  6.057443357432225e-02, 3.245809790231314e-02, -1.970544336856320e-02, 1.109427749641707e-02, -4.902582371081301e-03,  2.504659569535776e-02, 3.561068214214968e-03, 6.719007561114648e-03, -8.954022971555828e-03, -1.534289065977650e-02,  2.361922740164159e-02, 3.245809790231755e-02, 1.460842354858560e-01, -6.523698461733719e-02, -1.209044640939816e-03,  2.567305994590982e-02, 5.571033999804395e-02, -2.110100418912339e-02, 4.517243885682654e-02, -2.104719847601077e-02, -2.126725107355804e-02, -1.786256105656458e-02, -1.970544336856517e-02, -6.523698461733859e-02, 8.671192563248353e-02, -1.500444939833294e-02, 7.831319272633124e-04, -3.125795929402617e-02, 7.121705936870345e-03, -2.228977693181229e-02,  9.008882460532820e-03, 7.579443295634186e-03, 2.800826959587293e-03, 1.109427749641629e-02, -1.209044640942995e-03, -1.500444939833084e-02, 5.938807784277762e-02, 1.871758034907758e-02, -1.157130776038525e-02, 9.193038437729572e-03,  1.334119414758406e-02, -1.482521897655541e-02, -2.772009465489091e-02, -3.005672004823018e-02, -4.902582371082408e-03,  2.567305994589714e-02, 7.831319272712869e-04, 1.871758034907195e-02, 1.353658786260171e-01, -8.175212320905985e-02,  6.247847763620293e-02, 2.304732407950599e-02, -1.090031748178057e-01, -1.994367215841274e-02, 5.061028740232452e-02,  2.504659569536001e-02, 5.571033999805670e-02, -3.125795929403180e-02, -1.157130776037811e-02, -8.175212320905795e-02,  1.901194806978594e-01, -6.493959927882730e-02, -6.598295153734989e-03, 6.902232741972106e-02, 2.887003501470461e-03, -2.558450321543880e-02, 3.561068214211736e-03, -2.110100418914485e-02, 7.121705936877337e-03, 9.193038437714419e-03,  6.247847763618007e-02, -6.493959927880079e-02, 1.809314565632642e-01, -4.738698209553611e-02, -9.825210827893174e-02,  3.845867764051725e-02, 9.375488215479198e-04, 6.719007561115523e-03, 4.517243885682818e-02, -2.228977693181132e-02,  1.334119414759008e-02, 2.304732407952614e-02, -6.598295153761442e-03, -4.738698209550044e-02, 1.215393211427238e-01,  8.079703455100704e-03, -3.711689258723937e-02, 2.710143472343172e-02, -8.954022971553725e-03, -2.104719847598956e-02,  9.008882460521405e-03, -1.482521897654382e-02, -1.090031748177921e-01, 6.902232741970574e-02, -9.825210827894663e-02,  8.079703455133310e-03, 1.946257174938296e-01, 2.118280460864147e-02, -8.938375236568798e-03, -1.534289065977696e-02, -2.126725107355564e-02, 7.579443295630987e-03, -2.772009465489438e-02, -1.994367215842144e-02, 2.887003501484355e-03,  3.845867764050352e-02, -3.711689258724442e-02, 2.118280460865229e-02, 1.337282487261468e-01, 3.721567651794819e-02,  1.774308033043163e-02, 2.535245852065382e-02, -1.763227835053749e-02, 1.937467245382686e-03, -2.649122139053900e-02,  5.366723804831447e-02, -2.458769815267896e-02, -6.434547898400778e-03, 1.567733541348410e-02, -1.357266955309398e-02,  1.774308033042594e-02, 5.823364071197408e-02, 2.461978887911226e-02, -1.534788243825851e-02, 7.745228956589504e-03, -9.710368220699171e-03, 2.457282505545787e-02, -1.003645772374752e-03, -4.842741180601145e-04, -3.918474336564108e-03, -1.199301497125140e-02, 2.535245852063340e-02, 2.461978887912047e-02, 1.310665461297283e-01, -5.519405084248980e-02, -8.047202196080794e-03, 1.219599164943801e-02, 5.486261111306896e-02, -1.983897444390045e-02, 1.827397099124433e-02, -1.757690512172458e-02, -1.117051176604691e-02, -1.763227835052290e-02, -1.534788243826282e-02, -5.519405084249202e-02,  8.311085652067340e-02, -1.182120633747921e-02, 9.561523794125320e-03, -2.926861951574359e-02, 5.742014766987810e-03, -6.990674303889001e-03, 9.171871126456193e-03, 3.468680231604138e-03, 1.937467245377530e-03, 7.745228956587147e-03, -8.047202196090598e-03, -1.182120633747350e-02, 5.802064053457948e-02, 1.140034620912769e-02, -8.430516473634367e-03,  1.058497113855259e-02, 5.013454248634912e-03, -7.108567645152138e-03, -1.845503377389644e-02, -2.649122139054725e-02, -9.710368220703116e-03, 1.219599164941144e-02, 9.561523794141200e-03, 1.140034620912208e-02, 9.968021074211826e-02, -5.998694382145830e-02, 4.423323680815435e-02, 1.398642150261472e-02, -6.506406562923207e-02, -6.818993000759373e-04,  5.366723804831348e-02, 2.457282505546820e-02, 5.486261111310197e-02, -2.926861951576012e-02, -8.430516473620290e-03, -5.998694382144331e-02, 1.703817817833378e-01, -4.366021081118480e-02, -1.338764247109286e-02, 3.767556088408813e-02, -7.916868599535953e-04, -2.458769815269152e-02, -1.003645772386803e-03, -1.983897444395330e-02, 5.742014767011522e-03,  1.058497113853290e-02, 4.423323680811999e-02, -4.366021081117218e-02, 1.427957163010488e-01, -2.915248075143731e-02, -6.365491608212596e-02, 2.476874692484477e-02, -6.434547898399124e-03, -4.842741180587796e-04, 1.827397099124662e-02, -6.990674303887703e-03, 5.013454248642139e-03, 1.398642150263940e-02, -1.338764247112564e-02, -2.915248075139323e-02,  9.535829210368971e-02, 4.848354984519406e-03, -1.834442765361889e-02, 1.567733541350482e-02, -3.918474336557731e-03, -1.757690512168045e-02, 9.171871126432011e-03, -7.108567645140868e-03, -6.506406562921760e-02, 3.767556088409338e-02, -6.365491608215859e-02, 4.848354984558890e-03, 1.343784571963818e-01, 3.470579716479807e-03, -1.357266955308902e-02, -1.199301497125489e-02, -1.117051176605081e-02, 3.468680231603957e-03, -1.845503377390349e-02, -6.818993000928921e-04, -7.916868599342776e-04, 2.476874692481917e-02, -1.834442765362594e-02, 3.470579716503204e-03, 1.016925687660520e-01,  4.321971207119854e-02, 1.956476620463100e-02, 2.489685926849219e-02, -1.861487238954306e-02, 2.243636825822945e-03, -3.183658991377039e-02, 6.500944437381201e-02, -2.495119680293242e-02, -1.118278411350104e-02, 1.860780240351692e-02, -1.043300297495678e-02, 1.956476620462499e-02, 7.902513964316007e-02, 3.743757429774125e-02, -2.115971709043254e-02,  1.407678850911955e-02, -7.670494913596657e-03, 3.318514451806567e-02,
                  3.386876714042598e-03, 3.388310739942904e-03, -1.573952377472757e-02, -2.117419677917748e-02, 2.489685926847111e-02, 3.743757429774852e-02, 1.435510531826900e-01, -5.241812990643402e-02, -7.544946571048518e-03, 1.423153147548789e-02, 6.571573059980584e-02, -1.433234479001455e-02,  2.244772199377904e-02, -2.095915873696458e-02, -9.220846337025856e-03, -1.861487238952798e-02, -2.115971709043618e-02, -5.241812990643580e-02, 9.454271029648909e-02, -1.572634359572563e-02, 1.586859633980586e-02, -3.726587657965097e-02,  7.867428382706265e-03, -1.097858267128712e-02, 2.555230294484984e-03, 6.014867818006364e-04, 2.243636825818098e-03,  1.407678850911843e-02, -7.544946571054416e-03, -1.572634359572268e-02, 7.597409762298424e-02, 6.985786692420226e-03, -5.774991458904755e-04, -1.310673677717700e-04, 1.181128213810048e-02, -2.654268641927082e-03, -3.298176557153513e-02, -3.183658991377833e-02, -7.670494913598322e-03, 1.423153147547124e-02, 1.586859633981527e-02, 6.985786692413593e-03,  1.177413459245218e-01, -6.852912356382523e-02, 3.376193788499560e-02, 2.018959277641430e-02, -7.315060373647908e-02, -7.525887471344778e-03, 6.500944437380926e-02, 3.318514451807103e-02, 6.571573059982405e-02, -3.726587657965830e-02, -5.774991458794565e-04, -6.852912356382224e-02, 2.142115386103429e-01, -3.426902415894828e-02, -1.303768735845819e-02,  4.610311069242434e-02, 3.496853727468219e-03, -2.495119680294382e-02, 3.386876714034608e-03, -1.433234479005291e-02,  7.867428382720920e-03, -1.310673677903940e-04, 3.376193788496806e-02, -3.426902415892874e-02, 1.563632548765068e-01, -3.589515326121134e-02, -6.364417534375133e-02, 2.774566883648966e-02, -1.118278411349932e-02, 3.388310739945166e-03,  2.244772199378398e-02, -1.097858267128776e-02, 1.181128213810692e-02, 2.018959277643781e-02, -1.303768735848552e-02, -3.589515326117115e-02, 1.226896342589492e-01, 4.478866840940319e-03, -2.258202694954642e-02, 1.860780240353729e-02, -1.573952377472398e-02, -2.095915873693371e-02, 2.555230294469357e-03, -2.654268641914065e-03, -7.315060373646420e-02,  4.610311069241346e-02, -6.364417534377464e-02, 4.478866840978552e-03, 1.716407305563972e-01, 2.803102663107689e-02, -1.043300297495117e-02, -2.117419677918010e-02, -9.220846337027799e-03, 6.014867817996650e-04, -3.298176557154068e-02, -7.525887471357601e-03, 3.496853727483373e-03, 2.774566883646878e-02, -2.258202694955169e-02, 2.803102663109515e-02,  1.484426464460484e-01 ))

  expect_equal(c(result_exp$lag_one_cov),
               c(NULL ))

  expect_equal(c(result_exp$fixed_effects),
               c(numeric(0) ))

  expect_equal(c(result_exp$n_iter),
               c(10 ))

  expect_equal(c(result_exp$Q),
               c( 0.09838852237126759, 0.05140918756478066, 0.11378507294515343, -0.09078805806262612, 0.02719767155293180, -0.05290452939263308,  0.07070072659809151, -0.05554707944678014, 0.11526395792495003, 0.11881884070868155, 0.02692412203630609, 0.05140918756478066,  0.18070898587386780, 0.38624749510275336, -0.23089328553314165, 0.05502541870514686, 0.09953844046870351, 0.20787769897346381,  0.04569575566736697, 0.07512156562174019, -0.20190286296130458, -0.10242053211100018, 0.11378507294515343, 0.38624749510275336,  1.56608602278498887, -0.86319307760929265, 0.06112478078223799, 0.43793191478287519, 0.82260478646035451, -0.28413741056078473,  0.43927901470634712, -0.56964149484040150, -0.40739492529819687, -0.09078805806262612, -0.23089328553314165, -0.86319307760929265,  0.53862760774702922, -0.03601608629558598, -0.14621803528743477, -0.57630403946470055, 0.25410269303610433, -0.23778632643303871,  0.18690631641477357, 0.22676602136325572, 0.02719767155293180, 0.05502541870514686, 0.06112478078223799, -0.03601608629558598,  0.13150468182004410, 0.18384054612210302, -0.23223041189381555, 0.20934184577675780, 0.11494013792053398, -0.23330587728650082, -0.06178030957995829, -0.05290452939263308, 0.09953844046870351, 0.43793191478287519, -0.14621803528743477, 0.18384054612210302,  0.79751653056014415, -0.60588921969678178, 0.71241919118590546, 0.19919278837192433, -1.05668917820002184, -0.12448776882467145,  0.07070072659809151, 0.20787769897346381, 0.82260478646035451, -0.57630403946470055, -0.23223041189381555, -0.60588921969678178,  1.72585199125573086, -1.23609558379502849, -0.04727168238344588, 0.74345070955969850, -0.30242464802816338, -0.05554707944678014,  0.04569575566736697, -0.28413741056078473, 0.25410269303610433, 0.20934184577675780, 0.71241919118590546, -1.23609558379502849,  1.91750486405226250, -0.34381881737516251, -1.31417418088687787, 0.55763211720650641, 0.11526395792495003, 0.07512156562174019,  0.43927901470634712, -0.23778632643303871, 0.11494013792053398, 0.19919278837192433, -0.04727168238344588, -0.34381881737516251,  0.57625369753258326, 0.01805390567072239, -0.28623143334541323, 0.11881884070868155, -0.20190286296130458, -0.56964149484040150,  0.18690631641477357, -0.23330587728650082, -1.05668917820002184, 0.74345070955969850, -1.31417418088687787, 0.01805390567072239,  1.70627043743598716, 0.06655144230407503, 0.02692412203630609, -0.10242053211100018, -0.40739492529819687, 0.22676602136325572, -0.06178030957995829, -0.12448776882467145, -0.30242464802816338, 0.55763211720650641, -0.28623143334541323, 0.06655144230407503,  0.57769489356176029 ))

  expect_equal(c(result_exp$Q_0),
               c(1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1 ))

  expect_equal(c(result_exp$n_risk),
               c(2007, 1936, 1882, 1818, 1767, 1656, 1543, 1364, 1254, 1103 ))

  expect_equal(c(result_exp$times),
               c( 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10 ))

  expect_equal(c(result_exp$risk_set),
               c(NULL ))

  expect_equal(c(result_exp$data),
               c(NULL ))

  expect_equal(c(result_exp$order),
               c(1 ))

  expect_equal(c(result_exp$F_),
               c(1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1 ))

  expect_equal(c(result_exp$method),
               c("UKF" ))

  expect_equal(c(result_exp$model),
               c("exp_combined" ))

  expect_equal(c(result_exp$est_Q_0),
               c(FALSE ))

  expect_equal(c(result_exp$LR),
               c(1 ))
})


test_that("UKF on simulated data works with exponential models with only one of the variables", {
  set.seed(9997)
  sims <- test_sim_func_exp(n_series = 1e3, n_vars = 10, t_0 = 0, t_max = 10,
                            x_range = 1, x_mean = 0, re_draw = T, beta_start = (1:10 - 5) / 2.5,
                            intercept_start = -5, sds = c(.1, rep(1, 10)))
  # tmp <- file("tmp.txt")
  # sink(tmp)
  suppressMessages(result_exp <- ddhazard(
    formula = survival::Surv(tstart, tstop, event) ~ . - id - tstart - tstop - event,
    data = sims$res,
    by = (by_ <- 1),
    Q_0 = diag(1, 11),
    Q = diag(1e-1, 11),
    control = list(est_Q_0 = F, eps = 10^-2, n_max = 10^3, method = "UKF",
                   debug = F, beta = 0, save_data = F, save_risk_set = F),
    max_T = 10,
    id = sims$res$id, order = 1,
    verbose = F,
    model = "exp_bin"))
  # sink()
  # close(tmp)

  # sum(sims$res$event)
  # diag(result_exp$Q)
  # result_exp$LR
  # matplot(sims$betas, type = "l", lty = 1)
  # matplot(result_exp$state_vecs, type = "l", lty = 2, add = T)
  # get_expect_equal(result_exp, file = "tmp.txt", eps = 1e-4)

  expect_equal(unname(c(result_exp$state_vecs)),
               c(-5.51472578565331162, -5.52219508683001958, -5.34488323395387077, -5.21075555005431834, -4.88775556731426342, -4.94594587629111349, -5.83949944994586634, -6.04584572385986707, -6.00715459434778776, -5.54529203934886805, -5.41434293273603906, -2.36828182241952367, -2.34643348809331131, -3.13077943715389440, -2.84319116551544360, -2.28893521611891471, -2.29281740382555510, -3.13779385652566134, -3.90308293853243837, -4.22876193574193326, -4.39688420932996671, -4.72106536486603368, -0.31726081128055211, -0.26093924589070894, -0.90080860049079814, -0.93277717600702781, 0.50002496005736741, 0.04053221471233476, -0.29237480751315248, -1.33851786435145992, -3.07880546634924457, -2.10865447657123095, -0.32368764821149787, 0.05202796800288381, 0.02787464987360805,  0.12039592793642881, 0.29101149297477358, -0.57835941878755381, -0.27995428690483565, -0.07817168699826887,  0.26076338542009680, 1.38022486694823732, 0.40619961013134936, -1.05777130519996310, -1.00228459361336886, -1.02891408118382377, -0.46966897565548715, -0.42959729157855192, -0.29134045366606748, 0.06230090570183255,  0.33145788833534290, 0.88809022619161526, 0.88105264757880275, 2.01310138044987186, 2.96228010930929386,  0.50949584249235291, 0.52395162301979992, -0.76844401970473197, 0.77403436744479848, 0.95247685933797110,  3.06001498089554413, 4.64665508234231961, 4.96384706874857162, 4.50826245195286113, 4.48596390880468299,  3.80788430596693406, -1.29479381091853640, -1.24499371383896595, -1.60208425032152135, -2.93642869481111468, -1.79754099901487874, -4.11688765511707189, -6.29158586411182341, -7.87541875965626215, -8.61909487294573573, -8.63830262778433777, -7.63473127875391633, -0.12336478912440094, -0.17471479067184426, -0.65584204024345971,  1.19771727407709228, 0.90638218254936387, 3.27123462637233464, 3.33071385746688975, 4.10933751644596246,  5.31328162118733172, 5.27609119105544622, 3.27603015523435337, 0.74564016958090285, 0.71536833754001450,  2.10466508373742878, 1.82275592057528657, 1.83126120890277622, 2.05613852741351710, 3.58962188925160497,  4.62092329144558267, 4.24814798486439305, 6.09980607801966812, 8.39569005566672288, 1.88960891214721993,  1.86116323629468328, 3.73779931445987845, 2.34056320002551610, 1.88647341174965044, 0.47564591612740509,  0.42350052088690754, 0.94852534173724834, 1.13282588956022967, 1.79476413758884612, 3.29532509139697227,  2.45438013226445273, 2.42150500231968646, 3.00230248857148396, 3.65408592284578315, 3.38519951125859730,  4.20224810465785659, 3.99737716703125656, 4.45080152280818275, 4.85694813031414352, 5.17394588622765461,  4.75850013043876885 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$state_vars)),
               c( 1.923268426396250e-01, 7.651461340533083e-02, 6.476243662324264e-02, -4.108768125916929e-02,  5.759452205219784e-02, -9.107459619691520e-02, 8.997378541900984e-02, 8.705128402278207e-02, -1.959511075422923e-02, -3.616909948148767e-03, 7.746738937863423e-02, 7.651461340533375e-02,  2.187369952043103e-01, 8.597079574167654e-02, -2.154985439677154e-02, -2.366026379966888e-02,  8.810741578752888e-03, 1.046915698749137e-01, 4.296689349890145e-02, -1.095550430185475e-01, -1.082551691390096e-01, -8.122532596068518e-04, 6.476243662324180e-02, 8.597079574167681e-02,  4.437350278320050e-01, -1.942371096687977e-01, 2.437406722958563e-02, 8.338858228601721e-02,  1.389373203507385e-01, -9.219869357379723e-02, 8.247546758962369e-02, -6.018873715806210e-02, -8.255252872902978e-03, -4.108768125917008e-02, -2.154985439677187e-02, -1.942371096687976e-01,  3.166986613951813e-01, -8.171995671528565e-02, -2.734427194061969e-02, -7.045920731056078e-02,  5.360695782731400e-02, -8.491707862423260e-02, -7.932746536880447e-03, -9.022038121459927e-03,  5.759452205219716e-02, -2.366026379966872e-02, 2.437406722958566e-02, -8.171995671528556e-02,  2.856134697108819e-01, 2.794476255624533e-04, -2.195955392063503e-02, 6.463517673394373e-02,  1.726597693395487e-01, 5.505967239810755e-02, 1.771472008629014e-02, -9.107459619691698e-02,  8.810741578752805e-03, 8.338858228601720e-02, -2.734427194061978e-02, 2.794476255623731e-04,  4.250066173054523e-01, -1.850088788857069e-01, 8.739554969639610e-02, 3.328620648287561e-02, -1.893428370620307e-01, -2.451998348517117e-02, 8.997378541900848e-02, 1.046915698749138e-01,  1.389373203507384e-01, -7.045920731056092e-02, -2.195955392063525e-02, -1.850088788857072e-01,  5.116606644985962e-01, -1.115609215873116e-01, -1.204050110842179e-01, 1.464883074324353e-02, -5.556582532179493e-02, 8.705128402278370e-02, 4.296689349890129e-02, -9.219869357379706e-02,  5.360695782731400e-02, 6.463517673394352e-02, 8.739554969639585e-02, -1.115609215873120e-01,  5.057506453732092e-01, -5.300066932771634e-02, -1.371024728197386e-01, 1.254099429061856e-01, -1.959511075423081e-02, -1.095550430185475e-01, 8.247546758962379e-02, -8.491707862423267e-02,  1.726597693395486e-01, 3.328620648287571e-02, -1.204050110842180e-01, -5.300066932771641e-02,  5.461117260204287e-01, 1.465105911926102e-01, -4.981186595954742e-04, -3.616909948141897e-03, -1.082551691390099e-01, -6.018873715806232e-02, -7.932746536880506e-03, 5.505967239810738e-02, -1.893428370620310e-01, 1.464883074324328e-02, -1.371024728197384e-01, 1.465105911926101e-01,  4.260765120933141e-01, 3.685973669667802e-02, 7.746738937863280e-02, -8.122532596071797e-04, -8.255252872903043e-03, -9.022038121459651e-03, 1.771472008629030e-02, -2.451998348517111e-02, -5.556582532179471e-02, 1.254099429061858e-01, -4.981186595954604e-04, 3.685973669667771e-02,  3.606631164652518e-01, 1.696778938618631e-01, 9.696308540217681e-02, 8.981798855361632e-02, -5.160440601433539e-02, 3.427119388365811e-02, -8.218449733543626e-02, 1.413671145309273e-01,  6.110301589750847e-02, -6.112921928925068e-02, -3.058400837463258e-02, 5.312166252702352e-02,  9.696308540217831e-02, 2.202754950180425e-01, 8.978600495075176e-02, -2.313288334756600e-02, -3.818482652873274e-02, -1.414212950403710e-02, 1.415850707392447e-01, 4.799306053423422e-02, -1.501560159485651e-01, -1.202608899705193e-01, 9.997778493057247e-03, 8.981798855361316e-02,  8.978600495075238e-02, 3.220867836793816e-01, -1.542425831477035e-01, 2.508951374407591e-02, -1.982962654081828e-03, 1.694673501881211e-01, -7.175645127315074e-02, 3.701795467960663e-02, -2.509051252666786e-02, 1.184452637746199e-03, -5.160440601433598e-02, -2.313288334756702e-02, -1.542425831477035e-01, 2.821743169576615e-01, -8.458133603966395e-02, 1.949989647672087e-02, -9.801454986964586e-02, 6.210420310912840e-02, -8.349237553441809e-02, -3.751079359254801e-02, -6.329097725615886e-03, 3.427119388365786e-02, -3.818482652873252e-02, 2.508951374407630e-02, -8.458133603966411e-02, 2.623819381522989e-01, -2.559490775366329e-03, -2.448968967785448e-02,  3.512901880557541e-02, 1.928278592733481e-01, 7.568827611252262e-02, 8.714188910267984e-03, -8.218449733544669e-02, -1.414212950403823e-02, -1.982962654082879e-03, 1.949989647672132e-02, -2.559490775366640e-03, 2.564402091984790e-01, -1.612473982645521e-01, 6.052714738665702e-02,  3.296540761742424e-02, -1.110388424688537e-01, -2.011928787688851e-02, 1.413671145309295e-01,  1.415850707392455e-01, 1.694673501881210e-01, -9.801454986964565e-02, -2.448968967785492e-02, -1.612473982645509e-01, 5.264587103783831e-01, -9.202687555439890e-02, -1.567033953535738e-01, -5.933957586483699e-03, -2.515244300292619e-02, 6.110301589750512e-02, 4.799306053423284e-02, -7.175645127315015e-02, 6.210420310912873e-02, 3.512901880557538e-02, 6.052714738665791e-02, -9.202687555440006e-02, 4.467798731001197e-01, -7.861540333766520e-02, -1.530645394373723e-01,  1.098797384483083e-01, -6.112921928925062e-02, -1.501560159485648e-01, 3.701795467960693e-02, -8.349237553441896e-02, 1.928278592733479e-01, 3.296540761742327e-02, -1.567033953535733e-01, -7.861540333766617e-02, 5.764234234481214e-01, 1.945922615874144e-01, -2.349563948638516e-02, -3.058400837461622e-02, -1.202608899705178e-01, -2.509051252666666e-02, -3.751079359254952e-02,  7.568827611252285e-02, -1.110388424688573e-01, -5.933957586480090e-03, -1.530645394373720e-01,  1.945922615874129e-01, 4.023040600658505e-01, 9.706230189333954e-03, 5.312166252702144e-02,  9.997778493056303e-03, 1.184452637745637e-03, -6.329097725614841e-03, 8.714188910268067e-03, -2.011928787688659e-02, -2.515244300292757e-02, 1.098797384483098e-01, -2.349563948638533e-02,  9.706230189331574e-03, 3.049398356563144e-01, 1.504443075202616e-01, 1.105552420795828e-01,  1.122920330950666e-01, -5.972967356594262e-02, 1.406279394351812e-02, -9.823196496940925e-02,  1.959322864422813e-01, 2.918203239279705e-02, -9.375891671218736e-02, -3.873699115444360e-02,  3.000787881044797e-02, 1.105552420795821e-01, 2.471303533320026e-01, 1.351359197457012e-01, -4.220445027543782e-02, -6.397279952485491e-02, -2.395411135527041e-02, 2.316233808940710e-01,  3.535106246762827e-02, -2.320744323195613e-01, -1.667486273768276e-01, -1.157820923321486e-02,  1.122920330950643e-01, 1.351359197457038e-01, 3.872436234107768e-01, -2.057029540360072e-01,  1.940787349041602e-02, -2.952633001226158e-02, 2.864306908160861e-01, -1.274989062641373e-01,  5.379993929266545e-03, -4.419142592961815e-02, -3.116027323503708e-02, -5.972967356594525e-02, -4.220445027544095e-02, -2.057029540360084e-01, 2.940701956795454e-01, -8.152492292444188e-02,  4.146649958848668e-02, -1.613437865593136e-01, 1.122572704897049e-01, -9.343821138514190e-02, -4.400297465488888e-02, 1.404642248151518e-02, 1.406279394351918e-02, -6.397279952485410e-02,  1.940787349041680e-02, -8.152492292444226e-02, 2.504595563799026e-01, 1.043859207033572e-02, -7.406379926667654e-02, 2.545435655099299e-02, 2.496150892398501e-01, 1.038061066342547e-01,  1.954462545321218e-02, -9.823196496942310e-02, -2.395411135527575e-02, -2.952633001226405e-02,  4.146649958848598e-02, 1.043859207033665e-02, 3.343480142309123e-01, -2.760085140529045e-01,  1.454181901347417e-01, 5.222664511724668e-02, -1.628667042931542e-01,
                  5.177612588905570e-03,  1.959322864422852e-01, 2.316233808940758e-01, 2.864306908160857e-01, -1.613437865593106e-01, -7.406379926667808e-02, -2.760085140529001e-01, 7.796913750250503e-01, -2.139413296689412e-01, -2.971858473419974e-01, -2.531853523720467e-02, -8.575835993774039e-02, 2.918203239278921e-02,  3.535106246762364e-02, -1.274989062641393e-01, 1.122572704897060e-01, 2.545435655099286e-02,  1.454181901347457e-01, -2.139413296689481e-01, 5.317792620678389e-01, -1.020272601974439e-01, -2.122985248559550e-01, 1.425346857813387e-01, -9.375891671218291e-02, -2.320744323195593e-01,  5.379993929269952e-03, -9.343821138514533e-02, 2.496150892398503e-01, 5.222664511723970e-02, -2.971858473419902e-01, -1.020272601974480e-01, 7.526570528541554e-01, 3.007451932896588e-01,  5.192725818951850e-03, -3.873699115442167e-02, -1.667486273768223e-01, -4.419142592961323e-02, -4.400297465489190e-02, 1.038061066342546e-01, -1.628667042931624e-01, -2.531853523719339e-02, -2.122985248559554e-01, 3.007451932896532e-01, 4.809355722024071e-01, 8.641122726576224e-03,  3.000787881044524e-02, -1.157820923321585e-02, -3.116027323503808e-02, 1.404642248151671e-02,  1.954462545321185e-02, 5.177612588908057e-03, -8.575835993774258e-02, 1.425346857813407e-01,  5.192725818951201e-03, 8.641122726573282e-03, 2.913840468160200e-01, 1.149945846220265e-01,  1.046139063416875e-01, 1.005450998462516e-01, -4.247304944925478e-02, -3.400288584738391e-03, -7.157406009287501e-02, 1.748516262715245e-01, 2.207080279636106e-02, -1.052088338015967e-01, -5.960966727297101e-02, 5.878908508304637e-03, 1.046139063416743e-01, 2.078423865140026e-01,  1.225998021697736e-01, -4.312047840514158e-02, -4.224941717566565e-02, -3.794340003892636e-02,  2.032167620503543e-01, 1.854281806425371e-02, -1.767108225700798e-01, -1.151684080394962e-01, -2.951054208764245e-03, 1.005450998462539e-01, 1.225998021697908e-01, 2.277798004795510e-01, -9.704471629265292e-02, -7.316250778657617e-03, -4.327861447395766e-02, 2.082038184677166e-01, -2.889834495208626e-02, -7.288398515501779e-02, -5.796509968377366e-02, -2.346105925476107e-03, -4.247304944926773e-02, -4.312047840515851e-02, -9.704471629266129e-02, 1.789471958447840e-01, -3.499829948473555e-02, 3.274444754849427e-02, -1.042425791680857e-01, 3.226767892477123e-02, -1.798712859751306e-03, 2.296092264387074e-03, 2.781675130879402e-03, -3.400288584727424e-03, -4.224941717565886e-02, -7.316250778649367e-03, -3.499829948474054e-02, 1.619094179675277e-01,  2.122148915069162e-02, -5.238493230072527e-02, 3.290409689120000e-02, 1.297268236044833e-01,  3.931695771327144e-02, 3.850577452940047e-04, -7.157406009291165e-02, -3.794340003896743e-02, -4.327861447397299e-02, 3.274444754848786e-02, 2.122148915070957e-02, 2.255211191032607e-01, -2.189785263431913e-01, 8.052749495176870e-02, 7.534877362850149e-02, -5.832910492567631e-02,  2.297450338497860e-02, 1.748516262715375e-01, 2.032167620503930e-01, 2.082038184677139e-01, -1.042425791680637e-01, -5.238493230074754e-02, -2.189785263431524e-01, 6.105953880658713e-01, -1.119146985057568e-01, -2.569358321293200e-01, -4.295631840408058e-02, -4.755489093441648e-02,  2.207080279632574e-02, 1.854281806422075e-02, -2.889834495210870e-02, 3.226767892477759e-02,  3.290409689120706e-02, 8.052749495179860e-02, -1.119146985058155e-01, 3.135885419512443e-01, -1.803720771278426e-02, -1.012938131054138e-01, 8.960657466983553e-02, -1.052088338015637e-01, -1.767108225700665e-01, -7.288398515498715e-02, -1.798712859775127e-03, 1.297268236044885e-01,  7.534877362843360e-02, -2.569358321292451e-01, -1.803720771282670e-02, 4.749115349497947e-01,  1.481771433135240e-01, -1.368014082646261e-02, -5.960966727291452e-02, -1.151684080394610e-01, -5.796509968373995e-02, 2.296092264371670e-03, 3.931695771326468e-02, -5.832910492572964e-02, -4.295631840399788e-02, -1.012938131054282e-01, 1.481771433134816e-01, 2.741694091041582e-01, -2.023901803642790e-02, 5.878908508299044e-03, -2.951054208766423e-03, -2.346105925480015e-03,  2.781675130882899e-03, 3.850577452922804e-04, 2.297450338498584e-02, -4.755489093442435e-02,  8.960657466984062e-02, -1.368014082646393e-02, -2.023901803643337e-02, 2.127196609432764e-01,  7.884507694119000e-02, 6.217031134326555e-02, 6.082231523427767e-02, -2.582029361848511e-02,  7.227896704301873e-03, -5.813463742383916e-02, 1.140888291856140e-01, 6.372260314136601e-03, -4.959303967124516e-02, -2.041431987332066e-02, -1.167339263149599e-03, 6.217031134325237e-02,  1.440762318793035e-01, 7.450840640103493e-02, -1.319374636865608e-02, -2.998159606759434e-02, -7.366005105793327e-04, 1.030389221032142e-01, 3.641083885061675e-02, -1.120177880228459e-01, -8.980006840607194e-02, 2.934134201481192e-03, 6.082231523428121e-02, 7.450840640105369e-02,  1.799259229623879e-01, -7.389152371379230e-02, 9.550908079831810e-03, -1.105134229418945e-02,  1.180239165849273e-01, -3.189203251861993e-02, -5.010973575337814e-03, -2.503262550023855e-02, -2.586651943392674e-03, -2.582029361849950e-02, -1.319374636867442e-02, -7.389152371380220e-02,  1.527513961843042e-01, -5.035428958488210e-02, 2.131241575847860e-02, -4.997478518463368e-02,  3.686222845036032e-02, -5.509035300863929e-02, -2.833339048704352e-02, -3.908077294498344e-03,  7.227896704312961e-03, -2.998159606758751e-02, 9.550908079840315e-03, -5.035428958488706e-02,  1.464524760332621e-01, 6.938748043779704e-03, -1.844510937789231e-02, 3.639948373375270e-03,  1.157379375907579e-01, 4.412675815115287e-02, -9.130247423228518e-03, -5.813463742387757e-02, -7.366005106208377e-04, -1.105134229421142e-02, 2.131241575847629e-02, 6.938748043796019e-03,  2.107729091852714e-01, -1.577704839081306e-01, 8.060879028453928e-02, 1.321180957677317e-02, -9.573183762590790e-02, 9.504345663860995e-03, 1.140888291856297e-01, 1.030389221032548e-01,  1.180239165849295e-01, -4.997478518461354e-02, -1.844510937791372e-02, -1.577704839080881e-01,  3.906561630008243e-01, -6.597778274668570e-02, -1.070148475229292e-01, 1.999881572477829e-02, -6.107254858358883e-03, 6.372260314098209e-03, 3.641083885058187e-02, -3.189203251865190e-02,  3.686222845037225e-02, 3.639948373380766e-03, 8.060879028456810e-02, -6.597778274674980e-02,  2.693260198350797e-01, -7.670134449607025e-02, -1.256911233125002e-01, 4.504689913986815e-02, -4.959303967121158e-02, -1.120177880228315e-01, -5.010973575302887e-03, -5.509035300866566e-02,  1.157379375907633e-01, 1.321180957670902e-02, -1.070148475228531e-01, -7.670134449611166e-02,  3.741411694324744e-01, 1.338792567087503e-01, -3.377206674249771e-02, -2.041431987326260e-02, -8.980006840603588e-02, -2.503262550019719e-02, -2.833339048706384e-02, 4.412675815114740e-02, -9.573183762595992e-02, 1.999881572486528e-02, -1.256911233125162e-01, 1.338792567087111e-01,  2.634451507138137e-01, -1.684745947722385e-02, -1.167339263156333e-03, 2.934134201478414e-03, -2.586651943397977e-03, -3.908077294494165e-03, -9.130247423230747e-03, 9.504345663867594e-03, -6.107254858366912e-03, 4.504689913987148e-02, -3.377206674249831e-02, -1.684745947722823e-02,  1.566202611399500e-01, 7.915118781589095e-02, 6.233740724459259e-02, 6.482165612965121e-02, -3.010220346341680e-02, 4.074146749083670e-03, -7.228576480717061e-02, 1.312003094352456e-01, -7.519680365112214e-03, -5.762943418783051e-02,
                  -1.231030707750911e-02, -2.956952790501847e-03,  6.233740724458192e-02, 1.322671411043326e-01, 8.645837964131108e-02, -2.600390970703586e-02, -3.110871401876582e-02, -1.873628432760631e-02, 1.378413050135795e-01, 1.191488107494714e-02, -1.190149327268250e-01, -8.522208703179270e-02, -1.254190735895348e-02, 6.482165612965346e-02,  8.645837964132758e-02, 1.659832980502135e-01, -7.163878879951270e-02, -8.640872143759738e-03, -2.397343534247874e-02, 1.532390351080777e-01, -3.724827560608520e-02, -5.202280954409259e-02, -4.439746012250072e-02, -1.756990987847567e-02, -3.010220346342928e-02, -2.600390970705221e-02, -7.163878879952265e-02, 1.382572131130904e-01, -3.312310808478803e-02, 3.064093568377319e-02, -7.593778215392974e-02, 3.939576730729226e-02, -1.992637656709249e-02, -1.432631681491491e-02,  4.277326212722293e-03, 4.074146749090493e-03, -3.110871401876208e-02, -8.640872143754490e-03, -3.312310808479101e-02, 1.300460509939250e-01, 1.315782536711428e-02, -4.259785189801090e-02,  1.850379791758631e-02, 1.097567748381745e-01, 4.499054889596873e-02, 2.814742982864419e-03, -7.228576480721134e-02, -1.873628432764597e-02, -2.397343534250838e-02, 3.064093568377745e-02,  1.315782536712380e-02, 2.187020792138230e-01, -1.836535457776415e-01, 8.400598049028749e-02,  3.805185043220503e-02, -9.153394459277631e-02, 4.651372563554924e-03, 1.312003094352665e-01,  1.378413050136193e-01, 1.532390351080901e-01, -7.593778215391762e-02, -4.259785189802306e-02, -1.836535457775998e-01, 4.708542061243732e-01, -9.041064061237297e-02, -1.775837458459035e-01, -1.416045538633922e-02, -3.127513966267502e-02, -7.519680365152041e-03, 1.191488107491182e-02, -3.724827560612421e-02, 3.939576730730872e-02, 1.850379791758932e-02, 8.400598049031574e-02, -9.041064061243934e-02, 2.517510458406271e-01, -3.686733375266981e-02, -9.974643207203536e-02,  4.533588543160323e-02, -5.762943418780638e-02, -1.190149327268153e-01, -5.202280954406507e-02, -1.992637656711333e-02, 1.097567748381767e-01, 3.805185043215460e-02, -1.775837458458431e-01, -3.686733375270641e-02, 3.681302902701662e-01, 1.349885020161586e-01, -8.214669998812908e-03, -1.231030707745430e-02, -8.522208703175926e-02, -4.439746012245792e-02, -1.432631681493666e-02,  4.499054889596479e-02, -9.153394459282255e-02, -1.416045538625861e-02, -9.974643207205017e-02,  1.349885020161261e-01, 2.564741915661607e-01, 7.341061097110618e-03, -2.956952790511558e-03, -1.254190735895831e-02, -1.756990987848528e-02, 4.277326212728989e-03, 2.814742982861716e-03,  4.651372563565679e-03, -3.127513966268963e-02, 4.533588543160834e-02, -8.214669998812180e-03,  7.341061097105018e-03, 1.544513327821151e-01, 6.029993878322461e-02, 4.079491813593259e-02,  4.263055959584169e-02, -1.561195140179050e-02, 2.206108417193948e-03, -7.200686564808284e-02,  1.013065453722817e-01, -3.066517477706996e-02, -2.834482119730742e-02, 1.151429932854253e-02, -1.265689672061148e-02, 4.079491813591950e-02, 7.337587037432006e-02, 4.751895148458753e-02, -1.512332287892633e-02, -1.051972188268189e-02, -3.293762967631041e-02, 8.705841956901533e-02, -1.732604488013207e-02, -4.228653822848852e-02, -1.809789011627698e-02, -1.353225983434616e-02,  4.263055959585210e-02, 4.751895148461319e-02, 1.265485340861700e-01, -4.819617921365004e-02,  9.642238762272220e-05, -2.734526685143091e-02, 1.050670253785065e-01, -5.328006334812711e-02, -5.059880776587024e-03, -2.745101017559774e-03, -1.952604631764567e-02, -1.561195140181033e-02, -1.512332287895023e-02, -4.819617921366237e-02, 9.727195693355956e-02, -1.853329586088249e-02,  2.391397642055701e-02, -4.984946773842716e-02, 3.355699484296215e-02, -1.347282808524600e-02, -1.230642031946300e-02, 5.472099151893607e-03, 2.206108417207723e-03, -1.051972188267279e-02,  9.642238763708268e-05, -1.853329586089125e-02, 9.387148704231768e-02, 1.585887131284051e-02, -1.358767683456746e-02, 7.997327725765807e-03, 5.496938726122347e-02, 1.615776451312439e-02, -1.113581814292797e-02, -7.200686564811240e-02, -3.293762967634532e-02, -2.734526685142444e-02,  2.391397642053468e-02, 1.585887131285969e-02, 2.728185922224905e-01, -2.067811694919798e-01,  1.206112323477494e-01, 2.891343133253782e-02, -1.274828877071630e-01, 3.779808771297322e-03,  1.013065453722943e-01, 8.705841956905505e-02, 1.050670253784797e-01, -4.984946773838424e-02, -1.358767683459668e-02, -2.067811694919593e-01, 3.675887170111033e-01, -1.034472012109119e-01, -7.065345219208102e-02, 6.751150145679750e-02, -8.498477621367144e-03, -3.066517477710650e-02, -1.732604488016814e-02, -5.328006334814851e-02, 3.355699484296504e-02, 7.997327725776058e-03,  1.206112323477951e-01, -1.034472012109861e-01, 1.963242807927235e-01, -2.149287499672274e-02, -9.572188179119789e-02, 3.852753257855494e-03, -2.834482119727118e-02, -4.228653822847041e-02, -5.059880776536812e-03, -1.347282808528262e-02, 5.496938726122872e-02, 2.891343133246912e-02, -7.065345219198620e-02, -2.149287499677908e-02, 1.991559323019309e-01, 4.441880896930762e-02, -1.662935913078489e-02, 1.151429932859554e-02, -1.809789011624310e-02, -2.745101017528283e-03, -1.230642031947640e-02, 1.615776451311786e-02, -1.274828877072125e-01, 6.751150145687584e-02, -9.572188179120679e-02, 4.441880896926612e-02, 1.947284837268420e-01, 4.711645505975742e-03, -1.265689672061973e-02, -1.353225983434968e-02, -1.952604631764991e-02, 5.472099151896778e-03, -1.113581814293072e-02, 3.779808771310165e-03, -8.498477621379268e-03, 3.852753257857330e-03, -1.662935913078135e-02, 4.711645505968894e-03, 1.062032819862684e-01, 7.180678375094193e-02,  4.536152854658289e-02, 4.124778858363513e-02, -1.634472261845873e-02, 6.778870459954496e-03, -7.991022391305840e-02, 9.975371651844014e-02, -1.480656660468040e-02, -3.224968297859077e-02,  8.830436465680953e-03, -6.023554384126243e-03, 4.536152854656975e-02, 8.947673319223709e-02,  4.883850791874277e-02, -1.024554370713608e-02, -1.474764549949513e-02, 1.942179796509298e-04,  6.930037747442964e-02, 2.158875356378528e-02, -6.921046841078574e-02, -6.259685810344713e-02, -1.264701529237801e-02, 4.124778858364706e-02, 4.883850791876887e-02, 1.294872602218096e-01, -5.362032705924439e-02, 3.322156353450076e-03, -3.955425825830154e-03, 8.289849954486676e-02, -3.635380768927119e-02, 2.576631753256708e-03, -2.125970259624195e-02, -2.180443510157943e-02, -1.634472261847979e-02, -1.024554370716040e-02, -5.362032705925709e-02, 1.048268400437008e-01, -2.657979256739235e-02, 2.018963444747276e-02, -3.980095230584275e-02, 2.835735499638295e-02, -2.870622515832207e-02, -1.333006592408401e-02, 2.131388532145129e-03, 6.778870459968635e-03, -1.474764549948646e-02, 3.322156353464101e-03, -2.657979256740106e-02, 9.763502332171542e-02, -1.807313946464196e-02, 1.088463802203004e-02, -1.202705098338784e-02, 6.586567952022188e-02,  4.313874681619144e-02, -9.295259283228923e-03, -7.991022391308890e-02, 1.942179796112099e-04, -3.955425825832336e-03, 2.018963444745418e-02, -1.807313946461631e-02, 3.036344496622769e-01, -1.710649113803310e-01, 1.293241899656336e-01, -3.164417360692750e-02, -1.856650509302118e-01, -2.359593882098162e-03, 9.975371651845460e-02, 6.930037747447429e-02, 8.289849954484826e-02, -3.980095230580323e-02, 1.088463802199599e-02, -1.710649113802983e-01, 2.897189215504852e-01, -7.057573745070393e-02,
                  -3.266395609202632e-02, 6.254975469872842e-02, -1.069863664292045e-02, -1.480656660471917e-02, 2.158875356374442e-02, -3.635380768930312e-02, 2.835735499639106e-02, -1.202705098337228e-02, 1.293241899656740e-01, -7.057573745078582e-02, 2.322543546268551e-01, -8.489871362675225e-02, -1.436582791209620e-01, 2.470246867377858e-02, -3.224968297855348e-02, -6.921046841076768e-02, 2.576631753307262e-03, -2.870622515835898e-02, 6.586567952022636e-02, -3.164417360700635e-02, -3.266395609192273e-02, -8.489871362681667e-02, 2.461449838369645e-01,  1.158932135329678e-01, -1.654847466963343e-02, 8.830436465735302e-03, -6.259685810340970e-02, -2.125970259620282e-02, -1.333006592410119e-02, 4.313874681617989e-02, -1.856650509302678e-01,  6.254975469882201e-02, -1.436582791209809e-01, 1.158932135329194e-01, 2.793326526015258e-01,  5.179309167261120e-03, -6.023554384135286e-03, -1.264701529238293e-02, -2.180443510158791e-02,  2.131388532150703e-03, -9.295259283230741e-03, -2.359593882089413e-03, -1.069863664293306e-02,  2.470246867377889e-02, -1.654847466962854e-02, 5.179309167258600e-03, 1.170417921785367e-01,  8.151976959448776e-02, 5.552433221616531e-02, 4.985074343000902e-02, -1.820467809521597e-02,  4.865093369009681e-03, -8.458760386698602e-02, 1.136667245067876e-01, -1.286910419067104e-02, -5.080178004118413e-02, -3.430215084528583e-03, -9.034951873023096e-03, 5.552433221615199e-02,  9.730609999542164e-02, 5.752226936817545e-02, -1.435630736671011e-02, -1.434452824205424e-02, -1.845877647225312e-02, 8.971149441393403e-02, 6.585727510454648e-03, -7.564052209101366e-02, -5.747745807519288e-02, -1.800621448851866e-02, 4.985074343002163e-02, 5.752226936820266e-02,  1.491425304206482e-01, -6.373909550700108e-02, 4.533225040650334e-03, -1.361641577393216e-02,  1.009423752081495e-01, -4.235064900694302e-02, -3.666104519828278e-03, -2.682252552457886e-02, -2.504540771097547e-02, -1.820467809523766e-02, -1.435630736673514e-02, -6.373909550701490e-02,  1.118086426403502e-01, -2.847619499984452e-02, 1.841908212258101e-02, -4.551855189807393e-02,  1.998875238896664e-02, -1.834353620345256e-02, -8.569615779364832e-04, 2.638981319008164e-03,  4.865093369024244e-03, -1.434452824204469e-02, 4.533225040665410e-03, -2.847619499985321e-02,  1.001488755227271e-01, 1.599625090059480e-03, 5.412834468439985e-04, 1.117789162932812e-02,  5.497148836123068e-02, 2.447657011891331e-02, -6.617770051838986e-03, -8.458760386701664e-02, -1.845877647229228e-02, -1.361641577392678e-02, 1.841908212255727e-02, 1.599625090083364e-03,  2.925842913990256e-01, -1.833834830894805e-01, 1.309605721697385e-01, 1.150959208779152e-02, -1.424761803763894e-01, 9.872412163379500e-03, 1.136667245068021e-01, 8.971149441397805e-02,  1.009423752081235e-01, -4.551855189802916e-02, 5.412834468107638e-04, -1.833834830894540e-01,  3.132275488327307e-01, -8.776043874441811e-02, -5.874616474840760e-02, 4.047473855518853e-02, -2.157276916937838e-02, -1.286910419071039e-02, 6.585727510413646e-03, -4.235064900696957e-02,  1.998875238897036e-02, 1.117789162934230e-02, 1.309605721697866e-01, -8.776043874450241e-02,  2.421035384028955e-01, -5.242995895035300e-02, -1.115702122051751e-01, 3.889907821252486e-02, -5.080178004114590e-02, -7.564052209099435e-02, -3.666104519775365e-03, -1.834353620348971e-02,  5.497148836123619e-02, 1.150959208771268e-02, -5.874616474830332e-02, -5.242995895041783e-02,  2.247111774651911e-01, 7.873128392660701e-02, -1.435444204310261e-02, -3.430215084473722e-03, -5.747745807515550e-02, -2.682252552454367e-02, -8.569615779503627e-04, 2.447657011890387e-02, -1.424761803764450e-01, 4.047473855527745e-02, -1.115702122051887e-01, 7.873128392655919e-02,  2.229031560676283e-01, 2.421648654329044e-03, -9.034951873032292e-03, -1.800621448852345e-02, -2.504540771098213e-02, 2.638981319012522e-03, -6.617770051841075e-03, 9.872412163392104e-03, -2.157276916939302e-02, 3.889907821252640e-02, -1.435444204309762e-02, 2.421648654323654e-03,  1.201417692591440e-01, 8.866902042472589e-02, 6.776363429959428e-02, 7.253903503525334e-02, -3.109196000433137e-02, 4.182214047614752e-03, -8.417025933936019e-02, 1.316636905252350e-01, -3.101979929284263e-02, -5.778419360389092e-02, -1.188019879182483e-02, -2.076562292635449e-02,  6.776363429958092e-02, 1.010410854585899e-01, 7.258789900068488e-02, -2.441576984955044e-02, -9.432131827966440e-03, -3.762366921312024e-02, 1.099097886140499e-01, -1.336442457224784e-02, -7.509627347493571e-02, -4.772889503000497e-02, -2.398694031973857e-02, 7.253903503526708e-02,  7.258789900071175e-02, 1.881389914770610e-01, -7.983557161516319e-02, 5.297538993856518e-03, -2.702845457162629e-02, 1.296341608006866e-01, -5.530776668839563e-02, -1.668804219951507e-02, -3.454417934810972e-02, -2.724350226640559e-02, -3.109196000435394e-02, -2.441576984957538e-02, -7.983557161517664e-02, 1.228494738890125e-01, -2.936736308546739e-02, 2.436555944806306e-02, -5.883425284656953e-02, 1.740518505131010e-02, -2.848687277770920e-03, 8.241105664157863e-03, -4.740470734368018e-04, 4.182214047629726e-03, -9.432131827957334e-03, 5.297538993870072e-03, -2.936736308547536e-02, 9.905571032218671e-02, 1.191188831030088e-02, -3.149946646971280e-03,  1.217526952143257e-02, 4.501482450519646e-02, 1.670400210261658e-02, -4.340063153708508e-03, -8.417025933939115e-02, -3.762366921316018e-02, -2.702845457162344e-02, 2.436555944804071e-02,  1.191188831032629e-02, 2.403578006647200e-01, -1.664737895005366e-01, 9.910993026694921e-02,  4.259339377865044e-02, -8.672845546041852e-02, 1.986561238240263e-02, 1.316636905252502e-01,  1.099097886140949e-01, 1.296341608006648e-01, -5.883425284652717e-02, -3.149946647004586e-03, -1.664737895005079e-01, 3.240333803645861e-01, -6.982029925927097e-02, -8.974140405753184e-02,  4.419264755390064e-03, -2.715991913101059e-02, -3.101979929288304e-02, -1.336442457228953e-02, -5.530776668842614e-02, 1.740518505131653e-02, 1.217526952144659e-02, 9.910993026699484e-02, -6.982029925935479e-02, 1.968263381715005e-01, -2.268817193857966e-02, -6.596396590570906e-02,  2.788087040973855e-02, -5.778419360385161e-02, -7.509627347491749e-02, -1.668804219946431e-02, -2.848687277807653e-03, 4.501482450520242e-02, 4.259339377856922e-02, -8.974140405742796e-02, -2.268817193864429e-02, 2.114806403869727e-01, 5.533084663307276e-02, 4.659762271594113e-03, -1.188019879176927e-02, -4.772889502996761e-02, -3.454417934807355e-02, 8.241105664142980e-03,  1.670400210260634e-02, -8.672845546047478e-02, 4.419264755480242e-03, -6.596396590572452e-02,  5.533084663302317e-02, 1.792778890012046e-01, 8.988377289049977e-03, -2.076562292636394e-02, -2.398694031974369e-02, -2.724350226641477e-02, -4.740470734307298e-04, -4.340063153710943e-03,  1.986561238241459e-02, -2.715991913102566e-02, 2.788087040974116e-02, 4.659762271597917e-03,  8.988377289044563e-03, 1.167520052292264e-01, 1.064225194588110e-01, 8.606568405424259e-02,  8.744205132569716e-02, -2.938676848516011e-02, -6.952804835198181e-03, -1.090617337717296e-01,  1.750076853841718e-01, -3.973523957861839e-02, -8.716439287684516e-02, -1.190429908460009e-02, -2.483082724791528e-02, 8.606568405422918e-02, 1.333769962448216e-01, 9.913873900858011e-02, -2.761558134671105e-02, -1.377138376512205e-02, -5.450424102797564e-02, 1.596548898914298e-01,
                  -2.851822153902586e-02, -9.456273446056396e-02, -5.550814628091294e-02, -4.078765318912025e-02,  8.744205132571065e-02, 9.913873900860676e-02, 1.873535054582560e-01, -6.342651840865599e-02, -5.882825052322210e-03, -4.312874335870159e-02, 1.665751739079502e-01, -5.135564752100596e-02, -4.885088580528868e-02, -3.592199089519407e-02, -3.042922433522877e-02, -2.938676848518265e-02, -2.761558134673596e-02, -6.342651840866986e-02, 1.286865132174838e-01, -3.318114613074258e-02,  2.443044398354230e-02, -5.612065059109483e-02, 7.436115932348608e-03, -5.363821664440271e-04,  1.418868039152088e-03, -5.814587309261654e-03, -6.952804835183249e-03, -1.377138376511253e-02, -5.882825052309471e-03, -3.318114613074968e-02, 1.101344619077267e-01, 7.948098055927832e-03,  1.383727207230595e-03, 1.108759375914911e-03, 5.144034848460022e-02, 2.840081194922839e-02, -1.000786905704745e-02, -1.090617337717614e-01, -5.450424102801318e-02, -4.312874335870071e-02,  2.443044398352226e-02, 7.948098055948566e-03, 2.658339996039858e-01, -2.031020876292502e-01,  7.313124135438476e-02, 7.543657467265748e-02, -7.444319190081528e-02, 1.268267647594060e-02,  1.750076853841875e-01, 1.596548898914721e-01, 1.665751739079306e-01, -5.612065059105464e-02,  1.383727207201729e-03, -2.031020876292242e-01, 4.328402800116424e-01, -5.996657563388319e-02, -1.403798755809677e-01, -1.206001811796065e-02, -4.518400641303288e-02, -3.973523957865949e-02, -2.851822153906505e-02, -5.135564752103827e-02, 7.436115932357157e-03, 1.108759375923439e-03,  7.313124135442961e-02, -5.996657563396290e-02, 1.896600657962084e-01, -2.545878696935677e-03, -3.979724577947064e-02, 3.215713349088678e-02, -8.716439287680597e-02, -9.456273446054531e-02, -4.885088580523916e-02, -5.363821664796653e-04, 5.144034848460688e-02, 7.543657467258476e-02, -1.403798755808723e-01, -2.545878696990966e-03, 2.417259483086340e-01, 5.074372818684569e-02,  2.224267313531813e-02, -1.190429908454405e-02, -5.550814628087719e-02, -3.592199089515652e-02,  1.418868039135546e-03, 2.840081194922195e-02, -7.444319190086790e-02, -1.206001811787594e-02, -3.979724577948329e-02, 5.074372818680239e-02, 1.912764426547819e-01, 3.789963756442578e-02, -2.483082724792496e-02, -4.078765318912445e-02, -3.042922433523809e-02, -5.814587309255104e-03, -1.000786905705155e-02, 1.268267647595364e-02, -4.518400641304743e-02, 3.215713349089044e-02,  2.224267313531902e-02, 3.789963756441862e-02, 1.554301123728286e-01 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$lag_one_cov)),
               c(NULL )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$fixed_effects)),
               c(numeric(0) )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$n_iter)),
               c(11 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$Q)),
               c( 0.234481701593572067, 0.166290038022716580, 0.237448191399598107, -0.164398510602372150, 0.053887657389158079, -0.272426923593383163, 0.436330449328763303, -0.038173403372556494, -0.096456990554951744, 0.046045727662697822,  0.059410968438053977, 0.166290038022716580, 0.405987159685059662, 0.329737881040278236, -0.102362171752270970, -0.200538355540479662, 0.017436117503002304, 0.538035127010136671, 0.074006187431132076, -0.607294090275498633, -0.449945908806761552, -0.070496079558895558, 0.237448191399598107, 0.329737881040278236, 1.436573186727477580, -0.930353327157208532, 0.206376716345247513, -0.129577888745402103, 1.088704788982885141, -0.895837653399328415,  0.419584594285682622, 0.084558428671240110, -0.300362844056459743, -0.164398510602372150, -0.102362171752270970, -0.930353327157208532, 0.713727097855611281, -0.284262748388453113, 0.181168128941013806, -0.656942875482579458,  0.673851343194700636, -0.563397436756561065, -0.287532367827041391, 0.161996336364456650, 0.053887657389158079, -0.200538355540479662, 0.206376716345247513, -0.284262748388453113, 0.445167185352934536, 0.007230595499526832, -0.200828189094678200, -0.083222051465666247, 0.810810821724626818, 0.374121973998201462, 0.091121854962377652, -0.272426923593383163, 0.017436117503002304, -0.129577888745402103, 0.181168128941013806, 0.007230595499526832,  1.586254554974476427, -1.384505017364019652, 1.155982802205204774, -0.003167366021633500, -1.111390056672403315,  0.184783283429711426, 0.436330449328763303, 0.538035127010136671, 1.088704788982885141, -0.656942875482579458, -0.200828189094678200, -1.384505017364019652, 2.355682389234000418, -1.446916440271179383, -0.536652213387566857,  0.494879263947109571, -0.490407529529279407, -0.038173403372556494, 0.074006187431132076, -0.895837653399328415,  0.673851343194700636, -0.083222051465666247, 1.155982802205204774, -1.446916440271179383, 1.931935722853363568, -0.599811869743584358, -1.135782416851197407, 0.560041718173677983, -0.096456990554951744, -0.607294090275498633,  0.419584594285682622, -0.563397436756561065, 0.810810821724626818, -0.003167366021633500, -0.536652213387566857, -0.599811869743584358, 1.964339024967495018, 1.025887953503292094, 0.044564690384365974, 0.046045727662697822, -0.449945908806761552, 0.084558428671240110, -0.287532367827041391, 0.374121973998201462, -1.111390056672403315,  0.494879263947109571, -1.135782416851197407, 1.025887953503292094, 1.445367000794241630, -0.085797722035871535,  0.059410968438053977, -0.070496079558895558, -0.300362844056459743, 0.161996336364456650, 0.091121854962377652,  0.184783283429711426, -0.490407529529279407, 0.560041718173677983, 0.044564690384365974, -0.085797722035871535,  0.347110756442550905 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$Q_0)),
               c(1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$n_risk)),
               c(2007, 1936, 1882, 1818, 1767, 1656, 1543, 1364, 1254, 1103 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$times)),
               c( 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$risk_set)),
               c(NULL )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$data)),
               c(NULL )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$order)),
               c(1 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$F_)),
               c(1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$method)),
               c("UKF" )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$model)),
               c("exp_bin" )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$est_Q_0)),
               c(FALSE )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$LR)),
               c(1 )
               , tolerance = 1e-04)


  # tmp <- file("tmp.txt")
  # sink(tmp)
  suppressMessages(result_exp <- ddhazard(
    formula = survival::Surv(tstart, tstop, event) ~ . - id - tstart - tstop - event,
    data = sims$res,
    by = (by_ <- 1),
    Q_0 = diag(1, 11),
    Q = diag(1e-1, 11),
    control = list(est_Q_0 = F, eps = 10^-2, n_max = 10^3, method = "UKF",
                   debug = F, beta = 0, save_data = F, save_risk_set = F),
    max_T = 10,
    id = sims$res$id, order = 1,
    verbose = F,
    model = "exp_clip_time"))
  # sink()
  # close(tmp)

  # matplot(sims$betas, type = "l", lty = 1)
  # matplot(result_exp$state_vecs, type = "l", lty = 2, add = T)
  # get_expect_equal(result_exp, file = "tmp.txt", eps = 1e-4)

  expect_equal(unname(c(result_exp$state_vecs)),
               c(-4.96829405282980829, -4.97218787626814684, -4.69890301470080463, -4.53132134049340252, -4.52621517422552522, -4.36457769292643594, -4.44698175906288906, -4.42005252660851067, -4.38037463752706824, -4.20005942520948672, -3.90704938790220879, -1.78878763382033634, -1.77698392263970639, -2.18209058470032069, -2.07725612444599905, -1.70729235462211504, -1.79466382752894260, -1.79758130317842069, -2.10217942043570227, -2.18529250634468086, -2.06292051352156891, -1.94999044184049741, -0.18608507717204603, -0.13078736794849877, -1.02958443567604796, -0.48349968380247293, 0.65008608557536907, 0.23856479699055852, 0.51834637872098210, -0.65811443728023500, -1.57798327554322748, -0.79827283499693213, 0.74895687277232170, 0.30525906141803505, 0.28214114459441725,  0.58677727146351966, 0.29228660061540146, -0.20667572869345002, -0.08157832078843474, -0.07110202489750096,  0.40640351820476672, 0.87265221337344112, 0.51359822136257827, -0.26955819502751499, -0.48746618055204616, -0.49400336174141019, -0.49572029461355205, -0.14720110129441377, -0.06786304370614088, 0.55258949627775122,  0.77861022458443474, 0.95205781787566068, 1.17021660839018793, 1.21361927597768493, 1.29994166182304682,  0.39656223004122670, 0.41890397021705067, -0.59250448592701943, 0.52687423854559579, 0.91703036956649919,  1.97408038169435973, 2.74232301341550855, 2.50639179355112329, 2.44264351602918772, 1.99769054522592437,  2.13302534664174193, -0.86817731578734536, -0.83350511309336639, -1.25062529844459092, -2.18345671464711089, -1.40350203807979557, -3.44199885003930062, -4.58177950693437719, -5.66812420570431819, -6.39629032659168217, -5.73563761271334549, -5.08630392239134910, 0.18552048343627406, 0.17355655287216673, -0.94075119535183405,  1.08329080523038157, 1.53386414283911843, 3.65333122343202632, 3.35774529894387141, 3.33309763806488135,  4.28499034866227557, 2.68767265105968400, 1.48418511620306615, 0.69090910605267497, 0.68405651628198760,  1.51816863598024954, 1.68609885631251211, 1.64955172249907056, 2.08895152170156795, 2.72968285972460079,  2.95742910039770868, 3.01283369529838341, 3.93058058766898188, 5.11142017663907033, 2.16485715504296605,  2.13503515581510150, 4.34363358191370352, 2.55273188296189213, 1.71877921477770523, 0.06338137217568252, -0.74051478624418710, -0.28806996016382802, -0.48016407459615007, 0.90720508270254374, 1.80195092898352494,  2.85078909154983240, 2.82597605236520488, 3.68819179107644368, 4.08313364938431711, 3.69944678625011214,  4.03099468623271751, 3.08394322644877050, 3.23281549536706914, 3.57949151091281870, 3.23551448502083439,  3.18711602561397012 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$state_vars)),
               c( 1.096783306865856e-01, 1.223720108221427e-02, 3.311498418068894e-02, -3.098732548483929e-02,  3.205315563981391e-02, -7.945246909673387e-03, -1.086746542706959e-02, 1.750248339594519e-02,  6.142435680444020e-02, 3.007680993848401e-02, 3.357688699734641e-02, 1.223720108221636e-02,  1.721244452115538e-01, 5.754858444803848e-02, -1.145549226256705e-02, 1.111700183348185e-02,  2.115744709526676e-03, 6.105281991358084e-02, 1.824360745893039e-02, -1.020699745768905e-03, -3.443037266514935e-02, -4.232448092923687e-02, 3.311498418068561e-02, 5.754858444803865e-02,  4.660034472477186e-01, -1.634223721299321e-01, 6.710049961794636e-05, 1.132483696442696e-01,  1.399616566483372e-01, -5.148427755842601e-02, 1.048383414446241e-01, -5.306382994163256e-02, -2.984680094478277e-02, -3.098732548484201e-02, -1.145549226256707e-02, -1.634223721299318e-01,  2.448679290706081e-01, -1.310187729515732e-02, -5.680402333471588e-02, -6.282087708970752e-02,  1.828104426963068e-02, -3.827042652559009e-02, 6.366440345966590e-03, -2.323363677145352e-02,  3.205315563981163e-02, 1.111700183348178e-02, 6.710049961785207e-05, -1.310187729515747e-02,  1.905054525393444e-01, 3.186439172619112e-02, -6.497454205925815e-02, 5.265411519788481e-02,  8.633298399643118e-02, -7.821969327029830e-03, -2.178014056907350e-02, -7.945246909675584e-03,  2.115744709526464e-03, 1.132483696442696e-01, -5.680402333471594e-02, 3.186439172619109e-02,  2.592588723426820e-01, -9.899936388964248e-02, 7.637875189419335e-02, 3.915738370310428e-02, -1.706135027420279e-01, -3.090030027944633e-02, -1.086746542706971e-02, 6.105281991358090e-02,  1.399616566483374e-01, -6.282087708970767e-02, -6.497454205925834e-02, -9.899936388964276e-02,  5.143404326409446e-01, -7.754461398261757e-02, -1.198027724445888e-01, 6.637019755780921e-02, -2.457284653485116e-03, 1.750248339594478e-02, 1.824360745893035e-02, -5.148427755842597e-02,  1.828104426963072e-02, 5.265411519788502e-02, 7.637875189419331e-02, -7.754461398261733e-02,  5.106900951109334e-01, -5.623152365741792e-02, -2.123654438748633e-01, 1.167903996038298e-01,  6.142435680443928e-02, -1.020699745768788e-03, 1.048383414446244e-01, -3.827042652558998e-02,  8.633298399643138e-02, 3.915738370310426e-02, -1.198027724445888e-01, -5.623152365741797e-02,  4.291996186039631e-01, 9.459482657427673e-02, -4.081922761056747e-03, 3.007680993848498e-02, -3.443037266514967e-02, -5.306382994163239e-02, 6.366440345966819e-03, -7.821969327029403e-03, -1.706135027420279e-01, 6.637019755780936e-02, -2.123654438748636e-01, 9.459482657427666e-02,  4.935236891087682e-01, 8.993621446321606e-02, 3.357688699734679e-02, -4.232448092923660e-02, -2.984680094478220e-02, -2.323363677145298e-02, -2.178014056907326e-02, -3.090030027944635e-02, -2.457284653485077e-03, 1.167903996038300e-01, -4.081922761056652e-03, 8.993621446321595e-02,  4.261081263594825e-01, 8.487783993947629e-02, 1.532180542842222e-02, 3.817856338390612e-02, -2.903152582743428e-02, 2.592523378897296e-02, -1.417445057480369e-03, -3.540308498085065e-04,  9.008705278239892e-03, 4.818073299600709e-02, 1.374574106257395e-02, 1.285702978592700e-02,  1.532180542842389e-02, 1.462164053410551e-01, 4.925089920821783e-02, -1.238303173558391e-02,  8.919041720479224e-03, 3.723499351241374e-03, 4.725582662790179e-02, 1.348636383093897e-02,  3.851721999858813e-03, -2.606356099650800e-02, -3.162472001608044e-02, 3.817856338389961e-02,  4.925089920821822e-02, 3.664795461437427e-01, -1.356791736205928e-01, 6.532642833262151e-03,  7.916537206836807e-02, 1.088087524983128e-01, -4.609405583877440e-02, 1.064585072056873e-01, -1.786522941437037e-02, -1.812026042401353e-02, -2.903152582743605e-02, -1.238303173558380e-02, -1.356791736205923e-01, 2.114982509531455e-01, -1.347961236544184e-02, -4.207766959108700e-02, -5.703485778786205e-02, 2.100790045131769e-02, -4.272121091178598e-02, -5.923654632063615e-03, -1.980623708674559e-02, 2.592523378896947e-02, 8.919041720479224e-03, 6.532642833262235e-03, -1.347961236544189e-02, 1.617171721590609e-01, 3.301934458428757e-02, -5.729023011438714e-02,  4.863654532117784e-02, 7.430543901288159e-02, -1.730827423049437e-02, -1.967652226763875e-02, -1.417445057486234e-03, 3.723499351241357e-03, 7.916537206836756e-02, -4.207766959108718e-02,  3.301934458428713e-02, 1.916877641120287e-01, -8.015848039253264e-02, 6.297353385247337e-02,  3.791544282823243e-02, -1.323454635117059e-01, -2.302601024239723e-02, -3.540308498067683e-04,  4.725582662790196e-02, 1.088087524983131e-01, -5.703485778786273e-02, -5.729023011438773e-02, -8.015848039253358e-02, 4.155526953888083e-01, -8.610243607216728e-02, -8.038191137398888e-02,  8.960934070499477e-02, 9.437458327630231e-03, 9.008705278236263e-03, 1.348636383093918e-02, -4.609405583877491e-02, 2.100790045131780e-02, 4.863654532117764e-02, 6.297353385247301e-02, -8.610243607216583e-02, 3.925788736419374e-01, -3.757596825755817e-02, -1.887469682759591e-01,  8.307169568879802e-02, 4.818073299600508e-02, 3.851721999858858e-03, 1.064585072056890e-01, -4.272121091178583e-02, 7.430543901288232e-02, 3.791544282823317e-02, -8.038191137398865e-02, -3.757596825755827e-02, 3.607034668224696e-01, 7.060050407568377e-02, -1.008516843937404e-02,  1.374574106257982e-02, -2.606356099650868e-02, -1.786522941436846e-02, -5.923654632063168e-03, -1.730827423049305e-02, -1.323454635117051e-01, 8.960934070499420e-02, -1.887469682759598e-01,  7.060050407568455e-02, 4.356907035832503e-01, 7.021437939762007e-02, 1.285702978592836e-02, -3.162472001608012e-02, -1.812026042401173e-02, -1.980623708674489e-02, -1.967652226763800e-02, -2.302601024239664e-02, 9.437458327630394e-03, 8.307169568879846e-02, -1.008516843937338e-02,  7.021437939761946e-02, 3.563922322895015e-01, 6.557337804023924e-02, 1.604572549026172e-02,  4.311046376400050e-02, -2.763033320378344e-02, 2.245881842151116e-02, 4.483340576441558e-03, -2.832479892480311e-04, 4.704452999003208e-03, 4.609833322244213e-02, 4.424654000084495e-03,  2.361623489364723e-03, 1.604572549026376e-02, 1.320671703796669e-01, 6.869029936297716e-02, -2.275258605726651e-02, 4.864527594630002e-03, 1.158088940030149e-02, 6.313130835601061e-02,  1.092734414682666e-02, -3.120202984663704e-03, -3.593990885333182e-02, -3.487095652178077e-02,  4.311046376399416e-02, 6.869029936297780e-02, 4.287603997096411e-01, -1.713471208220324e-01,  7.005409422814119e-03, 9.265462724404396e-02, 1.633382579716098e-01, -8.766526132093312e-02,  1.248656867860608e-01, -1.996456505985083e-02, -4.696870659521834e-02, -2.763033320378516e-02, -2.275258605726645e-02, -1.713471208220317e-01, 2.058719364389435e-01, -1.130635099770515e-02, -4.676072197303216e-02, -8.287968807838104e-02, 4.210269100606902e-02, -5.377359293771621e-02, -4.712390387690611e-03, -3.571826711626178e-03, 2.245881842150806e-02, 4.864527594630120e-03,  7.005409422814433e-03, -1.130635099770532e-02, 1.428135672882907e-01, 4.613811667289955e-02, -8.056304543985056e-02, 6.114712803630206e-02, 7.705235475054770e-02, -3.449014344381332e-02, -1.441638090013263e-02, 4.483340576437624e-03, 1.158088940030187e-02, 9.265462724404222e-02, -4.676072197303193e-02, 4.613811667289884e-02, 1.970115636545755e-01, -1.013333403238923e-01,  1.087628170917496e-01, 4.273925273550980e-02, -1.768496146805942e-01,
                  -2.678880228619648e-02, -2.832479892481490e-04, 6.313130835601086e-02, 1.633382579716109e-01, -8.287968807838200e-02, -8.056304543985125e-02, -1.013333403238925e-01, 5.088789041522516e-01, -1.600546035369150e-01, -1.083123524136617e-01, 1.260155703267400e-01, -1.211063539463455e-02, 4.704452999001706e-03,  1.092734414682793e-02, -8.766526132093479e-02, 4.210269100606982e-02, 6.114712803630208e-02,  1.087628170917513e-01, -1.600546035369160e-01, 5.320111970391529e-01, -8.916270513577065e-02, -3.144599555484638e-01, 1.103547649475920e-01, 4.609833322244011e-02, -3.120202984664432e-03,  1.248656867860626e-01, -5.377359293771630e-02, 7.705235475054842e-02, 4.273925273550918e-02, -1.083123524136602e-01, -8.916270513577196e-02, 3.903326090298950e-01, 1.070302204592794e-01, -1.448398556086102e-02, 4.424654000087354e-03, -3.593990885333393e-02, -1.996456505984739e-02, -4.712390387690926e-03, -3.449014344381193e-02, -1.768496146805961e-01, 1.260155703267425e-01, -3.144599555484640e-01, 1.070302204592782e-01, 5.333329418488041e-01, 6.163444985294470e-02,  2.361623489365548e-03, -3.487095652178084e-02, -4.696870659521633e-02, -3.571826711625580e-03, -1.441638090013182e-02, -2.678880228619694e-02, -1.211063539463342e-02, 1.103547649475918e-01, -1.448398556086065e-02, 6.163444985294499e-02, 3.429806943412921e-01, 4.591673803459820e-02,  2.016990487146565e-02, 3.732876871658453e-02, -2.006155131853705e-02, 1.693716905775319e-02,  1.343809988975871e-02, 3.789191755960911e-03, 1.979534187710711e-02, 1.703080954552545e-02, -3.205249168047454e-02, -1.727120189030928e-02, 2.016990487146884e-02, 1.009376418311050e-01,  3.858304354534446e-02, -1.104491135844068e-02, 2.543962165763365e-03, -4.522702861208087e-03,  4.078189646226595e-02, -6.323623342881701e-03, 5.887007317208232e-03, 3.553877740687120e-03, -1.278774219579156e-02, 3.732876871657786e-02, 3.858304354534670e-02, 2.250175862937665e-01, -7.829772213693856e-02, 1.253374270918159e-02, 3.628238608021851e-02, 6.162187773148070e-02, -5.037581321300730e-02, 8.250520660252043e-02, 2.178977885835711e-02, -9.505471519957610e-03, -2.006155131853831e-02, -1.104491135844120e-02, -7.829772213693806e-02, 1.357553539195484e-01, -1.238093480159176e-02, -2.064766687011100e-02, -3.090454693441083e-02, 2.670795297810048e-02, -3.402390098598555e-02, -1.637542243429821e-02, -8.299807675241438e-03, 1.693716905775287e-02,  2.543962165762807e-03, 1.253374270917953e-02, -1.238093480159121e-02, 1.006244190614748e-01,  2.905122823529395e-02, -3.351289601333778e-02, 2.962191440202108e-02, 4.552655215942651e-02, -1.281704146120840e-02, -6.028779264601221e-03, 1.343809988976594e-02, -4.522702861209831e-03,  3.628238608020133e-02, -2.064766687010500e-02, 2.905122823529588e-02, 1.255111886858093e-01, -7.637415940178723e-02, 5.805139852893512e-02, 3.447376858197736e-02, -7.631045412458114e-02,  1.046568967845578e-02, 3.789191755948754e-03, 4.078189646227097e-02, 6.162187773149670e-02, -3.090454693441683e-02, -3.351289601333782e-02, -7.637415940176673e-02, 2.896669626964565e-01, -9.486961375144844e-02, -4.900940993262574e-02, 9.293706278950901e-02, -1.051611542873883e-02,  1.979534187712182e-02, -6.323623342882547e-03, -5.037581321303099e-02, 2.670795297811000e-02,  2.962191440202725e-02, 5.805139852894329e-02, -9.486961375148550e-02, 3.207451877089164e-01, -3.438260682738341e-02, -1.451529975301265e-01, 1.033238863731206e-01, 1.703080954552262e-02,  5.887007317205319e-03, 8.250520660252397e-02, -3.402390098598705e-02, 4.552655215942354e-02,  3.447376858196136e-02, -4.900940993260630e-02, -3.438260682740614e-02, 2.378667275680483e-01,  3.197543013838508e-02, -2.277541169307724e-02, -3.205249168049033e-02, 3.553877740686193e-03,  2.178977885838578e-02, -1.637542243430857e-02, -1.281704146121571e-02, -7.631045412460052e-02,  9.293706278956322e-02, -1.451529975301405e-01, 3.197543013835950e-02, 2.579840542314212e-01, -2.052357039849367e-02, -1.727120189030956e-02, -1.278774219579270e-02, -9.505471519954203e-03, -8.299807675241501e-03, -6.028779264602848e-03, 1.046568967844438e-02, -1.051611542872443e-02,  1.033238863731058e-01, -2.277541169307771e-02, -2.052357039847724e-02, 2.289348735463895e-01,  3.595290478345478e-02, 1.805495269561065e-02, 2.706702918036426e-02, -1.529142263825263e-02,  8.636171427531446e-03, 1.542510132702314e-03, 1.885713322799638e-02, -2.586934901297426e-03,  7.033551420704953e-03, -1.520482321693056e-02, -2.352027155671402e-02, 1.805495269561401e-02,  8.741130033112426e-02, 4.395544035775752e-02, -1.334290653823901e-02, 6.208007281423728e-03,  7.881433496677411e-03, 2.728270322317438e-02, 1.085359754711658e-02, 8.490739181071585e-03, -1.613566503163507e-02, -1.159630157634586e-02, 2.706702918035834e-02, 4.395544035775927e-02,  2.172497194716645e-01, -7.590290844394590e-02, 1.211626713765201e-02, 4.986070084898706e-02,  5.870028613997261e-02, -2.615001802871268e-02, 6.954945308871982e-02, -1.371860432038661e-02, -1.797355016420450e-02, -1.529142263825418e-02, -1.334290653823930e-02, -7.590290844394502e-02,  1.218506059777722e-01, -1.211389874561347e-02, -2.687778200885831e-02, -2.721211458374934e-02,  1.396970417886386e-02, -3.124848972907468e-02, -9.156648538682808e-04, -5.685710510174385e-03,  8.636171427531345e-03, 6.208007281422850e-03, 1.211626713764912e-02, -1.211389874561262e-02,  8.313512314125554e-02, 2.641645196406604e-02, -2.335163610035553e-02, 2.341087951648247e-02,  3.031927588426914e-02, -2.239774258560858e-02, -2.019381301614392e-02, 1.542510132710419e-03,  7.881433496674996e-03, 4.986070084896994e-02, -2.687778200885260e-02, 2.641645196406816e-02,  7.846210736849792e-02, -2.801347974881908e-02, 1.061630504173332e-02, 3.019385578488992e-02, -3.625480007752883e-02, -1.193182499019647e-02, 1.885713322798379e-02, 2.728270322318004e-02,  5.870028613998973e-02, -2.721211458375579e-02, -2.335163610035489e-02, -2.801347974879736e-02,  2.256493943278029e-01, -2.389036284681045e-02, -3.197402769979200e-02, 4.420490880038058e-02,  2.747540068931205e-02, -2.586934901281893e-03, 1.085359754711520e-02, -2.615001802873566e-02,  1.396970417887290e-02, 2.341087951648843e-02, 1.061630504174266e-02, -2.389036284684936e-02,  2.129261857401528e-01, -4.102966943800348e-02, -9.042803454924578e-02, 4.155561554102531e-02,  7.033551420702352e-03, 8.490739181068196e-03, 6.954945308872162e-02, -3.124848972907551e-02,  3.031927588426597e-02, 3.019385578487207e-02, -3.197402769977219e-02, -4.102966943802766e-02,  1.983226787822267e-01, 2.032925488890256e-02, -3.606942019904616e-02, -1.520482321694768e-02, -1.613566503163522e-02, -1.371860432035907e-02, -9.156648538778257e-04, -2.239774258561652e-02, -3.625480007755054e-02, 4.420490880043806e-02, -9.042803454926064e-02, 2.032925488887439e-02,  2.004702734757642e-01, 1.752215034693595e-02, -2.352027155671459e-02, -1.159630157634696e-02, -1.797355016420195e-02, -5.685710510174019e-03, -2.019381301614627e-02, -1.193182499020910e-02,  2.747540068932754e-02, 4.155561554100938e-02, -3.606942019904753e-02, 1.752215034695389e-02,  1.919352059454676e-01, 3.265071658316981e-02, 1.306817301157032e-02, 1.871505854863672e-02, -1.125679154665730e-02, 7.299887275347708e-03, -3.167367345500147e-03, 1.415288659725381e-02, -9.471907217910885e-03, 8.557305386150807e-03,
                  -3.117743319914563e-03, -1.924457292455443e-02,  1.306817301157368e-02, 8.043947428604432e-02, 4.980052071687351e-02, -1.613394783007573e-02,  4.213681863388568e-03, 1.063220163750843e-02, 3.406695470616677e-02, 9.879746528247758e-03,  3.494339823261888e-03, -2.573809207844308e-02, -1.903973465276798e-02, 1.871505854863079e-02,  4.980052071687541e-02, 2.396018445724980e-01, -8.876643628303806e-02, 4.048512332023427e-03,  6.119214511833619e-02, 7.844472858622784e-02, -3.817817839828655e-02, 6.674686659357888e-02, -2.731776844381964e-02, -3.489580347190301e-02, -1.125679154665886e-02, -1.613394783007609e-02, -8.876643628303731e-02, 1.185401987581707e-01, -6.308107826384614e-03, -2.977672031479952e-02, -3.967106696822190e-02, 1.966874025252598e-02, -2.785047659050406e-02, 3.140632605326728e-03,  7.074111088315868e-04, 7.299887275347583e-03, 4.213681863387692e-03, 4.048512332020599e-03, -6.308107826383816e-03, 7.391650841884796e-02, 2.338238392451760e-02, -2.900278706717399e-02,  2.604286765097846e-02, 2.941467924244617e-02, -1.997274773804425e-02, -1.739542588410304e-02, -3.167367345491737e-03, 1.063220163750605e-02, 6.119214511831913e-02, -2.977672031479386e-02,  2.338238392452011e-02, 9.122694693427877e-02, -3.045365378929759e-02, 2.357608836286906e-02,  2.570143613659742e-02, -6.308548887459330e-02, -2.065289347287343e-02, 1.415288659724093e-02,  3.406695470617232e-02, 7.844472858624475e-02, -3.967106696822804e-02, -2.900278706717394e-02, -3.045365378927548e-02, 2.554056778368617e-01, -3.002402177591699e-02, -4.684878751324406e-02,  3.675695926578264e-02, 1.894809036428081e-02, -9.471907217894726e-03, 9.879746528246469e-03, -3.817817839830956e-02, 1.966874025253491e-02, 2.604286765098499e-02, 2.357608836287796e-02, -3.002402177595622e-02, 2.600245571060802e-01, -6.284331533396581e-02, -1.286195745179811e-01,  4.981432110638128e-02, 8.557305386148028e-03, 3.494339823258555e-03, 6.674686659358099e-02, -2.785047659050496e-02, 2.941467924244291e-02, 2.570143613657902e-02, -4.684878751322329e-02, -6.284331533399132e-02, 2.093924752581740e-01, 4.278206801368688e-02, -3.768742121030118e-02, -3.117743319932406e-03, -2.573809207844330e-02, -2.731776844379212e-02, 3.140632605317297e-03, -1.997274773805302e-02, -6.308548887461539e-02, 3.675695926584127e-02, -1.286195745179974e-01,  4.278206801365753e-02, 2.457323213017761e-01, 3.193667922046702e-02, -1.924457292455489e-02, -1.903973465276903e-02, -3.489580347190037e-02, 7.074111088318843e-04, -1.739542588410533e-02, -2.065289347288623e-02, 1.894809036429631e-02, 4.981432110636524e-02, -3.768742121030223e-02,  3.193667922048542e-02, 1.975694793369391e-01, 2.969453581800157e-02, 1.013332118536387e-02,  1.279478998941962e-02, -6.710796186843102e-03, 1.581119238229850e-03, -1.313247664611373e-02,  2.390711155533298e-02, -2.869969091037577e-02, 3.836758935212717e-03, 1.294066147041163e-02, -2.136595141432926e-02, 1.013332118536695e-02, 6.514769206563858e-02, 2.670265011664250e-02, -4.707838028321711e-03, 3.693882956610697e-03, 2.798682196519887e-03, 2.083365976301615e-02,  8.150612492976075e-03, -9.987120141388417e-04, -1.702822182784596e-02, -1.368177779679585e-02,  1.279478998941206e-02, 2.670265011664509e-02, 1.424445315608139e-01, -3.956407405935173e-02,  5.809360176144353e-04, 2.498781657625412e-02, 4.441065534291304e-02, -4.428485274516851e-02,  4.037202174575288e-02, 7.972065762674235e-03, -2.124410609814997e-02, -6.710796186843913e-03, -4.707838028322492e-03, -3.956407405935152e-02, 7.973862519619787e-02, -2.138357750604173e-03, -1.061732866470268e-02, -2.163401808007834e-02, 2.220491326529592e-02, -1.328501784228337e-02, -1.103110583088564e-02, -1.032125970860708e-03, 1.581119238230788e-03, 3.693882956610478e-03,  5.809360176161214e-04, -2.138357750605802e-03, 5.943834206263062e-02, 1.660785423062577e-02, -1.235049772194527e-02, 1.156638111802505e-02, 1.478829878000452e-02, -1.182774775042373e-02, -2.056382633966393e-02, -1.313247664610097e-02, 2.798682196518920e-03, 2.498781657624651e-02, -1.061732866470259e-02, 1.660785423063230e-02, 9.048074930552011e-02, -3.725563160147091e-02,  4.045214627877101e-02, -1.632621215954336e-03, -7.319629331212014e-02, -1.889314943118092e-02,  2.390711155531372e-02, 2.083365976302019e-02, 4.441065534291572e-02, -2.163401808007600e-02, -1.235049772195242e-02, -3.725563160145069e-02, 1.781913008289503e-01, -1.793538503328349e-02, -1.367608226561727e-02, 4.308332958404617e-02, 2.785880373130084e-02, -2.869969091035039e-02,  8.150612492976840e-03, -4.428485274517527e-02, 2.220491326529486e-02, 1.156638111804215e-02,  4.045214627878704e-02, -1.793538503333054e-02, 2.271256509463152e-01, -7.839944598592165e-02, -1.353271040734118e-01, 1.399937247971067e-02, 3.836758935208518e-03, -9.987120141413345e-04,  4.037202174575832e-02, -1.328501784228553e-02, 1.478829877999689e-02, -1.632621215985049e-03, -1.367608226557682e-02, -7.839944598597169e-02, 1.544321335766320e-01, 6.159255381821035e-02, -2.739458389322294e-02, 1.294066147038520e-02, -1.702822182784795e-02, 7.972065762687896e-03, -1.103110583088642e-02, -1.182774775044270e-02, -7.319629331215091e-02, 4.308332958411572e-02, -1.353271040734331e-01, 6.159255381815938e-02, 2.327611514413557e-01, 3.117959632731494e-02, -2.136595141432850e-02, -1.368177779679654e-02, -2.124410609814496e-02, -1.032125970861839e-03, -2.056382633966507e-02, -1.889314943119404e-02, 2.785880373131725e-02, 1.399937247969281e-02, -2.739458389322033e-02, 3.117959632733621e-02, 1.450336777790332e-01, 3.138577064816310e-02,  1.017406681111556e-02, 1.162273623136625e-02, -6.972516151215692e-03, -1.398526011672011e-04, -2.068943897713963e-02, 3.500378928370760e-02, -2.738872748963340e-02, 9.087112701334091e-04,  2.163634140255117e-02, -1.278740210812496e-02, 1.017406681111856e-02, 6.300374854847696e-02,  3.003496029076137e-02, -7.454615756782858e-03, 5.243533350406532e-03, 7.788678785998032e-03,  1.706645182330176e-02, 1.201740919785289e-02, 2.214468238911436e-03, -2.176845561732527e-02, -1.453864826764609e-02, 1.162273623135740e-02, 3.003496029076489e-02, 1.499862344059820e-01, -4.577136392458645e-02, 1.473215176794771e-03, 3.007786965446699e-02, 5.001449347125014e-02, -2.771182700057501e-02, 3.731768402676634e-02, -1.099340569947552e-02, -2.919073705775486e-02, -6.972516151215895e-03, -7.454615756784115e-03, -4.577136392458671e-02, 8.048476744100456e-02, -1.765438477136190e-03, -9.346220088042263e-03, -3.301712255029169e-02, 1.402073818694787e-02, -7.029524576466046e-03, -3.315673651795216e-03, 1.816913663663520e-03, -1.398526011653274e-04,  5.243533350405964e-03, 1.473215176798088e-03, -1.765438477138871e-03, 5.858996162485933e-02,  2.024142753150613e-02, -1.667668915426324e-02, 1.423995440657427e-02, 1.754995663378659e-02, -1.720352195439018e-02, -1.986406479906502e-02, -2.068943897712328e-02, 7.788678785995759e-03,  3.007786965446185e-02, -9.346220088044260e-03, 2.024142753151626e-02, 1.152203587576947e-01, -5.444288919613405e-02, 7.004771544474681e-02, -3.988485037190605e-03, -1.192041323967690e-01, -2.720495570462812e-02, 3.500378928368349e-02, 1.706645182330764e-02, 5.001449347124629e-02, -3.301712255028497e-02, -1.667668915427316e-02, -5.444288919610717e-02, 2.006128654950760e-01, -3.168422484633096e-02,
                  -1.877610184942733e-02, 6.515661562056671e-02, 2.490864265818831e-02, -2.738872748960094e-02, 1.201740919785065e-02, -2.771182700057705e-02, 1.402073818694262e-02,  1.423995440659839e-02, 7.004771544476543e-02, -3.168422484639431e-02, 2.512558955297656e-01, -7.232594004749393e-02, -1.706668812405724e-01, 2.058689093143128e-02, 9.087112701283073e-04,  2.214468238909753e-03, 3.731768402677570e-02, -7.029524576469769e-03, 1.754995663377475e-02, -3.988485037234158e-03, -1.877610184936753e-02, -7.232594004756826e-02, 1.555734263923687e-01,  6.200739492584200e-02, -2.087043044256694e-02, 2.163634140251802e-02, -2.176845561732445e-02, -1.099340569946343e-02, -3.315673651793498e-03, -1.720352195441799e-02, -1.192041323968085e-01,  6.515661562066095e-02, -1.706668812406069e-01, 6.200739492576950e-02, 2.934696000173413e-01,  4.203135528462265e-02, -1.278740210812376e-02, -1.453864826764691e-02, -2.919073705774734e-02,  1.816913663661078e-03, -1.986406479906686e-02, -2.720495570464480e-02, 2.490864265820911e-02,  2.058689093140523e-02, -2.087043044256152e-02, 4.203135528465225e-02, 1.451970612740345e-01,  3.340916219869111e-02, 1.141586942888990e-02, 1.968242641001073e-02, -1.191308858074568e-02, -8.539893730233881e-05, -1.716185948501141e-02, 4.220184865382245e-02, -2.032173016914915e-02, -1.717880321512732e-03, 1.561928780683047e-02, -9.750546175456801e-03, 1.141586942889307e-02,  6.491684961824848e-02, 3.009201470109563e-02, -7.646926856876382e-03, 5.322083527887229e-03,  6.077634853135755e-03, 1.756683054507858e-02, 8.142657234690945e-03, 4.990077217381596e-03, -2.041654669292622e-02, -1.673932112613878e-02, 1.968242641000252e-02, 3.009201470109863e-02,  1.773706870452760e-01, -5.866774770030344e-02, 3.061526532183151e-03, 3.753583906064679e-02,  4.902255047467671e-02, -2.930634882914926e-02, 6.074058430373729e-02, -9.846274817137483e-03, -2.557037430565384e-02, -1.191308858074616e-02, -7.646926856877394e-03, -5.866774770030365e-02,  8.999353603547275e-02, -6.000178248583470e-03, -1.380991315004908e-02, -2.953267098640378e-02,  9.063661776760996e-03, -2.014447726744090e-02, -3.239913569856028e-03, -2.319218291541914e-03, -8.539893730023220e-05, 5.322083527886310e-03, 3.061526532184691e-03, -6.000178248585388e-03,  6.293456569652242e-02, 2.889756602512032e-02, -2.404490816916779e-02, 2.905563798052226e-02,  1.711190506158035e-02, -2.737921896295846e-02, -1.712772279377467e-02, -1.716185948499391e-02,  6.077634853132576e-03, 3.753583906063762e-02, -1.380991315004926e-02, 2.889756602513149e-02,  1.193372722422526e-01, -6.774898648675616e-02, 7.299624296152808e-02, 1.619266612991213e-02, -1.075139883530367e-01, -2.077912630691970e-02, 4.220184865379756e-02, 1.756683054508557e-02,  4.902255047467911e-02, -2.953267098639979e-02, -2.404490816917692e-02, -6.774898648672316e-02,  2.180245453120518e-01, -4.824026482366298e-02, -1.980717945755922e-02, 6.664827330640738e-02,  1.803038375698861e-02, -2.032173016911549e-02, 8.142657234687999e-03, -2.930634882915591e-02,  9.063661776757864e-03, 2.905563798054586e-02, 7.299624296154199e-02, -4.824026482372790e-02,  2.484084405122211e-01, -6.010210837358746e-02, -1.531302000903932e-01, 2.418510189741646e-02, -1.717880321517760e-03, 4.990077217379258e-03, 6.074058430374432e-02, -2.014447726744368e-02,  1.711190506156833e-02, 1.619266612986649e-02, -1.980717945749873e-02, -6.010210837366353e-02,  1.631570660261873e-01, 4.476881162872911e-02, -1.833628130935894e-02, 1.561928780679564e-02, -2.041654669292461e-02, -9.846274817121005e-03, -3.239913569856351e-03, -2.737921896298748e-02, -1.075139883530764e-01, 6.664827330650980e-02, -1.531302000904345e-01, 4.476881162865363e-02,  2.612914471003710e-01, 4.367270775177928e-02, -9.750546175455829e-03, -1.673932112613956e-02, -2.557037430564688e-02, -2.319218291544095e-03, -1.712772279377752e-02, -2.077912630693945e-02,  1.803038375701300e-02, 2.418510189738671e-02, -1.833628130935429e-02, 4.367270775181312e-02,  1.494483187827050e-01, 3.685448389587359e-02, 1.202575183593217e-02, 2.165349028959057e-02, -1.365609595879187e-02, -1.927990636763794e-03, -2.072698381452399e-02, 5.229056602732973e-02, -2.662597982211012e-02, -1.019463252719269e-02, 1.436916789674932e-02, -1.444416011045866e-02,  1.202575183593509e-02, 6.598134943307042e-02, 2.255918204108881e-02, -6.031526647896451e-03,  3.818274932922489e-03, -1.436441546515058e-03, 1.820857673207023e-02, -1.785255851777770e-03,  4.280705861406700e-03, -7.862816567529473e-03, -1.384170228358905e-02, 2.165349028958140e-02,  2.255918204109266e-02, 1.502766936561202e-01, -4.934829958148216e-02, -4.681532416370302e-03,  1.799117196700894e-02, 5.030265553758431e-02, -2.359487179669371e-02, 2.425503538185954e-02, -1.026846449026761e-02, -1.396776938719608e-02, -1.365609595879201e-02, -6.031526647897877e-03, -4.934829958148268e-02, 9.365563344008393e-02, -5.019856473008316e-03, -4.735047552587930e-03, -3.315611469014364e-02, 6.665796229146760e-03, -5.211542025856392e-03, -5.524129477698127e-04, -1.948083542639816e-03, -1.927990636761903e-03, 3.818274932922301e-03, -4.681532416365162e-03, -5.019856473011655e-03, 6.248557685916076e-02, 1.937917660901730e-02, -1.333041700955019e-02,  2.083659216400132e-02, 9.232476513577949e-03, -1.434018786982430e-02, -1.334324840523708e-02, -2.072698381450669e-02, -1.436441546517212e-03, 1.799117196700456e-02, -4.735047552589911e-03,  1.937917660902921e-02, 9.905341194589576e-02, -5.488357710974667e-02, 5.638925952649759e-02,  6.949635488978429e-03, -7.317522305658197e-02, -2.729938309690037e-03, 5.229056602730404e-02,  1.820857673207609e-02, 5.030265553757780e-02, -3.315611469013597e-02, -1.333041700956359e-02, -5.488357710971725e-02, 2.010704606242281e-01, -2.756608924520650e-02, -1.852133658929966e-02,  3.638498439182319e-02, 8.894607000816349e-03, -2.662597982207566e-02, -1.785255851779931e-03, -2.359487179669334e-02, 6.665796229140703e-03, 2.083659216402999e-02, 5.638925952651733e-02, -2.756608924527627e-02, 1.996036103276144e-01, -3.706837255467921e-02, -9.523529490903804e-02,  2.300684836339446e-02, -1.019463252719812e-02, 4.280705861405562e-03, 2.425503538187073e-02, -5.211542025860673e-03, 9.232476513564871e-03, 6.949635488930501e-03, -1.852133658923088e-02, -3.706837255476268e-02, 1.245209363281395e-01, 2.058340226157841e-02, -1.460061597838263e-02,  1.436916789671442e-02, -7.862816567528757e-03, -1.026846449025669e-02, -5.524129477679921e-04, -1.434018786985609e-02, -7.317522305662484e-02, 3.638498439192668e-02, -9.523529490907685e-02,  2.058340226149861e-02, 1.743793720379189e-01, 1.612685506925629e-02, -1.444416011045709e-02, -1.384170228358970e-02, -1.396776938718747e-02, -1.948083542642737e-03, -1.334324840523809e-02, -2.729938309707283e-03, 8.894607000838009e-03, 2.300684836336696e-02, -1.460061597837557e-02,  1.612685506928765e-02, 1.215790753826524e-01, 4.406567017009581e-02, 1.611740362826771e-02,  2.581900303932544e-02, -1.641733351022277e-02, -2.165498188716516e-03, -2.174125555097793e-02,  6.581369202002144e-02, -1.845585977541059e-02, -1.668500506061901e-02, 9.077881152541975e-03, -1.026610683104599e-02, 1.611740362827080e-02, 9.034121237303860e-02, 4.077518987340040e-02, -1.276965720024562e-02, 9.641057767751347e-03, 3.440024702998694e-03, 2.685036729334433e-02,
                  6.911036324454846e-03, 1.424399260057004e-02, -2.056928478452555e-02, -2.234029693025205e-02,  2.581900303931689e-02, 4.077518987340401e-02, 1.882353825292709e-01, -5.823515172339511e-02, -6.779759598555563e-03, 2.438696472723667e-02, 7.213565189930593e-02, -2.383716087657534e-02,  4.313168202177811e-02, -1.470507486826667e-02, -1.115847796291711e-02, -1.641733351022322e-02, -1.276965720024685e-02, -5.823515172339550e-02, 1.160453907677249e-01, -3.883180980235598e-03, -5.740269631523021e-03, -4.593533368435826e-02, 1.106688678419915e-02, -1.046643996789759e-02, -4.882187224010031e-03, -9.071966068499776e-03, -2.165498188715080e-03, 9.641057767751315e-03, -6.779759598550860e-03, -3.883180980238587e-03, 8.006092053523055e-02, 1.905315303796826e-02, -1.357067905573067e-02, 1.890493176608177e-02, 1.592709016561800e-02, -1.417448157896628e-02, -2.443151568645934e-02, -2.174125555096189e-02, 3.440024702997452e-03, 2.438696472723284e-02, -5.740269631524908e-03, 1.905315303797764e-02, 1.145332224136425e-01, -6.422337095279862e-02,  3.377231607292852e-02, 1.562041525124946e-02, -7.566890539225979e-02, -1.574626809016209e-02,  6.581369201999808e-02, 2.685036729334922e-02, 7.213565189930071e-02, -4.593533368435188e-02, -1.357067905574233e-02, -6.422337095277469e-02, 2.761814442740831e-01, -4.609244473598451e-03, -1.699733263606490e-02, 4.915484664647285e-02, 3.581922018943756e-02, -1.845585977537833e-02,  6.911036324454330e-03, -2.383716087657411e-02, 1.106688678419332e-02, 1.890493176610583e-02,  3.377231607294717e-02, -4.609244473657070e-03, 2.269521042598199e-01, -5.103764387607640e-02, -9.474221410694206e-02, 2.537905888537836e-02, -1.668500506062495e-02, 1.424399260056872e-02,  4.313168202178783e-02, -1.046643996790098e-02, 1.592709016560652e-02, 1.562041525120744e-02, -1.699733263600381e-02, -5.103764387614956e-02, 1.760596787046464e-01, 2.762449398451911e-02, -2.273225737140558e-02, 9.077881152508988e-03, -2.056928478452658e-02, -1.470507486825759e-02, -4.882187224007887e-03, -1.417448157899301e-02, -7.566890539229953e-02, 4.915484664656278e-02, -9.474221410697714e-02, 2.762449398444938e-02, 2.231335067253775e-01, 5.370806730610783e-02, -1.026610683104483e-02, -2.234029693025261e-02, -1.115847796290884e-02, -9.071966068502364e-03, -2.443151568646059e-02, -1.574626809017875e-02, 3.581922018945909e-02, 2.537905888535216e-02, -2.273225737139933e-02, 5.370806730613756e-02, 1.868978800382087e-01 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$lag_one_cov)),
               c(NULL )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$fixed_effects)),
               c(numeric(0) )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$n_iter)),
               c(11 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$Q)),
               c( 0.053963565318124060, -0.001767528994999920, 0.060992995090108679, -0.042421834696197050, 0.029494293763047928, -0.005052976354507091, -0.015870825899571974, -0.027467805791203483, 0.115902242384520543, 0.088237892810946322,  0.054265979472639753, -0.001767528994999920, 0.091099559442920600, 0.214069873226740204, -0.087772769994196198,  0.001280646596698863, 0.074625056426907893, 0.154153310027002138, 0.036148188763894161, -0.018769853446250782, -0.141629958299829994, -0.085698367382245677, 0.060992995090108679, 0.214069873226740204, 1.132608875544732996, -0.507348362989995749, -0.006994054096283742, 0.283817902147758039, 0.620872450168335588, -0.307399212494731899,  0.263390051365182121, -0.164188734526401980, -0.270906649133739852, -0.042421834696197050, -0.087772769994196198, -0.507348362989995749, 0.271432595006915522, -0.001383054759469348, -0.117677114383347881, -0.285462202375545981,  0.139601704858916570, -0.131465011724609809, 0.033309804848352796, 0.077012883129182294, 0.029494293763047928,  0.001280646596698863, -0.006994054096283742, -0.001383054759469348, 0.124568596984347829, 0.154764867610498846, -0.267759751328224271, 0.242606172872785020, 0.120084991227200760, -0.207847877369502554, -0.001763776198704824, -0.005052976354507091, 0.074625056426907893, 0.283817902147758039, -0.117677114383347881, 0.154764867610498846,  0.606249984945859799, -0.428173608969356134, 0.723555267988979844, 0.029708199536202396, -0.948678888011278199, -0.128854073866992824, -0.015870825899571974, 0.154153310027002138, 0.620872450168335588, -0.285462202375545981, -0.267759751328224271, -0.428173608969356134, 1.338886587194153543, -0.878859198510111517, -0.184262544989179960,  0.681462142547708649, -0.109250876130561558, -0.027467805791203483, 0.036148188763894161, -0.307399212494731899,  0.139601704858916570, 0.242606172872785020, 0.723555267988979844, -0.878859198510111517, 1.943640055002746392, -0.386443546233010138, -1.634876815833099961, 0.242798945874415306, 0.115902242384520543, -0.018769853446250782,  0.263390051365182121, -0.131465011724609809, 0.120084991227200760, 0.029708199536202396, -0.184262544989179960, -0.386443546233010138, 0.612091094624331644, 0.371299280029425971, -0.032895311857572079, 0.088237892810946322, -0.141629958299829994, -0.164188734526401980, 0.033309804848352796, -0.207847877369502554, -0.948678888011278199,  0.681462142547708649, -1.634876815833099961, 0.371299280029425971, 1.940945042490894057, 0.184545433574474338,  0.054265979472639753, -0.085698367382245677, -0.270906649133739852, 0.077012883129182294, -0.001763776198704824, -0.128854073866992824, -0.109250876130561558, 0.242798945874415306, -0.032895311857572079, 0.184545433574474338,  0.425755992345377976 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$Q_0)),
               c(1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$n_risk)),
               c(2007, 1936, 1882, 1818, 1767, 1656, 1543, 1364, 1254, 1103 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$times)),
               c( 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$risk_set)),
               c(NULL )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$data)),
               c(NULL )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$order)),
               c(1 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$F_)),
               c(1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$method)),
               c("UKF" )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$model)),
               c("exp_clip_time" )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$est_Q_0)),
               c(FALSE )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$LR)),
               c(1 )
               , tolerance = 1e-04)

  # tmp <- file("tmp.txt")
  # sink(tmp)
  suppressMessages(result_exp <- ddhazard(
    formula = survival::Surv(tstart, tstop, event) ~ . - id - tstart - tstop - event,
    data = sims$res,
    by = (by_ <- 1),
    Q_0 = diag(1, 11),
    Q = diag(1e-1, 11),
    control = list(est_Q_0 = F, eps = 10^-2, n_max = 10^3, method = "UKF",
                   debug = F, beta = 0, save_data = F, save_risk_set = F),
    max_T = 10,
    id = sims$res$id, order = 1,
    verbose = F,
    model = "exp_clip_time_w_jump"))
  # sink()
  # close(tmp)

  # matplot(sims$betas, type = "l", lty = 1)
  # matplot(result_exp$state_vecs, type = "l", lty = 2, add = T)
  get_expect_equal(result_exp, file = "tmp.txt", eps = 1e-4)

  expect_equal(unname(c(result_exp$state_vecs)),
               c(-5.0594371476177792957, -5.0478733970133857767, -4.9334671766549735850, -4.8815224477289946847, -4.6518383803269678367, -4.8255088043369402939, -5.1814754775224596273, -5.4138676406436632149, -5.5507926688071425048, -5.3738774400176838597, -5.0471394548345527298, -2.0606454791055957010, -2.0307587988395248146, -2.7826766582939157502, -2.3641808443619458657, -1.6558308133545325269, -1.6588922695524082052, -1.9555817984390442277, -2.4899354411818466204, -2.8385995558018612961, -2.4418045482669281654, -1.9828888586895438984, 0.0216505485141385429, 0.1124175086556257752, -1.1937532418545704349, -0.7511548531329212963, 0.8830536123916569169, 0.3141661791133426584,  0.6997909322998783210, -0.7416504680954694173, -2.2503639777357982155, -1.1953016380600671731,  1.0429737375406584743, 0.0451961688521653401, 0.0001473237809711536, 0.6104276511088350121,  0.4346933830827347522, -0.3838235059472282495, -0.0623915711294139030, 0.0003451267811909464,  0.6703165097873091671, 1.4420313471928185578, 0.9798243155928579862, -0.0492860410896038825, -0.6690066287189109229, -0.6735373936992608002, -0.5785567288235119454, -0.2023448934722905657, -0.0238710197370076427, 0.5087142398741091309, 0.6163718488880316482, 0.8055410249091116137,  1.0663628265037887743, 1.4795133418143338044, 1.8114622004735174698, 0.7799161845963035189,  0.7851121242123426658, -0.4507645121608844230, 0.6951907815964391002, 0.8407016239129085911,  1.9155919305761575089, 3.3978056865227568295, 3.3953518479612325720, 3.0911287462362841971,  2.9233347848660655721, 3.0170446651534277294, -0.7581651680028925755, -0.6708480391471846049, -1.4274732405169312521, -2.7380734111501521255, -1.4051660936602168306, -3.7470968927827108530, -5.5116630054595274757, -7.0732454685484622559, -8.3662798551544739212, -7.9520298037881795850, -6.9406920072580788883, -0.0660648981554861486, -0.1035314777584011026, -1.4371887548942199331,  1.3715288172632853492, 1.5424020334392452014, 4.0190151205836421866, 3.2497756584402730518,  3.7308264141838054506, 4.8328910464300847138, 4.1829005816312321286, 2.2684078186527476539,  0.6535769523649280766, 0.6649317206123965818, 1.7065780332627826787, 1.6497127613035829086,  1.9663474056541780932, 2.2602227409674595648, 3.7576263046298725712, 3.8325831715029043956,  3.7002393885647149041, 5.1437507264240274196, 7.4614857599256740883, 1.8423912487808946636,  1.8247406946227560098, 4.4394132952691602867, 2.4428357542292697957, 1.8313224231755262927,  0.4403614632044715038, -0.2835457300146555504, 0.0879419679259914044, 0.4221748716656135159,  1.0670333321679628114, 1.9421508425895401828, 2.5533495622797919289, 2.5222242921228721357,  3.4678576388655866225, 4.1469120370650855989, 3.7597911924297959807, 4.4428848583517259740,  3.5777066772648886683, 3.9949285408166574918, 4.6604923299362219780, 4.4784637695642812716,  3.9314964226768864819 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$state_vars)),
               c( 7.167277001264338e-02, 3.012363708032218e-02, 5.882822884692038e-02, -2.760415605378606e-02,  1.196579158077147e-02, -4.610038938386102e-02, 5.387965373688391e-02, 1.872833546740221e-02,  1.768673566770112e-02, 2.893552337194889e-02, 4.568221641955467e-02, 3.012363708032506e-02,  1.559421246389784e-01, 1.281871444302712e-01, -5.280846161569639e-02, 3.902307216278133e-02, -5.267126729343964e-03, 9.158228953308320e-02, 7.957511584487488e-02, 2.287244012979956e-02, -5.478082770760546e-02, -1.533879142624358e-02, 5.882822884692988e-02, 1.281871444302721e-01,  4.485709591834057e-01, -1.720263020881177e-01, 1.751618891935770e-02, 8.829393534062770e-02,  1.718822283325272e-01, -3.905712982672408e-02, 1.531449532199659e-01, -9.761461812920548e-02, -4.648576827300215e-02, -2.760415605378082e-02, -5.280846161569607e-02, -1.720263020881183e-01,  2.179819400721940e-01, -2.623503504090503e-02, -2.747430457073803e-02, -1.010066058433570e-01,  1.072113467794219e-02, -4.288699583457414e-02, 3.004684916988342e-02, -1.344972820513043e-03,  1.196579158077243e-02, 3.902307216278106e-02, 1.751618891935822e-02, -2.623503504090511e-02,  1.963668153780556e-01, -6.901289666348790e-03, -4.041137969050288e-02, 8.768840761006164e-02,  1.200354088285352e-01, 2.609265934673368e-02, 5.385025472625678e-03, -4.610038938386449e-02, -5.267126729344020e-03, 8.829393534062754e-02, -2.747430457073749e-02, -6.901289666348651e-03,  2.896174404484965e-01, -1.483750805526049e-01, 2.763458774614324e-02, 4.705059215808152e-02, -2.073408203652864e-01, -7.839744541438277e-02, 5.387965373688750e-02, 9.158228953308301e-02,  1.718822283325276e-01, -1.010066058433565e-01, -4.041137969050276e-02, -1.483750805526052e-01,  5.391353326203976e-01, -9.869578392245480e-02, -1.347350456490493e-01, 4.304504224779240e-02, -4.468638234397154e-02, 1.872833546740240e-02, 7.957511584487503e-02, -3.905712982672443e-02,  1.072113467794220e-02, 8.768840761006158e-02, 2.763458774614327e-02, -9.869578392245495e-02,  5.674252881428099e-01, -8.932538199299858e-02, -1.904160006601428e-01, 1.395314153432834e-01,  1.768673566769957e-02, 2.287244012979956e-02, 1.531449532199656e-01, -4.288699583457416e-02,  1.200354088285350e-01, 4.705059215808143e-02, -1.347350456490495e-01, -8.932538199299873e-02,  5.229698717201849e-01, 1.080783289446774e-01, -1.128197300935103e-02, 2.893552337194662e-02, -5.478082770760544e-02, -9.761461812920612e-02, 3.004684916988310e-02, 2.609265934673372e-02, -2.073408203652858e-01, 4.304504224779261e-02, -1.904160006601427e-01, 1.080783289446775e-01,  4.633874399959663e-01, 9.131165828498068e-02, 4.568221641955156e-02, -1.533879142624368e-02, -4.648576827300303e-02, -1.344972820513171e-03, 5.385025472625251e-03, -7.839744541438236e-02, -4.468638234397199e-02, 1.395314153432834e-01, -1.128197300935108e-02, 9.131165828498067e-02,  3.583463368589295e-01, 5.858531429161296e-02, 2.968426592378971e-02, 5.788082971471342e-02, -2.848932748917576e-02, 9.416456285027294e-03, -3.567485799632455e-02, 5.900451648527225e-02,  8.108020758518172e-04, 1.671781683144797e-02, 2.325871386224580e-02, 2.597391333591106e-02,  2.968426592380016e-02, 1.287489668091214e-01, 1.066854525904777e-01, -4.754839017059703e-02,  3.300262116478588e-02, -4.781421931079097e-03, 7.251213322083151e-02, 4.699304162263356e-02,  3.418230929371583e-02, -3.390442871676720e-02, -1.045749323673784e-02, 5.788082971474427e-02,  1.066854525904795e-01, 3.057142171542159e-01, -1.247356339368496e-01, 2.296495038598789e-02,  2.832028701715424e-02, 1.421001575074346e-01, -1.583879897031409e-02, 1.113354801080669e-01, -4.022806709019446e-02, -1.656222693777873e-02, -2.848932748918034e-02, -4.754839017059706e-02, -1.247356339368501e-01, 1.825647613333068e-01, -2.545470410649040e-02, -4.431569702624852e-03, -9.203306712504480e-02, 9.396561890660170e-03, -3.609097574650284e-02, 6.220607910964564e-03, -8.130361119492498e-03, 9.416456285027621e-03, 3.300262116478567e-02, 2.296495038598807e-02, -2.545470410649055e-02, 1.651103938911544e-01, 1.037518549586777e-03, -3.115627788331878e-02,  6.658166129142791e-02, 1.053557638383436e-01, 1.736702303595023e-02, 2.479871003881105e-03, -3.567485799632747e-02, -4.781421931077737e-03, 2.832028701715682e-02, -4.431569702623895e-03,  1.037518549586652e-03, 1.884029620302545e-01, -1.119439443747721e-01, 3.693580442235277e-02,  2.019572070165112e-02, -1.414754620765688e-01, -5.013396544279257e-02, 5.900451648529673e-02,  7.251213322083114e-02, 1.421001575074345e-01, -9.203306712504515e-02, -3.115627788331794e-02, -1.119439443747742e-01, 4.226103735613917e-01, -9.735044760014092e-02, -7.415581051446388e-02,  6.409136853639831e-02, -1.666687459789357e-02, 8.108020758443094e-04, 4.699304162263390e-02, -1.583879897031510e-02, 9.396561890661202e-03, 6.658166129142781e-02, 3.693580442235378e-02, -9.735044760014275e-02, 3.263185778759286e-01, -1.218304399767721e-02, -1.236979054782301e-01,  6.936622471334426e-02, 1.671781683144820e-02, 3.418230929371500e-02, 1.113354801080627e-01, -3.609097574650166e-02, 1.053557638383429e-01, 2.019572070165130e-02, -7.415581051446733e-02, -1.218304399767609e-02, 3.940017473133067e-01, 7.142274586255370e-02, -1.990128738578369e-03,  2.325871386223864e-02, -3.390442871676967e-02, -4.022806709020067e-02, 6.220607910964774e-03,  1.736702303595016e-02, -1.414754620765667e-01, 6.409136853639401e-02, -1.236979054782279e-01,  7.142274586255282e-02, 3.563016981391966e-01, 6.925228955586318e-02, 2.597391333589974e-02, -1.045749323674011e-02, -1.656222693778601e-02, -8.130361119490632e-03, 2.479871003880453e-03, -5.013396544279026e-02, -1.666687459790020e-02, 6.936622471334697e-02, -1.990128738580527e-03,  6.925228955586143e-02, 2.731357771700718e-01, 5.330653307476149e-02, 4.072091025579824e-02,  8.706560259942013e-02, -4.346427777852640e-02, 8.516209687290336e-03, -3.635528344051271e-02,  9.583871823179786e-02, -1.424426915067813e-02, 1.950043559023341e-02, 1.979753583149558e-02,  9.932216109921566e-03, 4.072091025580786e-02, 1.437973674663465e-01, 1.809347234715767e-01, -8.616254641838025e-02, 3.087516450777031e-02, 3.368977872281426e-03, 1.391868565542526e-01,  4.074702159002137e-02, 3.694469487812006e-02, -5.943327548249874e-02, -2.479102270832014e-02,  8.706560259944825e-02, 1.809347234715766e-01, 5.084746177383471e-01, -2.287478797746938e-01,  3.127577676045908e-02, 3.234407311068040e-02, 3.271536112101924e-01, -1.060103947844561e-01,  1.817151569988994e-01, -5.651031197411087e-02, -8.385158706011910e-02, -4.346427777852942e-02, -8.616254641837928e-02, -2.287478797746941e-01, 2.166831531438628e-01, -2.387866411000700e-02, -1.908357810763409e-03, -1.926188970820574e-01, 5.466777641858728e-02, -6.455738650835281e-02,  1.475058330830203e-02, 2.685240373217318e-02, 8.516209687290732e-03, 3.087516450777033e-02,  3.127577676045948e-02, -2.387866411000723e-02, 1.479696654476005e-01, 1.543197022514889e-02, -5.369461125908330e-02, 7.987982727092241e-02, 1.158293797927296e-01, 4.667062535512419e-03,  1.402491608309304e-02, -3.635528344051601e-02, 3.368977872282443e-03, 3.234407311068285e-02, -1.908357810761705e-03, 1.543197022514891e-02, 2.055654853324825e-01, -1.621419453784208e-01,  9.005756336797091e-02, 3.750539494735759e-02, -1.804897490857513e-01,
                  -4.284523540834103e-02,  9.583871823182009e-02, 1.391868565542506e-01, 3.271536112101928e-01, -1.926188970820588e-01, -5.369461125908284e-02, -1.621419453784231e-01, 6.968486804557599e-01, -2.381153068643731e-01, -9.273974114682185e-02, 8.931750079146805e-02, -8.932124479946996e-02, -1.424426915068471e-02,  4.074702159002258e-02, -1.060103947844569e-01, 5.466777641858889e-02, 7.987982727092238e-02,  9.005756336797237e-02, -2.381153068643740e-01, 5.614254603297200e-01, -1.074029325976367e-01, -2.553298686795891e-01, 1.430517414716550e-01, 1.950043559023330e-02, 3.694469487811954e-02,  1.817151569988961e-01, -6.455738650835213e-02, 1.158293797927289e-01, 3.750539494735791e-02, -9.273974114682507e-02, -1.074029325976351e-01, 5.095161674583821e-01, 1.197161969309647e-01, -2.130990456094691e-02, 1.979753583148985e-02, -5.943327548250043e-02, -5.651031197411607e-02,  1.475058330830074e-02, 4.667062535512231e-03, -1.804897490857509e-01, 8.931750079146548e-02, -2.553298686795891e-01, 1.197161969309657e-01, 4.457572842898987e-01, 4.909995359490401e-02,  9.932216109911594e-03, -2.479102270832182e-02, -8.385158706012649e-02, 2.685240373217504e-02,  1.402491608309241e-02, -4.284523540833961e-02, -8.932124479947601e-02, 1.430517414716567e-01, -2.130990456094816e-02, 4.909995359490375e-02, 2.787597766205456e-01, 3.715433212608905e-02,  3.681654398407083e-02, 6.977957202116783e-02, -3.156137303805041e-02, 6.959589530424292e-03, -1.815433427860915e-02, 7.065886854190621e-02, 3.440928638681742e-03, 3.970059038596287e-03, -9.015631724237201e-03, -3.979747859577262e-03, 3.681654398407454e-02, 1.105670717991367e-01,  1.383876053869151e-01, -6.145161731992374e-02, 2.137514333891412e-02, -1.060755037582573e-02,  1.101144402998182e-01, 2.022793019178445e-02, 3.103621746807496e-02, -2.161803464640541e-02, -7.022069157706062e-03, 6.977957202118393e-02, 1.383876053869160e-01, 3.401632110280678e-01, -1.422188232339052e-01, 2.973535945720125e-02, -1.619731376559568e-03, 2.198414633578150e-01, -4.416491695288442e-02, 1.115905845612983e-01, -1.672659659503592e-02, -2.816119249405696e-02, -3.156137303804689e-02, -6.145161731992379e-02, -1.422188232339055e-01, 1.503141257765231e-01, -2.108622732963969e-02, 9.583319153611233e-03, -1.206258068344662e-01, 2.789086915562859e-02, -4.163684789686467e-02, -4.088872305053989e-03, 4.018994918859389e-03, 6.959589530424642e-03,  2.137514333891577e-02, 2.973535945720395e-02, -2.108622732964083e-02, 9.398589710441260e-02,  7.108547960342163e-03, -1.489083245102066e-04, 3.660612612692234e-02, 5.041219506321049e-02,  8.035795722096239e-04, 2.198472996993314e-03, -1.815433427861957e-02, -1.060755037583201e-02, -1.619731376572792e-03, 9.583319153624370e-03, 7.108547960341306e-03, 1.302833088220102e-01, -1.220672142289923e-01, 4.684483538334677e-02, 3.203566189617073e-02, -7.389105237124011e-02,  6.567835185548664e-03, 7.065886854192209e-02, 1.101144402998186e-01, 2.198414633578212e-01, -1.206258068344738e-01, -1.489083245120523e-04, -1.220672142289831e-01, 4.445233756882537e-01, -1.267645837452819e-01, -1.640582873613504e-02, 7.504113977680514e-02, -4.669966965899288e-02,  3.440928638665132e-03, 2.022793019177329e-02, -4.416491695291853e-02, 2.789086915565161e-02,  3.660612612691997e-02, 4.684483538334702e-02, -1.267645837453050e-01, 3.402974473187749e-01, -5.560725118627209e-02, -1.234067649723737e-01, 1.180154463094714e-01, 3.970059038600834e-03,  3.103621746808633e-02, 1.115905845613219e-01, -4.163684789687845e-02, 5.041219506321175e-02,  3.203566189617747e-02, -1.640582873612196e-02, -5.560725118626167e-02, 2.679938450129155e-01,  2.769975632246939e-02, -3.698441007059478e-02, -9.015631724225287e-03, -2.161803464639330e-02, -1.672659659500466e-02, -4.088872305079204e-03, 8.035795722124567e-04, -7.389105237124995e-02,  7.504113977683471e-02, -1.234067649723951e-01, 2.769975632249405e-02, 1.993706273152567e-01, -3.386015186300662e-02, -3.979747859582772e-03, -7.022069157707558e-03, -2.816119249406471e-02,  4.018994918860659e-03, 2.198472996993189e-03, 6.567835185542749e-03, -4.669966965899435e-02,  1.180154463094609e-01, -3.698441007058783e-02, -3.386015186299365e-02, 1.795527482352147e-01,  2.302905485364657e-02, 1.625952643448258e-02, 2.702569046542433e-02, -1.047123687666176e-02,  2.141152299260445e-03, -1.578580131224611e-02, 3.703500712187388e-02, -2.249549514314777e-03, -3.577963329339316e-03, 1.144177433177325e-03, -2.120310964970618e-03, 1.625952643448469e-02,  6.906029573190303e-02, 7.272983809431377e-02, -2.736319047340137e-02, 1.696995601018141e-02,  6.802375617270460e-03, 3.797182005968341e-02, 3.638740158298797e-02, 1.803853441786689e-02, -3.183503548688139e-02, -3.949393100300163e-03, 2.702569046543649e-02, 7.272983809431652e-02,  2.122600244610777e-01, -7.373604440365017e-02, 2.125699348758542e-02, 3.567885573365567e-02,  8.032942778537458e-02, -1.056951195512564e-02, 9.078435790999502e-02, -3.620721289346360e-02, -2.316447945567726e-02, -1.047123687665644e-02, -2.736319047340232e-02, -7.373604440365079e-02,  1.015921701836715e-01, -1.810272518545696e-02, -9.588810409233327e-03, -3.826521349540048e-02,  3.580718245213820e-03, -3.720622792129109e-02, 3.690409181078904e-03, -5.242021871338962e-03,  2.141152299260662e-03, 1.696995601018349e-02, 2.125699348758681e-02, -1.810272518545756e-02,  8.519494568153134e-02, 1.205264998536541e-02, -1.149688044653000e-02, 3.712187456678021e-02,  4.598051468766087e-02, -9.732068191527509e-03, -7.070567143414812e-03, -1.578580131225947e-02,  6.802375617258900e-03, 3.567885573362342e-02, -9.588810409211244e-03, 1.205264998536475e-02,  1.032752605390561e-01, -6.494946272574387e-02, 2.568883993078648e-02, 3.516965539796366e-02, -6.607780449473585e-02, -1.824288457786061e-02, 3.703500712188906e-02, 3.797182005968974e-02,  8.032942778540143e-02, -3.826521349541753e-02, -1.149688044653226e-02, -6.494946272571836e-02,  2.515419789979263e-01, -5.132457003297494e-02, -3.303508658035801e-02, 4.924831262468179e-02,  8.432684208344594e-03, -2.249549514333493e-03, 3.638740158296606e-02, -1.056951195519483e-02,  3.580718245252444e-03, 3.712187456677518e-02, 2.568883993076809e-02, -5.132457003300484e-02,  2.364042634542226e-01, -4.314823409299454e-02, -1.044579819119158e-01, 3.802852646085619e-02, -3.577963329334966e-03, 1.803853441788377e-02, 9.078435791002849e-02, -3.720622792130897e-02,  4.598051468766490e-02, 3.516965539798203e-02, -3.303508658034934e-02, -4.314823409296849e-02,  2.394108104407009e-01, 1.424273157406598e-02, -4.046854140647841e-02, 1.144177433193514e-03, -3.183503548685918e-02, -3.620721289340103e-02, 3.690409181039321e-03, -9.732068191523089e-03, -6.607780449474002e-02, 4.924831262473164e-02, -1.044579819119560e-01, 1.424273157410792e-02,  1.885551752652391e-01, 2.649083818434955e-03, -2.120310964974691e-03, -3.949393100302232e-03, -2.316447945568750e-02, -5.242021871336839e-03, -7.070567143415607e-03, -1.824288457787521e-02,  8.432684208351728e-03, 3.802852646083374e-02, -4.046854140647022e-02, 2.649083818460244e-03,  1.305827318764843e-01, 2.152894597037844e-02, 1.553359499314140e-02, 3.019427670792271e-02, -1.399879074348338e-02, -2.344449325556255e-05, -1.988789666800739e-02, 4.626367778575315e-02, -1.134915738929078e-02, -3.983532506832291e-03,
                  7.535032973566248e-03, -3.696054673680078e-03,  1.553359499314262e-02, 5.147839080035219e-02, 6.265717615840098e-02, -2.750520973734088e-02,  7.612969660081649e-03, -7.973473855931874e-03, 5.578490293962841e-02, 5.814935976255821e-03,  8.431107870574444e-03, -1.385896083617351e-02, -1.179643780907430e-02, 3.019427670793426e-02,  6.265717615840603e-02, 1.829216487075910e-01, -7.422769298318629e-02, 1.779953676938876e-03,  3.749685250745117e-03, 1.254818281952972e-01, -4.069592039400399e-02, 4.545459214002594e-02, -1.921740669610410e-02, -3.544663744190699e-02, -1.399879074347764e-02, -2.750520973734288e-02, -7.422769298318632e-02, 1.009849517338980e-01, -6.449076183390340e-03, 7.416635166407185e-03, -7.521578685149621e-02, 2.165718171569351e-02, -1.442495609912645e-02, -2.188335076442187e-03,  4.676313606399871e-03, -2.344449325549468e-05, 7.612969660083705e-03, 1.779953676940107e-03, -6.449076183390794e-03, 7.711763053739057e-02, 6.190871185262852e-03, -2.386194331733330e-02,  3.581499959856874e-02, 4.238775650657638e-02, 6.963356822811372e-04, -1.735693530368429e-03, -1.988789666802247e-02, -7.973473855946635e-03, 3.749685250706460e-03, 7.416635166432650e-03,  6.190871185261657e-03, 1.116120308766598e-01, -9.647341646634242e-02, 3.482566189050070e-02,  2.161136860173965e-02, -7.919318515201693e-02, -1.959714773430259e-02, 4.626367778576887e-02,  5.578490293963891e-02, 1.254818281953266e-01, -7.521578685151524e-02, -2.386194331733431e-02, -9.647341646631077e-02, 3.471616642098148e-01, -9.076720236859703e-02, -5.687406819057411e-02,  5.835211019488253e-02, -1.351146566462145e-02, -1.134915738931348e-02, 5.814935976226582e-03, -4.069592039408865e-02, 2.165718171574037e-02, 3.581499959856289e-02, 3.482566189048349e-02, -9.076720236864444e-02, 2.435297937162671e-01, -4.448406108232292e-02, -1.025543084485897e-01,  4.730681746159357e-02, -3.983532506826092e-03, 8.431107870595048e-03, 4.545459214006767e-02, -1.442495609914864e-02, 4.238775650658023e-02, 2.161136860175759e-02, -5.687406819055710e-02, -4.448406108229735e-02, 2.238469405134202e-01, 3.457508449687267e-02, -3.253888960173032e-02,  7.535032973586360e-03, -1.385896083614486e-02, -1.921740669602647e-02, -2.188335076489458e-03,  6.963356822862980e-04, -7.919318515202343e-02, 5.835211019494827e-02, -1.025543084486338e-01,  3.457508449691691e-02, 2.006845640759249e-01, 1.529987293945865e-02, -3.696054673684236e-03, -1.179643780907671e-02, -3.544663744191702e-02, 4.676313606402003e-03, -1.735693530369395e-03, -1.959714773431746e-02, -1.351146566461497e-02, 4.730681746156895e-02, -3.253888960172062e-02,  1.529987293948552e-02, 1.362922776580280e-01, 1.698345805332418e-02, 9.728059581417248e-03,  1.847345414915454e-02, -5.434543696229008e-03, -7.443384090239579e-04, -1.649209240057675e-02,  2.926676515812495e-02, -1.190950814124652e-02, -1.359643545099792e-03, 8.659279060959441e-03, -3.975009483049971e-03, 9.728059581418379e-03, 3.471787296434744e-02, 3.313490219334930e-02, -9.873336461915053e-03, 5.771230181063163e-03, -7.319939750318584e-03, 2.884896370538505e-02,  5.640201838947896e-03, 3.309443658439513e-03, -6.427172557522015e-03, -7.270007080834654e-03,  1.847345414916596e-02, 3.313490219335170e-02, 1.292389106584147e-01, -4.003375057654213e-02,  2.036524341348054e-03, 9.358504553274253e-03, 6.553198893637133e-02, -4.955332138584324e-02,  4.935162306311391e-02, 9.804519875639380e-04, -2.677694293461533e-02, -5.434543696223250e-03, -9.873336461915861e-03, -4.003375057654172e-02, 7.026347869463825e-02, -5.263322976888366e-03, -1.302022336575116e-03, -2.887457409765445e-02, 1.666624188007960e-02, -1.700685017873553e-02, -4.671730469142567e-03, 3.762012343146645e-04, -7.443384090239059e-04, 5.771230181065418e-03,  2.036524341351045e-03, -5.263322976889561e-03, 6.735307590648201e-02, -2.679627597917539e-04, -7.925611887874995e-03, 1.273813095532027e-02, 3.396410865612087e-02, 1.019584506133184e-02, -8.654027498541415e-03, -1.649209240059207e-02, -7.319939750335781e-03, 9.358504553236750e-03, -1.302022336550406e-03, -2.679627597943360e-04, 1.066086993877544e-01, -6.164135892298738e-02,  2.428845926046734e-02, -2.186818482975418e-03, -7.805414040893589e-02, -2.476242037648690e-02,  2.926676515814092e-02, 2.884896370539548e-02, 6.553198893639868e-02, -2.887457409767302e-02, -7.925611887877753e-03, -6.164135892296143e-02, 2.001875305943942e-01, -2.632804481099704e-02, -1.654153955946475e-02, 3.970369509526565e-02, 1.737621329582115e-02, -1.190950814127078e-02,  5.640201838918107e-03, -4.955332138592841e-02, 1.666624188012775e-02, 1.273813095531617e-02,  2.428845926046357e-02, -2.632804481105290e-02, 1.996561480955452e-01, -9.201153867468886e-02, -9.729652718165727e-02, 1.423390035114122e-02, -1.359643545093036e-03, 3.309443658459808e-03,  4.935162306315774e-02, -1.700685017875920e-02, 3.396410865612522e-02, -2.186818482958246e-03, -1.654153955944497e-02, -9.201153867466219e-02, 1.960349448729911e-01, 6.422649001219041e-02, -3.310019006907275e-02, 8.659279060980523e-03, -6.427172557490269e-03, 9.804519876412668e-04, -4.671730469189870e-03, 1.019584506133847e-02, -7.805414040894723e-02, 3.970369509533015e-02, -9.729652718168953e-02, 6.422649001223475e-02, 1.873478612624098e-01, 1.891652884138207e-02, -3.975009483054504e-03, -7.270007080836393e-03, -2.677694293462687e-02, 3.762012343180221e-04, -8.654027498540681e-03, -2.476242037649752e-02, 1.737621329582210e-02, 1.423390035112109e-02, -3.310019006906350e-02, 1.891652884140415e-02, 1.005149084248485e-01, 1.845559309252536e-02,  1.038015356563933e-02, 1.735376944918249e-02, -5.852359270637762e-03, -3.330026465517221e-04, -1.945602503884486e-02, 3.026829811076331e-02, -6.650173096804698e-03, -3.785183172517007e-03,  8.993660866349198e-03, -1.396621701339215e-03, 1.038015356564045e-02, 3.581636054465551e-02,  3.524858452753649e-02, -1.134832544469041e-02, 6.961015716224051e-03, -3.705016353459881e-03,  2.453135313482654e-02, 1.211685162472483e-02, 4.595911500503353e-03, -1.217145645142701e-02, -7.768480017809764e-03, 1.735376944919428e-02, 3.524858452753981e-02, 1.137105016239655e-01, -3.665735224041329e-02, 3.085469208923466e-03, 1.258275432061065e-02, 5.531036923017682e-02, -1.743918679169361e-02, 3.369132856487861e-02, -1.634293821334919e-02, -2.544581846598113e-02, -5.852359270632305e-03, -1.134832544469174e-02, -3.665735224041366e-02, 6.815565613975680e-02, -5.328580021200143e-03, -1.440072426449343e-03, -2.672590428472887e-02, 3.611098775539798e-03, -1.042066489132485e-02, 2.811818019232764e-03, 5.986640995775411e-04, -3.330026465520834e-04,  6.961015716225439e-03, 3.085469208922621e-03, -5.328580021199510e-03, 6.819803222755780e-02, -3.232967061992556e-03, -7.358579351479485e-03, 1.328623991488260e-02, 3.825355401955910e-02,  1.437411347983454e-02, -4.249680286491889e-03, -1.945602503886077e-02, -3.705016353479694e-03,  1.258275432056500e-02, -1.440072426421068e-03, -3.232967061995962e-03, 1.146262709864094e-01, -5.418400713963014e-02, 3.374965746864264e-02, -1.322702451039567e-02, -9.545695866239561e-02, -2.687734650221636e-02, 3.026829811078093e-02, 2.453135313484135e-02, 5.531036923021793e-02, -2.672590428475291e-02, -7.358579351477906e-03, -5.418400713959880e-02, 1.732862450030798e-01, -1.802212748291642e-02,
                  -1.257624757246874e-02, 3.361375681453023e-02, 1.056118496674240e-02, -6.650173096830502e-03, 1.211685162469111e-02, -1.743918679179161e-02, 3.611098775593030e-03,  1.328623991487439e-02, 3.374965746863658e-02, -1.802212748297386e-02, 1.930601671482707e-01, -6.885027988190148e-02, -9.816391798371782e-02, 2.437931119645036e-02, -3.785183172510650e-03,  4.595911500523174e-03, 3.369132856491796e-02, -1.042066489134596e-02, 3.825355401956523e-02, -1.322702451037444e-02, -1.257624757246020e-02, -6.885027988186339e-02, 1.796366964189284e-01,  6.561767852394440e-02, -1.953760176935209e-02, 8.993660866371121e-03, -1.217145645139213e-02, -1.634293821326225e-02, 2.811818019181390e-03, 1.437411347984283e-02, -9.545695866240714e-02,  3.361375681459991e-02, -9.816391798375140e-02, 6.561767852399476e-02, 2.056908091198588e-01,  2.418680102247677e-02, -1.396621701344118e-03, -7.768480017812052e-03, -2.544581846599464e-02,  5.986640995815726e-04, -4.249680286492865e-03, -2.687734650222891e-02, 1.056118496674586e-02,  2.437931119642841e-02, -1.953760176934615e-02, 2.418680102250029e-02, 1.049301675107037e-01,  2.262551078604903e-02, 1.385157897223463e-02, 2.489426533302463e-02, -9.314442202159426e-03,  2.399863370374871e-04, -2.330268607829802e-02, 3.764376793170945e-02, -8.394035271726847e-03, -3.344026040756354e-03, 9.390569236465475e-03, -5.364387408150362e-04, 1.385157897223594e-02,  4.226890079316520e-02, 4.666788347445996e-02, -1.837126768681637e-02, 8.415389297453708e-03, -4.157390835504554e-03, 2.960305394865804e-02, 1.048249670994671e-02, 9.644321079131452e-03, -1.484612593207553e-02, -9.786645367442988e-03, 2.489426533303627e-02, 4.666788347446129e-02,  1.451356874445934e-01, -5.091714065699254e-02, 6.492204997186042e-03, 1.597551567228955e-02,  5.969707948025396e-02, -1.960084876138878e-02, 5.814742586730410e-02, -2.226999917236502e-02, -2.819359262829479e-02, -9.314442202153937e-03, -1.837126768681697e-02, -5.091714065699337e-02,  7.754138789812902e-02, -1.003109196719886e-02, -1.602597454126950e-03, -2.719324226642784e-02, -2.250888711479201e-03, -2.143220745757091e-02, 7.199849219025290e-03, 3.793241956118317e-04,  2.399863370373052e-04, 8.415389297454844e-03, 6.492204997185257e-03, -1.003109196719825e-02,  7.227126168576434e-02, 4.585546454147679e-03, -1.467393476632591e-02, 2.490287958972667e-02,  3.923310771149649e-02, 6.414549448195308e-03, -2.613685191514400e-03, -2.330268607831321e-02, -4.157390835525515e-03, 1.597551567224636e-02, -1.602597454099805e-03, 4.585546454144628e-03,  1.270882125798274e-01, -7.442432397059813e-02, 4.534112139744309e-02, 9.342649864960839e-03, -9.405419447875553e-02, -2.330717880640627e-02, 3.764376793172618e-02, 2.960305394867329e-02,  5.969707948029481e-02, -2.719324226645133e-02, -1.467393476632485e-02, -7.442432397056636e-02,  1.965240731074598e-01, -3.706421247192051e-02, -1.715391204724875e-02, 3.799635807245295e-02,  6.668283444370969e-03, -8.394035271750516e-03, 1.048249670991477e-02, -1.960084876147658e-02, -2.250888711431261e-03, 2.490287958971935e-02, 4.534112139744242e-02, -3.706421247197696e-02,  2.144727067655025e-01, -6.155158193972851e-02, -9.509195950824963e-02, 3.496518746224413e-02, -3.344026040750628e-03, 9.644321079148626e-03, 5.814742586733755e-02, -2.143220745758898e-02,  3.923310771150208e-02, 9.342649864976554e-03, -1.715391204723925e-02, -6.155158193969659e-02,  1.924174722673611e-01, 4.177416357520194e-02, -2.589207008432598e-02, 9.390569236485973e-03, -1.484612593204086e-02, -2.226999917228572e-02, 7.199849218977568e-03, 6.414549448202438e-03, -9.405419447877136e-02, 3.799635807252499e-02, -9.509195950827991e-02, 4.177416357524358e-02,  1.892982068718605e-01, 2.350183014644257e-02, -5.364387408195257e-04, -9.786645367444530e-03, -2.819359262830706e-02, 3.793241956149265e-04, -2.613685191515459e-03, -2.330717880641842e-02,  6.668283444375964e-03, 3.496518746222349e-02, -2.589207008432111e-02, 2.350183014646494e-02,  1.131840443275655e-01, 2.604692538015693e-02, 1.575864195678594e-02, 2.889825279952830e-02, -9.836687318759583e-03, -2.378367990022273e-03, -2.469121495751007e-02, 4.320453465900477e-02, -1.127297746433000e-02, -1.180354520057330e-02, 4.087014556759937e-03, -4.238046846645507e-03,  1.575864195678756e-02, 4.340041731493822e-02, 4.230601324168638e-02, -1.634613072847347e-02,  4.221068438570670e-03, -1.013323260261283e-02, 3.266726603703232e-02, 6.325129038022611e-03, -4.569044502395048e-03, -1.267739430108379e-02, -6.928300429594049e-03, 2.889825279953943e-02,  4.230601324168671e-02, 1.405450050281362e-01, -4.678248794492704e-02, -5.562775719945791e-03,  2.322604781570422e-03, 6.739720591815943e-02, -2.175141982577164e-02, 2.095811374848371e-02, -2.272015183721270e-02, -2.019114993207102e-02, -9.836687318753890e-03, -1.634613072847354e-02, -4.678248794492795e-02, 8.012205976315973e-02, -6.049463170092331e-03, 3.497725795635099e-03, -2.729917374643128e-02, -6.436355324601214e-03, -2.636123504876416e-03, 1.212603175493360e-02, -1.563644072482074e-03, -2.378367990021744e-03, 4.221068438572671e-03, -5.562775719942098e-03, -6.049463170093977e-03, 6.296301970433053e-02, 5.672063236294904e-03, -8.785608836065649e-03,  2.233618682819203e-02, 1.549045656754279e-02, 2.776317087606896e-03, -2.685258699657870e-03, -2.469121495752439e-02, -1.013323260263271e-02, 2.322604781528455e-03, 3.497725795661267e-03,  5.672063236292983e-03, 1.074107200348454e-01, -6.588228303668345e-02, 3.441704756073750e-02,  1.397737622596918e-02, -5.940707277664719e-02, -6.907540027967722e-03, 4.320453465901879e-02,  3.266726603704395e-02, 6.739720591818946e-02, -2.729917374644909e-02, -8.785608836068852e-03, -6.588228303664931e-02, 1.781447273706284e-01, -1.331102200256238e-02, -2.549659382784422e-02,  1.261824593315266e-02, 7.378511419086582e-03, -1.127297746434999e-02, 6.325129037995141e-03, -2.175141982584762e-02, -6.436355324559718e-03, 2.233618682819087e-02, 3.441704756073474e-02, -1.331102200262106e-02, 1.748760308601117e-01, -4.218030420003584e-02, -6.184378299006451e-02,  2.403242127277119e-02, -1.180354520056736e-02, -4.569044502377672e-03, 2.095811374852410e-02, -2.636123504897884e-03, 1.549045656754468e-02, 1.397737622598209e-02, -2.549659382781979e-02, -4.218030420002308e-02, 1.365025022907512e-01, 1.954028315073281e-02, -1.446030303073014e-02,  4.087014556778749e-03, -1.267739430105111e-02, -2.272015183713479e-02, 1.212603175488716e-02,  2.776317087610849e-03, -5.940707277666360e-02, 1.261824593323119e-02, -6.184378299010238e-02,  1.954028315076859e-02, 1.364836530803447e-01, 1.016747613794516e-02, -4.238046846648760e-03, -6.928300429594106e-03, -2.019114993207691e-02, -1.563644072482233e-03, -2.685258699656954e-03, -6.907540027981395e-03, 7.378511419093761e-03, 2.403242127274597e-02, -1.446030303071660e-02,  1.016747613797222e-02, 9.584538303967979e-02, 3.324864776588006e-02, 1.778810932481352e-02,  3.443190065964755e-02, -1.118224248126963e-02, -5.667377994066537e-03, -3.048115014715645e-02,  5.984097845073777e-02, -1.263278053004889e-02, -1.905500423785739e-02, 6.703000413805588e-03,  6.065384995525046e-04, 1.778810932481527e-02, 5.749897133627324e-02, 5.590184325036918e-02, -1.842587463804124e-02, 8.484023657315355e-03, -8.762955710381104e-03, 4.548093940260323e-02,
                  9.983792632777877e-03, -4.049281283220046e-05, -2.053191358034390e-02, -1.320004103279668e-02,  3.443190065965798e-02, 5.590184325036862e-02, 1.647699583982882e-01, -4.866270806822492e-02, -3.889392886040288e-03, 6.787544735315842e-03, 9.133126494275401e-02, -1.547068905240168e-02,  2.449903427960043e-02, -3.125426250516805e-02, -2.032787404813963e-02, -1.118224248126368e-02, -1.842587463804085e-02, -4.866270806822603e-02, 9.161328259158158e-02, -9.598852679686957e-03,  4.524004710067975e-03, -3.371912085764561e-02, -4.962622034111575e-03, -6.943821563010821e-03,  6.910080500059118e-03, -5.677939306824559e-03, -5.667377994065715e-03, 8.484023657317943e-03, -3.889392886034876e-03, -9.598852679689504e-03, 8.370544044411551e-02, -5.397012489524733e-04, -1.322174388894148e-03, 1.636599525715920e-02, 3.492955927531527e-02, 1.451977646455818e-02, -1.044661681279328e-02, -3.048115014717082e-02, -8.762955710400436e-03, 6.787544735271711e-03,  4.524004710094939e-03, -5.397012489535558e-04, 1.268709521896833e-01, -7.913051885323874e-02,  1.254609612558066e-02, 1.958306124013839e-02, -7.155568345559771e-02, -2.223783476745261e-02,  5.984097845075104e-02, 4.548093940261289e-02, 9.133126494278421e-02, -3.371912085766304e-02, -1.322174388900199e-03, -7.913051885320288e-02, 2.413792174641380e-01, 2.894863287203897e-04, -2.789261978813717e-02, 2.230250286183544e-02, 1.329328501898835e-02, -1.263278053006732e-02,  9.983792632752925e-03, -1.547068905247584e-02, -4.962622034071440e-03, 1.636599525716015e-02,  1.254609612557234e-02, 2.894863286675431e-04, 2.023194244032718e-01, -4.458975078533978e-02, -6.037658344540686e-02, 3.545131142792202e-02, -1.905500423785195e-02, -4.049281281492956e-05,  2.449903427964217e-02, -6.943821563032804e-03, 3.492955927531657e-02, 1.958306124015324e-02, -2.789261978811172e-02, -4.458975078532834e-02, 1.831305355588231e-01, 2.559899178083036e-02, -1.518763465098436e-02, 6.703000413823906e-03, -2.053191358031231e-02, -3.125426250508828e-02,  6.910080500012156e-03, 1.451977646456108e-02, -7.155568345561214e-02, 2.230250286191382e-02, -6.037658344544972e-02, 2.559899178086811e-02, 1.804700666357875e-01, 3.553313785560822e-02,  6.065384995498644e-04, -1.320004103279591e-02, -2.032787404814351e-02, -5.677939306825724e-03, -1.044661681279157e-02, -2.223783476746807e-02, 1.329328501899824e-02, 3.545131142789471e-02, -1.518763465096867e-02, 3.553313785563783e-02, 1.470116707057389e-01 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$lag_one_cov)),
               c(NULL )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$fixed_effects)),
               c(numeric(0) )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$n_iter)),
               c(16 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$Q)),
               c( 0.071835375217822975, 0.084582137073013819, 0.223009073678840208, -0.119493427285091769, 0.009466066750369165, -0.105852238869173665, 0.286014878503589087, -0.118975312014415205, 0.072963880690615518, 0.107698095119172083,  0.002872958417635795, 0.084582137073013819, 0.288734819961088074, 0.621729534316541876, -0.303252144247174782,  0.053635411434668392, 0.116885733846190804, 0.442798971229910410, 0.118361568285158841, 0.082773131032791825, -0.315826267425496354, -0.136380517125906014, 0.223009073678840208, 0.621729534316541876, 1.932685945718117804, -0.905791451142566761, 0.085404740049767947, 0.329293839225055951, 1.244877131927203617, -0.650129841073698089,  0.788440144053258907, -0.384243820856056351, -0.601271162264484405, -0.119493427285091769, -0.303252144247174782, -0.905791451142566761, 0.461690041188650646, -0.033998860719253059, -0.096917387322070156, -0.679425626959652584,  0.304430956820253784, -0.307646003939548052, 0.146947657159254730, 0.258853638738512304, 0.009466066750369165,  0.053635411434668392, 0.085404740049767947, -0.033998860719253059, 0.157324742598654699, 0.103871270200124460, -0.228727725719093555, 0.218879327167089854, 0.268755095005345612, -0.067954053092606492, 0.059982261283625368, -0.105852238869173665, 0.116885733846190804, 0.329293839225055951, -0.096917387322070156, 0.103871270200124460,  0.831596377207729498, -0.676116655852594395, 0.681523072886847880, 0.169124544659421366, -1.028206427066912010, -0.179906032010767825, 0.286014878503589087, 0.442798971229910410, 1.244877131927203617, -0.679425626959652584, -0.228727725719093555, -0.676116655852594395, 2.294864225472934116, -1.279880058703267487, -0.150563124713722130,  0.544049670939252605, -0.470788707991677768, -0.118975312014415205, 0.118361568285158841, -0.650129841073698089,  0.304430956820253784, 0.218879327167089854, 0.681523072886847880, -1.279880058703267487, 2.665165772665640365, -0.877979290100542320, -1.612205115048870097, 0.618469895385828417, 0.072963880690615518, 0.082773131032791825,  0.788440144053258907, -0.307646003939548052, 0.268755095005345612, 0.169124544659421366, -0.150563124713722130, -0.877979290100542320, 1.474062654916000525, 0.511708550100707038, -0.227308606612326103, 0.107698095119172083, -0.315826267425496354, -0.384243820856056351, 0.146947657159254730, -0.067954053092606492, -1.028206427066912010,  0.544049670939252605, -1.612205115048870097, 0.511708550100707038, 1.864419874539523292, 0.131108155608886501,  0.002872958417635795, -0.136380517125906014, -0.601271162264484405, 0.258853638738512304, 0.059982261283625368, -0.179906032010767825, -0.470788707991677768, 0.618469895385828417, -0.227308606612326103, 0.131108155608886501,  0.520127724335482933 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$Q_0)),
               c(1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$n_risk)),
               c(2007, 1936, 1882, 1818, 1767, 1656, 1543, 1364, 1254, 1103 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$times)),
               c( 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$risk_set)),
               c(NULL )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$data)),
               c(NULL )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$order)),
               c(1 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$F_)),
               c(1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$method)),
               c("UKF" )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$model)),
               c("exp_clip_time_w_jump" )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$est_Q_0)),
               c(FALSE )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$LR)),
               c(1 )
               , tolerance = 1e-04)
})





test_that("UKF second order model works", {
  for(m in c("logit", exp_model_names)){
    expect_no_error(result <- ddhazard(
      formula = survival::Surv(start, stop, event) ~ group,
      data = head_neck_cancer,
      by = 1, Q_0 = diag(1, 4),
      Q = diag(1e-1, 2),
      control = list(est_Q_0 = F, n_max = 10^4, eps = 10^-3,
                     method = "UKF", save_data = F, save_risk_set = F,
                     beta = 0),
      max_T = 30,
      id = head_neck_cancer$id, order = 2,
      verbose = F,
      model = m
    ))

    set.seed(9999)
    sim_f <- if(m == "logit") test_sim_func_logit else test_sim_func_exp
    sims <- sim_f(n_series = 1e3, n_vars = 3, t_0 = 0, t_max = 20,
                  lambda = 1/4,
                  x_range = 1, x_mean = 0.5, re_draw = T, beta_start = rep(0, 3),
                  intercept_start = -3, sds = c(.1, rep(.25, 3)),
                  tstart_sampl_func = function(...) max(0, runif(1, min = -4, max = 20 - 1)))

    expect_no_error(suppressWarnings(result_sim <-
                      ddhazard(formula = survival::Surv(tstart, tstop, event) ~ . - tstart - tstop - event - id,
                               by = 1,
                               data = sims$res,
                               Q_0 = diag(c(rep(.25, ncol(sims$res) + 1 - 4), rep(.25, ncol(sims$res) + 1 - 4))),
                               Q = diag(rep(.1, ncol(sims$res) + 1 - 4)),
                               control = list(est_Q_0 = F, method = "UKF", eps = 1e-1,
                                              ridge_eps = 1e-3),
                               id = sims$res$id,
                               verbose = F, model = m,
                               order = 2,
                               max_T = 20)))
  }
})



# Had issues with win builder. Thus, these lines
cat("\nFinished", test_name, "\n")
