#' Composed error term distribution
#'
#' Probablitiy density, distribution, quantile function and random number generation for the composed error term distribution.
#'
#' @aliases pcomperr, qcomperr, rcomperr
#'
#' @param x vector of quantiles.
#' @param mu vector of \eqn{\mu}
#' @param sigma_v vector of \eqn{\sigma_V}. Must be positive.
#' @param par_u vector of parameter of the (in)efficiency term. Must be positive.
#' @param s \eqn{s=-1} for production and \eqn{s=1} for cost function.
#' @param family \eqn{normhnorm} for normal-halfnormal and \eqn{normexp} for normal-exponential distribution.
#' @param deriv derivative of order \code{deriv} of the log density. Available are \code{0},\code{2} and \code{4}.
#' @param tri optional, index arrays for upper triangular matrices, generated by \code{\link[mgcv:trind.generator]{trind.generator()}} and supplied to \code{chainrule()}.
#' @param log.p logical; if \code{TRUE}, probabilities p are given as \code{log(p)}.
#' @param check logical; if TRUE, check inputs.
#'
#' @return \code{dcomperr} gives the density, \code{pcomperr} gives the distribution function, \code{qcomperr} gives the quantile function, and \code{rcomperr} generates random numbers, with given parameters. If the derivatives are calculated these are provided as the attributes \code{gradient}, \code{hessian}, \code{l3} and \code{l4} of the output of the density.
#'
#' @details This is wrapper function for the normal-halfnormal and normal-exponential distribution. A random variable \eqn{\mathcal{E}} follows a composed error distribution if \eqn{\mathcal{E} = V + s \cdot U }, where \eqn{V \sim N(\mu, \sigma_V^2)} and \eqn{U \sim HN(0,\sigma_U^2)} or \eqn{U \sim Exp(0,\sigma_U^2)}.
#' For more details see \code{dnormhnorm} and \code{dnormexp}. Here, \eqn{s=-1} for production and \eqn{s=1} for cost function.
#'
#'
#' @examples
#' pdf <- dcomperr(x=seq(-3, 3, by=0.1), mu=1, sigma_v=2, par_u=3, s=-1, family="normhnorm")
#' cdf <- pcomperr(q=seq(-3, 3, by=0.1), mu=1, sigma_v=2, par_u=3, s=-1, family="normhnorm")
#' q <- qcomperr(p=seq(0.1, 0.9, by=0.1), mu=1, sigma_v=2, par_u=3, s=-1, family="normhnorm")
#' r <- rcomperr(n=100, mu=1, sigma_v=2, par_u=3, s=-1, family="normhnorm")
#'
#' @references
#' \itemize{
#' \item \insertRef{aigner1977formulation}{dsfa}
#' \item \insertRef{kumbhakar2015practitioner}{dsfa}
#' \item \insertRef{schmidt2020analytic}{dsfa}
#' \item \insertRef{gradshteyn2014table}{dsfa}
#' \item \insertRef{azzalini2013skew}{dsfa}
#' }
#'
#' @export
#dcomperr
dcomperr<-function(x=0, mu=0, sigma_v=1, par_u=1,  s=-1, family="normhnorm", deriv = 0,  tri=NULL, log.p=FALSE, check=TRUE){
  #Check for correct inputs
  if (family!="normexp"&family!="normhnorm"){
    stop(paste("Wrong input for family", "\n", ""))
  }

  #Initialize out
  out<-NULL

  if(family=="normexp"){
    out<-dnormexp(x=x, mu=mu, sigma_v=sigma_v, lambda=par_u, s=s, deriv = deriv,  tri=tri, log.p = log.p, check=check)
  }

  if(family=="normhnorm"){
    out<-dnormhnorm(x=x, mu=mu, sigma_v=sigma_v, sigma_u=par_u, s=s, deriv = deriv,  tri=tri, log.p = log.p, check=check)
  }

  #Check for correct inputs
  if(is.null(out)){
    stop(paste("Incorrect inputs for dcomperr", "\n", ""))
  }

  return(out)
}

#' @describeIn dcomperr distribution function for the composed error distribution.
#' @param p vector of probabilities.
#' @param lower.tail logical; if TRUE (default), probabilities are \eqn{P[X \le x]}  otherwise, \eqn{P[X > x]}.
#' @export
pcomperr<-function(q, mu=0, sigma_v=1, par_u=1,  s=-1, family="normhnorm", deriv = 0,  tri=NULL, lower.tail = TRUE, log.p=FALSE, check=TRUE){

  #Check for correct inputs
  if (family!="normexp"&family!="normhnorm"){
    stop(paste("Wrong input for family", "\n", ""))
  }

  #Initialize out
  out<-NULL

  if(family=="normexp"){
    out<-pnormexp(q=q, mu=mu, sigma_v=sigma_v, lambda=par_u, s=s, deriv = deriv,  tri=tri, lower.tail = lower.tail, log.p = log.p, check=check)
  }

  if(family=="normhnorm"){
    out<-pnormhnorm(q=q, mu=mu, sigma_v=sigma_v, sigma_u=par_u, s=s, deriv = deriv,  tri=tri, lower.tail = lower.tail, log.p = log.p, check=check)
  }

  #Check for correct inputs
  if(is.null(out)){
    stop(paste("Incorrect inputs for pcomperr", "\n", ""))
  }

  return(out)
}

#' @describeIn dcomperr quantile function for the composed error distribution.
#' @param q vector of quantiles.
#' @export
qcomperr<-function(p, mu=0, sigma_v=1, par_u=1,  s=-1, family="normhnorm", lower.tail = TRUE, log.p=FALSE, check=TRUE){

  #Check for correct inputs
  if (family!="normexp"&family!="normhnorm"){
    stop(paste("Wrong input for family", "\n", ""))
  }

  #Initialize out
  out<-NULL

  if(family=="normexp"){
    out<-qnormexp(p=p, mu=mu, sigma_v=sigma_v, lambda=par_u, s=s, lower.tail = lower.tail, log.p = log.p, check=check)
  }

  if(family=="normhnorm"){
    out<-qnormhnorm(p=p, mu=mu, sigma_v=sigma_v, sigma_u=par_u, s=s, lower.tail = lower.tail, log.p = log.p, check=check)
  }

  #Check for correct inputs
  if(is.null(out)){
    stop(paste("Incorrect inputs for qcomperr", "\n", ""))
  }

  return(out)
}

#' @describeIn dcomperr random number generation for the composed error distribution.
#' @param n number of observations.
#' @export
rcomperr<-function(n, mu=0, sigma_v=1, par_u=1,  s=-1, family="normhnorm", check=TRUE){

  #Check for correct inputs
  if (family!="normexp"&family!="normhnorm"){
    stop(paste("Wrong input for family", "\n", ""))
  }

  #Initialize out
  out<-NULL

  if(family=="normexp"){
    out<-rnormexp(n=n, mu=mu, sigma_v=sigma_v, lambda=par_u, s=s, check=check)
  }

  if(family=="normhnorm"){
    out<-rnormhnorm(n=n, mu=mu, sigma_v=sigma_v, sigma_u=par_u, s=s, check=check)
  }

  #Check for correct inputs
  if(is.null(out)){
    stop(paste("Incorrect inputs for rcomperr", "\n", ""))
  }

  return(out)
}
