#' Summarise and mutate multiple columns.
#'
#' Apply one or more functions to one or more columns. Grouping variables
#' are always excluded from modification.
#'
#' @param tbl a tbl
#' @param funs List of function calls, generated by \code{\link{funs}}, or
#'   a character vector of function names.
#' @param vars,... Variables to include/exclude in mutate/summarise.
#'   You can use same specifications as in \code{\link{select}}.
#'
#'   For standard evaluation versions (ending in \code{_q}) these can
#'   be either a list of expressions or a character vector.
#' @param env The environment in which to evaluate the function calls.
#'   Should only be modified by advanced users.
#' @examples
#' # One function
#' by_species <- iris %>% group_by(Species)
#' by_species %>% summarise_each(funs(length))
#' by_species %>% summarise_each(funs(mean))
#' by_species %>% summarise_each(funs(mean), Petal.Width)
#' by_species %>% summarise_each(funs(mean), matches("Width"))
#'
#' by_species %>% mutate_each(funs(half = . / 2))
#' by_species %>% mutate_each(funs(min_rank))
#'
#' # Two functions
#' by_species %>% summarise_each(funs(min, max))
#' by_species %>% summarise_each(funs(min, max), Petal.Width, Sepal.Width)
#' by_species %>% summarise_each(funs(min, max), matches("Width"))
#'
#' # Alternative function specification
#' iris %>% summarise_each(funs(ul = length(unique(.))))
#' by_species %>% summarise_each(funs(ul = length(unique(.))))
#'
#' by_species %>% summarise_each(c("min", "max"))
#'
#' # Alternative variable specification
#' summarise_each_q(iris, funs(max), names(iris)[-5])
#' summarise_each_q(iris, funs(max), list(quote(-Species)))
#' @export
summarise_each <- function(tbl, funs, ...) {
  env <- parent.frame()
  summarise_each_q(tbl, funs, dots(...), env)
}

#' @export
#' @rdname summarise_each
summarise_each_q <- function(tbl, funs, vars, env = parent.frame()) {
  funs <- as.fun_list(funs, env)

  vars <- colwise_q(tbl, funs, vars)
  call <- as.call(c(list(quote(summarise), quote(tbl)), vars))

  eval_env <- new.env(parent = attr(funs, "env"))
  eval_env$tbl <- tbl

  eval(call, eval_env)
}

#' @export
#' @rdname summarise_each
mutate_each <- function(tbl, funs, ...) {
  env <- parent.frame()
  mutate_each_q(tbl, funs, dots(...), env)
}

#' @export
#' @rdname summarise_each
mutate_each_q <- function(tbl, funs, vars, env = parent.frame()) {
  funs <- as.fun_list(funs)

  vars <- colwise_q(tbl, funs, vars)
  call <- as.call(c(list(quote(mutate), quote(tbl)), vars))

  eval_env <- new.env(parent = env)
  eval_env$tbl <- tbl

  eval(call, eval_env)
}

colwise_q <- function(tbl, calls, vars) {
  stopifnot(is.fun_calls(calls))
  vars <- select_vars_q(tbl_vars(tbl), vars, env = parent.frame(),
    exclude = as.character(groups(tbl)))

  out <- vector("list", length(vars) * length(calls))
  dim(out) <- c(length(vars), length(calls))

  for (i in seq_along(vars)) {
    for (j in seq_along(calls)) {
      out[[i, j]] <- substitute_q(calls[[j]], list(. = as.name(vars[i])))
    }
  }
  dim(out) <- NULL

  if (length(calls) == 1) {
    names(out) <- names(vars)
  } else if (length(vars) == 1) {
    names(out) <- names(calls)
  } else {
    grid <- expand.grid(var = names(vars), call = names(calls))
    names(out) <- paste(grid$var, grid$call, sep = "_")
  }

  out
}

