% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dot.R
\name{dot}
\alias{dot}
\title{Decorrelation by Orthogonal Transformation (DOT)}
\usage{
dot(Z, C, tol.cor = NULL, tol.egv = NULL, ...)
}
\arguments{
\item{Z}{vector of association test statistics (i.e., Z-scores).}

\item{C}{correlation matrix among the association test statistics, as
obtained by \code{\link[=cst]{cst()}}.}

\item{tol.cor}{tolerance threshold for the largest correlation absolute value.}

\item{tol.egv}{tolerance threshold for the smallest eigenvalue.}

\item{...}{additional parameters.}
}
\value{
a  list with
\itemize{
\item{\code{X}:} {association test statistics, de-correlated.}
\item{\code{H}:} {orthogonal transformation, such that \code{X = H \%*\% Z}.}
\item{\code{M}:} {effective number of variants after de-correlation.}
\item{\code{L}:} {effective number of eigenvalues after truncation.}
}
}
\description{
\code{\link[=dot]{dot()}}  decorrelates   genetic  association  test  statistics   by  special
symmetric orthogonal transformation.
}
\details{
Genetic  association studies  typically provide  per-variant test  statistics
that can be  converted to asymptotically normal, signed  Z-scores. Once those
Z-scores are transformed to independent random variables, various methods can
be applied to combine them and obtain SNP-set overall association.

\code{\link[=dot]{dot()}}  uses  per-variant  genetic  association  test  statistics  and  the
correlation among them to decorrelate Z-scores.

To estimate the  correlation among genetic association  test statistics, use
\code{\link[=cst]{cst()}}.  If  P-values and  estimated effects  (i.e, beta  coefficients) are
given instead  of test statistics, \code{\link[=zsc]{zsc()}}  can be used to  recover the test
statistics (i.e., Z-scores).

\code{tol.cor}: variants  with correlation too close  to 1 in absolute  value are
considered to be collinear  and only one of them will  be retained to ensure
that  the  LD  matrix  is   full-rank.   The  maximum  value  for  tolerable
correlation  is  1   -  \code{tol.cor}.  The  default  value   for  \code{tol.cor}  is
\code{sqrt(.Machine$double.eps)}.

\code{tol.egv}: negative and close to  zero eigenvalues are truncated from matrix
\code{D} in \verb{H = EDE'}. The corresponding  columns of \code{E} are also deleted. Note
the  the dimention  of the  square matrix  \code{H} does  not change  after this
truncation. See DOT publication in the  reference below for more details on
definitions  of \code{E}  and \code{D}  matrices.  The  default eigenvalue  tolerance
value is \code{sqrt(.Machine$double.eps)}.

A number of methods are available for combining de-correlated P-values,
see \link{dot_sst} for details.
}
\examples{
## get genotype and covariate matrices
gno <- readRDS(system.file("extdata", 'rs208294_gno.rds', package="dotgen"))
cvr <- readRDS(system.file("extdata", 'rs208294_cvr.rds', package="dotgen"))

## estimate the correlation among association test statistics
sgm <- cst(gno, cvr)

## get the result of genetic association analysis (P-values and effects)
res <- readRDS(system.file("extdata", 'rs208294_res.rds', package="dotgen"))

## recover Z-score statistics
stt <- with(res, zsc(P, BETA))

## decorrelate Z-scores by DOT
result <- dot(stt, sgm)
print(result$X)          # decorrelated statistics
print(result$H)          # orthogonal transformation

## sum of squares of decorrelated statistics is a chi-square
ssq <- sum(result$X^2)
pvl <- 1 - pchisq(ssq, df=result$L)

print(ssq)            # sum of squares = 35.76306
print(pvl)            # chisq P-value =  0.001132132
}
\references{
\href{https://journals.plos.org/ploscompbiol/article?id=10.1371/journal.pcbi.1007819}{
Vsevolozhskaya, O. A., Shi, M., Hu, F., & Zaykin, D. V. (2020). DOT: Gene-set
analysis by combining decorrelated association statistics. PLOS Computational
Biology, 16(4), e1007819.}
}
\seealso{
\code{\link[=cst]{cst()}}, \code{\link[=zsc]{zsc()}}, \link{dot_sst}
}
