#' Summary function for crossnma object
#' 
#' @description
#' This function creates posterior summary statistics for the fitted
#' cross network meta-analysis / meta-regression model
#' 
#' @param object An object generated by the \code{\link{crossnma}}.
#' @param quantiles A numeric vector of probabilities to present
#'   posterior summaries. The default value is c(0.025, 0.5, 0.975)
#'   for the 95\% credible interval and the median.
#' @param exp A logical value indicating whether to exponentiate the
#'   parameters of relative treatment effect and covariate
#'   effect. Default is TRUE.
#' @param \dots Additional arguments to be passed to summary()
#'   function
#' 
#' @return \code{crossnma.summary} returns a matrix containing the
#'   following summary statistics (in columns) for each estimated
#'   parameter:
#' @return \code{Mean} the mean of the posterior distribution
#' @return \code{SD} the standard deviation of the posterior
#'   distribution
#' @return \code{2.5\%} (default) the 2.5\% quantile of the posterior
#'   distribution (the lower bound of the 95\% credible interval)
#' @return \code{50\%} (default) the median of the posterior
#'   distribution
#' @return \code{97.5\%} (default) the 97.5\% quantile of the
#'   posterior distribution (the upper bound of the 95\% credible
#'   interval)
#' @return \code{Rhat} Gelman-Rubin statistic. The further the value
#'   of Rhat from 1, the worse the mixing of chains and so the
#'   convergence.
#' @return \code{n.eff} An estimate of the effective sample size. The
#'   smaller the value of n.eff the greater the uncertainty associated
#'   with the corresponding parameter.
#' 
#' @author Tasnim Hamza \email{tasnim.hamza@@ispm.unibe.ch}
#'
#' @seealso \code{\link{print.summary.crossnma}}
#'
#' @examples
#' # We conduct a network meta-analysis assuming a random-effects
#' # model.
#' # The data comes from randomized-controlled trials and
#' # non-randomized studies (combined naively)
#' head(ipddata) # participant-level data
#' head(stddata) # study-level data
#'
#' # Create a JAGS model
#' mod <- crossnma.model(treat, id, relapse, n, design,
#'   prt.data = ipddata, std.data = stddata,
#'   reference = "A", trt.effect = "random", method.bias = "naive")
#'
#' # Fit JAGS model
#' # (suppress warning 'Adaptation incomplete' due to n.adapt = 20)
#' fit <-
#'   suppressWarnings(crossnma(mod, n.adapt = 20,
#'     n.iter = 50, thin = 1, n.chains = 3))
#'
#' # Display the output
#' summary(fit)
#'
#' @method summary crossnma
#' @export


summary.crossnma <- function(object, quantiles = c(0.025, 0.5, 0.975),
                             exp = TRUE, ...) {

  ## initial
  sum.fit <- summary((object$samples), quantiles = quantiles, ...)
  ## mean and sd
  mat <- sum.fit$statistics[, c('Mean','SD')]
  ## add quantiles
  mat <- cbind(mat, sum.fit$quantiles)
  ##
  if (exp)
    mat[row.names(mat)[!startsWith(rownames(mat), "tau")], -2] <-
      exp(mat[row.names(mat)[!startsWith(rownames(mat), "tau")], -2])
  ## add Rhat
  Rhat <- round(rhat(object$samples)$psrf[, 'Point est.'], 3)
  mat <- cbind(mat, Rhat)

  ## add number of effective sample size
  n.eff <- round(effectiveSize(object$samples))
  mat <- cbind(mat, n.eff)

  ## attach treatment names to basic parameters (d's)
  row.names(mat)[startsWith(rownames(mat), "d")] <-
    paste("d", as.character(object$model$trt.key$trt.ini), sep = ".")

  class(mat) <- c("summary.crossnma", class(mat))
  attr(mat, "exp") <- exp
  ##
  mat
}
