% File responsivityMetrics.Rd
\encoding{UTF-8}

\name{responsivityMetrics}
\title{Compute Metrics for a Light Responder (e.g. a camera) or a Material Responder (e.g. a scanner)}
\alias{responsivityMetrics}
\alias{responsivityMetrics.colorSpec}

\description{
This function computes a few technical metrics regarding some
geometric objects related to a responder:
the spherical chromaticity polygon, cone, convex cone, and color-solid.

Currently the function only works if the number of spectra in \code{x} is 3 (e.g. RGB or XYZ).
In this case the rows of \code{as.matrix(x)} (after weighting by step size)
are called the \emph{generators}; they are vectors in \eqn{R^3} 
and we require that they are all in some open linear halfspace (unless a generator is 0).
The 0-based rays through the generators intersect a plane inside the halfspace to form the vertices
of the \emph{chromaticity polygon} \eqn{P}.
The 0-based rays through points of the interior of \eqn{P} form a cone,
and the convex hull of this cone is a convex cone.
The central projection of \eqn{P} onto the unit sphere is the \emph{spherical chromaticity polygon} \eqn{P_S}.
If \code{type} is \code{'responsivity.material'}, then \code{x} has an
\emph{object-color solid} or \emph{Rösch Farbkörper},
which is a zonohedron \eqn{Z}.
See \cite{Centore} and vignette
\href{../doc/convexity.pdf}{\bold{Convexity and Transitions}}
for details.
\cr
Some simplification of the generators is performed during pre-processing.
Generators that are 0 (in all channels) are removed,
and a group of generators that are all positive multiples of each other is replaced by their sum.
The 3-vectors are called the \emph{condensed generators}.
These simplifications do not change any of the geometric objects defined above.
}


\usage{
\S3method{responsivityMetrics}{colorSpec}( x )

}

\arguments{
\item{x}{a \bold{colorSpec} object with \code{type}
equal to \code{'responsivity.light'} or \code{'responsivity.material'}, and 3 spectra}
}


\value{
\code{responsivityMetrics()} returns a list with these items:

\item{generators}{
a pair of integers, the 1st is the number of original generators,
and the 2nd is the number of condensed generators
}

\item{zeros}{vector of wavelengths at which the responsivity is 0 (in all 3 channels)}

\item{multiples}{a list of vectors of wavelengths; the responsivities in each vector (group) are positive multiples of each other}

\item{salient}{a logical where TRUE means that there is some open linear halfspace that contains all the non-zero generators.  If all the responsivities are non-negative, which is the usual case, then \code{salient=TRUE}.}

\item{normal}{If \code{salient=TRUE}, then the inward pointing unit normal for the previous halfspace.
Otherwise, \code{normal=NA}.}

If \code{salient=TRUE}, then the list also contains:

\item{concavities}{a \code{data.frame} with 2 columns: \code{wavelength} and \code{extangle},
where \code{extangle} is the external angle at the wavelength (for the spherical chromaticity polygon \eqn{P_S}), 
and is negative.
A negative angle means that \eqn{P_S} is concave at that vertex.}

\item{coneangle}{the solid angle of the cone generated by the generators.
This is identical to the area of the spherical chromaticity polygon, with concavities preserved.
}
  
\item{cxconeangle}{the solid angle of the convex cone generated by the generators, with no concavities.
This is identical to the area of the convex hull of the spherical chromaticity polygon.
If all responsivities are non-negative, which is the usual case, then this solid angle is less than
the solid angle of an octant, which is \eqn{\pi/2}.
We always have \code{coneangle <= cxconeangle}.
}
    
If the type of x is \code{'responsivity.material'} then the list also contains:    
  
\item{area}{the surface area of the object-color solid of \code{x}}

\item{volumesch}{the volume enclosed by the Schrödinger colors of \code{x}}
  
\item{volumeopt}{the volume of the object-color solid of \code{x}.
We always have \code{volumesch <= volumeopt}.
}

In case of global error, the function returns \code{NULL}.
}


\note{
To determine the value of \code{salient}, the package \bold{quadprog} might be required.
}

\seealso{
\code{\link{type}},
vignette \href{../doc/convexity.pdf}{\bold{Convexity and Transitions}}
}

\references{
Centore, Paul.
\emph{A zonohedral approach to optimal colours}.
\bold{Color Research & Application}.
Vol. 38.
No. 2.
pp. 110-119.
April 2013.

}


\examples{
D65.eye = product( D65.5nm, 'varmat', xyz1931.1nm, wave=wavelength(D65.5nm) )
responsivityMetrics( D65.eye )

##  $generators
##  [1] 81 65
##  
##  $zeros
##  numeric(0)
##  
##  $multiples
##  $multiples[[1]]
##   [1] 700 705 710 715 720 725 730 735 740 745 750 755 760 765 770 775 780
##  
##  $salient
##  [1] TRUE

##  $normal
##  [1] 0.5773503 0.5773503 0.5773503

##  $concavities
##     wavelength      extangle
##  2         385 -0.3870584861
##  4         395 -0.4393794850
##  13        440 -0.0255522232
##  14        445 -0.0075938225
##  41        580 -0.0010472855
##  42        585 -0.0006600542
##  44        595 -0.0017602134
##  45        600 -0.0004783000
##  46        605 -0.0026858850
##  49        620 -0.0037210312
##  53        640 -0.0011878846
##  54        645 -0.0029247630
##  
##  $coneangle
##  [1] 1.185259
##  
##  $cxconeangle
##  [1] 1.185282
##  
##  $area
##  [1] 39873.89
##  
##  $volumesch
##  [1] 511128.5
##
##  $volumeopt
##  [1] 511129.5
}




\keyword{colorSpec}
