% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/encoding.R
\name{compute_optimal_encoding}
\alias{compute_optimal_encoding}
\title{Compute the optimal encoding for each state}
\usage{
compute_optimal_encoding(
  data,
  basisobj,
  computeCI = TRUE,
  nBootstrap = 50,
  propBootstrap = 1,
  nCores = max(1, ceiling(detectCores()/2)),
  verbose = TRUE,
  ...
)
}
\arguments{
\item{data}{data.frame containing \code{id}, id of the trajectory, \code{time}, time at which a change occurs and \code{state}, associated state. All individuals must begin at the same time T0 and end at the same time Tmax (use \code{\link{cut_data}}).}

\item{basisobj}{basis created using the \code{fda} package (cf. \code{\link{create.basis}}).}

\item{computeCI}{if TRUE, perform a bootstrap to estimate the variance of encoding functions coefficients}

\item{nBootstrap}{number of bootstrap samples}

\item{propBootstrap}{size of bootstrap samples relative to the number of individuals: propBootstrap * number of individuals}

\item{nCores}{number of cores used for parallelization. Default is the half of cores.}

\item{verbose}{if TRUE print some information}

\item{...}{parameters for \code{\link{integrate}} function (see details).}
}
\value{
A list containing:
\itemize{
  \item \code{eigenvalues} eigenvalues
  \item \code{alpha} optimal encoding coefficients associated with each eigenvectors
  \item \code{pc} principal components
  \item \code{F} matrix containing the \eqn{F_{(x,i)(y,j)}}
  \item \code{V} matrix containing the \eqn{V_{(x,i)}}
  \item \code{G} covariance matrix of \code{V}
  \item \code{basisobj} \code{basisobj} input parameter
  \item \code{pt} output of \link{estimate_pt} function
  \item \code{bootstrap} Only if \code{computeCI = TRUE}. Output of every bootstrap run
  \item \code{varAlpha} Only if \code{computeCI = TRUE}. Variance of alpha parameters
  \item \code{runTime} Total elapsed time
}
}
\description{
Compute the optimal encoding for categorical functional data using an extension of the multiple correspondence analysis to a stochastic process.
}
\details{
See the vignette for the mathematical background: \code{RShowDoc("cfda", package = "cfda")}

Extra parameters (\emph{...}) for the \code{\link{integrate}} function can be:
\itemize{
 \item \emph{subdivisions} the maximum number of subintervals.
 \item \emph{rel.tol} relative accuracy requested.
 \item \emph{abs.tol} absolute accuracy requested.
}
}
\examples{
# Simulate the Jukes-Cantor model of nucleotide replacement
K <- 4
Tmax <- 5
PJK <- matrix(1 / 3, nrow = K, ncol = K) - diag(rep(1 / 3, K))
lambda_PJK <- c(1, 1, 1, 1)
d_JK <- generate_Markov(
  n = 10, K = K, P = PJK, lambda = lambda_PJK, Tmax = Tmax,
  labels = c("A", "C", "G", "T")
)
d_JK2 <- cut_data(d_JK, Tmax)

# create basis object
m <- 5
b <- create.bspline.basis(c(0, Tmax), nbasis = m, norder = 4)

# compute encoding
encoding <- compute_optimal_encoding(d_JK2, b, computeCI = FALSE, nCores = 1)
summary(encoding)

# plot the optimal encoding
plot(encoding)

# plot the two first components
plotComponent(encoding, comp = c(1, 2))

# extract the optimal encoding
get_encoding(encoding, harm = 1)
}
\references{
\itemize{
  \item Deville J.C. (1982) Analyse de données chronologiques qualitatives : comment analyser des calendriers ?, Annales de l'INSEE, No 45, p. 45-104.
  \item Deville J.C. et  Saporta G. (1980) Analyse harmonique qualitative, DIDAY et al. (editors), Data Analysis and Informatics, North Holland, p. 375-389.
  \item Saporta G. (1981) Méthodes exploratoires d'analyse de données temporelles, Cahiers du B.U.R.O, Université Pierre et Marie Curie, 37-38, Paris.
}
}
\seealso{
\link{plot.fmca} \link{print.fmca} \link{summary.fmca} \link{plotComponent} \link{get_encoding}
}
\author{
Cristian Preda, Quentin Grimonprez
}
