% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/stats.R
\name{run_sea}
\alias{run_sea}
\title{Perform superposed epoch analysis.}
\usage{
run_sea(x, key, years_before = 6, years_after = 4,
  time_span = c("key_period"), n_iter = 1000)
}
\arguments{
\item{x}{A data.frame climate reconstruction or tree-ring series with row names as years.}

\item{key}{A vector of event years for superposed epoch, such as fire years, or an fhx object
with a single \code{series} as produced by \code{composite}}

\item{years_before}{The number of lag years prior to the event year}

\item{years_after}{The number of lag years following the event year}

\item{time_span}{The length of the x time series to use. Defaults to "key_period"
which constrains the time series to the time period of key events; "all" will use the entire
time series}

\item{n_iter}{The number of iterations for bootstrap resampling}
}
\value{
A list of three data frames, following the output of EVENT.
(1) the actual events table, (2) the simulated events table, and (3) departures of actual from simulated
}
\description{
Perform superposed epoch analysis.
}
\details{
Superposed epoch analysis (SEA) helps to evaluate fire-climate
relationships in studies of tree-ring fire history. It works by compositing the values of
an anual timeseries or climate reconstruction for the fire years provided (\code{key}) and both positive and
negative lag years. Bootstrap resampling of the timeseries is performed to evaluate the statistical
significance of each year's mean value. Users interpret the departure of the actual event year
means from the simulated event year means.

The significance of lag-year departures from the average climate condition was first noted by
Baisan and Swetnam (1990) and used in an organized SEA by Swetnam (1993). Since then, the procedure
has been commonly applied in fire history studies. The FORTRAN program EVENT.exe was written by
Richard Holmes and Thomas Swetnam (Holmes and Swetnam 1994) to perform SEA for fire history
specifically. EVENT was incorporated in the FHX2 software by Henri Grissino-Mayer.

run_sea was designed to replicate EVENT as closely as possible. We have tried to stay true to their implementation of
SEA, although multiple versions of the analysis exist in the climate literature and for fire
history (e.g., FHAES implements a diferent procedure). The outcome of EVENT and run_sea should
only differ slightly in the values of the simulated events and the departures, because random
draws are used. The event year and lag significance levels should match, at least in the general
pattern.

We note that our implementation of run_sea borrows from the \code{dplR:::sea} function in how it performs
the bootstrap procedure, but differs in the kind of output provided for the user.
}
\examples{
\dontrun{
# Read in the Cook and Krusic (2004; The North American Drought Atlas) reconstruction
# of Palmer Drought Severity Index (PDSI) for the Jemez Mountains area (gridpoint 133).
target_url <- paste0('http://iridl.ldeo.columbia.edu',
                     '/SOURCES/.LDEO/.TRL/.NADA2004'
                     '/pdsiatlashtml/pdsiwebdata/1050w_350n_133.txt')
pdsi <- read.table(target_url, header = TRUE, row.names = 1)
pdsi <- subset(pdsi, select = "RECON")

# Run SEA on Peggy Mesa (pgm) data
data(pgm)
(pgm.comp <- composite(pgm))

(pgm.sea <- run_sea(pdsi, pgm.comp))

# Make a bargraph with confidence intervals
par(mar=c(2, 3, 1, 1), oma=c(3, 3, 1, 1))
bp <- barplot(pgm.sea[[3]]$mean_value,
              col=c(rep("grey75", 3),"grey45", "grey30",
                    "grey75", "grey30", rep("grey75", 4)),
              ylab = '', las=1, cex.axis=1.3, cex=1.3, ylim=c(-2, 2))
axis(1, at=bp, labels = -6:4, tick=FALSE, cex.axis=1.3)
lines(bp, pgm.sea[[3]]$lower_95_perc, lwd=2, lty=2)
lines(bp, pgm.sea[[3]]$upper_95_perc, lwd=2, lty=2)
lines(bp, pgm.sea[[3]]$lower_99_perc, lwd=2, lty=3)
lines(bp, pgm.sea[[3]]$upper_99_perc, lwd=2, lty=3)
mtext(expression(bold('PDSI departure')), side=2, line=2.2, cex=1.5)
mtext(expression(bold('Lag year')), side=1, line=3.3, cex=1.5)
}
\dontrun{
# For users who want to perform SEA very near to EVENT.exe and/or have reproducable draws from
# the bootstrap procedure, consider including the \\code{set.seed} function prior to \\code{run_sea}.
# Convention is to provide a long integer, such as a birthday (e.g. 3191982).
# In the EVENT.exe program, Richard Holmes used the number of days since 1 January 1935.
days <- as.numeric(Sys.Date() - as.Date("1jan1935", "\%d\%b\%Y"))
set.seed(days)
}

}
\references{
Baisan and Swetnam 1990, Fire history on desert mountain range: Rincon Mountain Wilderness, Arizona, U.S.A. Canadian Journal of Forest Research 20:1559-1569.

Bunn 2008, A dendrochronology program library in R (dplR), Dendrochronologia 26:115-124

Holmes and Swetnam 1994, EVENT program desription

Swetnam 1993, Fire history and climate change in giant sequoia groves, Science 262:885-889.
}
