#' Calculate graph vulnerability
#'
#' This function calculates the \emph{vulnerability} of the vertices of a graph.
#' Here, vulnerability is considered to be the proportional drop in global
#' efficiency when a given node is removed from the graph. The vulnerability of
#' the graph is considered the maximum across all vertices.
#'
#' @param g An \code{igraph} graph object
#' @param use.parallel Logical indicating whether or not to use \emph{foreach}
#'   (default: \code{TRUE})
#' @param weighted Logical indicating whether weighted efficiency should be
#'   calculated (default: \code{FALSE})
#' @export
#' @importFrom Matrix rowSums
#'
#' @return A numeric vector of length equal to the vertex count of \emph{g}
#'
#' @seealso \code{\link{efficiency}}
#' @author Christopher G. Watson, \email{cgwatson@@bu.edu}
#' @references Latora V., Marchiori M. (2005) \emph{Variability and protection
#' of infrastructure networks}. Physical Review E, 71:015103.

vulnerability <- function(g, use.parallel=TRUE, weighted=FALSE) {
  stopifnot(is_igraph(g))
  if (isTRUE(weighted)) {
    weights <- NULL
    g.attr <- 'E.global.wt'
    A <- as_adj(g, sparse=FALSE, names=FALSE, attr='weight')
  } else {
    weights <- NA
    g.attr <- 'E.global'
    A <- as_adj(g, sparse=FALSE, names=FALSE)
    weighted <- NULL
  }
  if (!g.attr %in% graph_attr_names(g)) {
    E.global <- efficiency(g, 'global', weights)
  } else {
    E.global <- graph_attr(g, g.attr)
  }

  vuln <- rep(0, nrow(A))
  nodes <- which(rowSums((A > 0) + 0) != 0)
  if (isTRUE(use.parallel)) {
    vuln[nodes] <- foreach(i=nodes, .combine='c') %dopar% {
      g.tmp <- graph_from_adjacency_matrix(A[-i, -i], mode='undirected', weighted=weighted)
      E.global.tmp <- efficiency(g.tmp, 'global', weights)
      1 - (E.global.tmp / E.global)
    }
  } else {
    for (i in nodes) {
      g.tmp <- graph_from_adjacency_matrix(A[-i, -i], mode='undirected', weighted=weighted)
      E.global.tmp <- efficiency(g.tmp, 'global', weights)
      vuln[i] <- 1 - (E.global.tmp / E.global)
    }
  }
  return(vuln)
}
