% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mcount.R
\name{mcount}
\alias{mcount}
\title{Count bipartite motifs}
\usage{
mcount(M, six_node, normalise)
}
\arguments{
\item{M}{A numeric matrix representing interactions between two groups of nodes. Each row corresponds to a node in one level
and each column corresponds to a node in the other level. Elements of M are positive numbers if nodes do interact, and 0
otherwise. Formally, M is an incidence matrix. When nodes i and j interact, m_ij > 0; if they do not interact, m_ij = 0.
If interactions are weighted (matrix elements are greater than 1), the function will automatically convert the matrix to a binary
matrix.}

\item{six_node}{Logical; should six node motifs be counted?}

\item{normalise}{Logical; should motif frequencies be normalised to control for network size?}
}
\value{
Returns a data frame with one row for each motif: either 17 rows (if \code{six_node} = FALSE) or 44 rows (if \code{six_node} = TRUE). The data frame has three columns.
The first column ("motif") indicates the motif ID as described in Simmons et al. (2017) (and originally in Appendix 1 of Baker et al. (2015)). The second column
("nodes") indicates how many nodes the motif contains. The third column ("frequency") is the number of times each motif appears in the network.

If \code{normalise} = TRUE, three additional columns are added to the output data frame, each corresponding to a different method of normalising motif
frequencies as described above.
}
\description{
Counts occurrences of motifs in a bipartite network
}
\details{
Counts the number of times each of the 17 motifs up to five nodes (if \code{six_node} = FALSE), or 44 motifs up to six nodes (if \code{six_node} = TRUE), occurs in a network.

Larger networks tend to contain more motifs. Controlling for this effect by normalising motif counts is important if different sized networks are being compared.
If \code{normalise} = TRUE, motif frequencies are normalised in three ways. The first method ("normalise_sum") converts each frequency to a relative frequency by expressing counts as
a proportion of the total number of motifs in the network. The second method ("normalise_sizeclass") uses a similar approach, but expresses counts as a proportion of the total number of
motifs within each motif size class (the number of nodes a motif contains). For example, the relative frequency of all two-node motifs will sum to one,
as will the relative frequency of all three-, four-, five- and six-node motifs. The final method ("normalise_nodesets") expresses frequencies as the number
of node sets that are involved in a motif as a proportion of the number of node sets that could be involved in that motif (Poisot and Stouffer, 2017). For example, in a motif
with three nodes in one level (A) and two nodes in the other level (P), the maximum number of node sets which could be involved in the motif is
given by the product of binomial coefficients, choosing three nodes from A and two from P.

Warning: including six node motifs is fine for most networks. However, for large networks, counting six node motifs can be slow and memory intensive. In some cases, R can crash if there is not enough memory.
}
\examples{
set.seed(123)
row <- 10
col <- 10
m <- matrix(sample(0:1, row*col, replace=TRUE), row, col)
mcount(M = m, six_node = TRUE, normalise = TRUE)
}
\references{
Baker, N., Kaartinen, R., Roslin, T., and Stouffer, D. B. (2015). Species’ roles in food webs show fidelity across a highly variable oak forest. Ecography, 38(2):130–139.

Poisot, T. & Stouffer, D. (2016). How ecological networks evolve. bioRxiv.

Simmons, B I., Sweering, M. J. M., Dicks, L. V., Sutherland, W. J. and Di Clemente, R. bmotif: a package for counting motifs in bipartite networks
}
