#' Document User Notice Stage in BID Framework
#'
#' @description
#' This function documents the observation and problem identification
#' stage. It represents stage 2 in the BID framework and now returns a
#' structured bid_stage object with enhanced metadata and external mapping
#' support.
#'
#' @param previous_stage A tibble or list output from the previous BID stage
#'        function (typically bid_interpret).
#' @param problem A character string describing the observed user problem.
#' @param theory A character string describing the behavioral theory that might
#'        explain the problem. If NULL, will be auto-suggested using external
#'        theory mappings.
#' @param evidence A character string describing evidence supporting the
#'        problem.
#' @param quiet Logical indicating whether to suppress informational messages.
#'        If NULL, uses getOption("bidux.quiet", FALSE).
#' @param ... Additional parameters. Deprecated parameters (e.g.,
#'        'target_audience') will generate warnings if provided.
#'
#' @return A bid_stage object containing the documented information for the
#'         "Notice" stage with enhanced metadata and validation.
#'
#' @examples
#' interpret_result <- bid_interpret(
#'   central_question = "How can we improve user task completion?",
#'   data_story = list(
#'     hook = "Users are struggling with complex interfaces",
#'     resolution = "Simplify key interactions"
#'   )
#' )
#'
#' # Auto-suggested theory
#' bid_notice(
#'   previous_stage = interpret_result,
#'   problem = "Users struggling with complex dropdowns and too many options",
#'   evidence = "User testing shows 65% abandonment rate on filter selection"
#' )
#'
#' # With explicit theory
#' notice_result <- bid_notice(
#'   previous_stage = interpret_result,
#'   problem = "Mobile interface is difficult to navigate",
#'   theory = "Fitts's Law",
#'   evidence = "Mobile users report frustration with small touch targets"
#' )
#'
#' summary(notice_result)
#'
#' @export
bid_notice <- function(
    previous_stage,
    problem,
    theory = NULL,
    evidence = NULL,
    quiet = NULL,
    ...) {
  # handle deprecated target_audience parameter via ...
  dots <- list(...)
  target_audience <- NULL
  if ("target_audience" %in% names(dots)) {
    cli::cli_warn(c(
      "!" = "The 'target_audience' parameter has been deprecated and removed.",
      "i" = "Audience information should now be managed through the bid_interpret() stage.",
      "i" = "This parameter will be ignored in this version."
    ))
    target_audience <- dots$target_audience
    # remove from dots to avoid issues
    dots$target_audience <- NULL
  }

  # check for any other unexpected parameters
  if (length(dots) > 0) {
    unexpected_params <- names(dots)
    cli::cli_warn(c(
      "!" = "Unexpected parameters provided: {paste(unexpected_params, collapse = ', ')}",
      "i" = "These will be ignored."
    ))
  }

  # standardized parameter validation
  validate_bid_stage_params(
    previous_stage,
    "Notice",
    list()
  )
  validate_character_param(problem, "problem", min_length = 1)
  validate_character_param(evidence, "evidence", min_length = 1)
  validate_character_param(theory, "theory", allow_null = TRUE)

  # clean input for processing
  problem_clean <- trimws(problem)
  evidence_clean <- trimws(evidence)

  if (nchar(problem_clean) < 10) {
    warning(
      paste(
        paste(
          "Problem description is very short (< 10 characters).",
          "Consider providing more detail."
        )
      ),
      call. = FALSE
    )
  }
  if (nchar(evidence_clean) < 10) {
    warning(
      paste(
        "Evidence description is very short (< 10 characters).",
        "Consider providing more detail."
      ),
      call. = FALSE
    )
  }

  auto_suggested_theory <- FALSE
  theory_confidence <- 1.0

  if (is.null(theory)) {
    suggestion_result <- .suggest_theory_from_text(
      problem_clean,
      evidence_clean,
      mappings = NULL,
      show_message = TRUE
    )

    theory <- suggestion_result$theory
    theory_confidence <- suggestion_result$confidence
    auto_suggested_theory <- suggestion_result$auto_suggested
  }

  # generate contextual suggestions using unified system
  context_data <- list(
    problem = problem_clean,
    evidence = evidence_clean,
    theory = theory
  )
  suggestions <- generate_stage_suggestions("Notice", context_data)

  # create result tibble
  result_data <- tibble::tibble(
    stage = "Notice",
    problem = problem, # use original problem to preserve exact string
    theory = theory %||% NA_character_,
    evidence = evidence, # use original evidence to preserve exact string
    suggestions = suggestions,
    timestamp = .now()
  )

  # create comprehensive metadata using standardized helper
  metadata <- get_stage_metadata(
    2,
    list(
      auto_suggested_theory = auto_suggested_theory,
      theory_confidence = theory_confidence,
      problem_length = nchar(problem_clean),
      evidence_length = nchar(evidence_clean),
      custom_mappings_used = FALSE
    )
  )

  # create and validate bid_stage object
  result <- bid_stage("Notice", result_data, metadata)

  # enhanced user feedback with progress tracking
  bid_message(
    "Stage 2 (Notice) completed. (40% complete)",
    glue::glue("Problem: {truncate_text(problem_clean, 60)}"),
    glue::glue("Theory: {theory}{if (auto_suggested_theory) ' (auto-suggested)' else ''}"),
    glue::glue("Evidence: {truncate_text(evidence_clean, 60)}"),
    if (auto_suggested_theory) {
      glue::glue("Theory confidence: {round(theory_confidence * 100)}%")
    },
    "Next: Use bid_anticipate() for Stage 3",
    quiet = quiet
  )

  return(result)
}

# legacy function name support for backward compatibility
suggest_theory_from_problem <- function(problem, evidence = NULL) {
  .Deprecated(
    "suggest_theory_from_mappings",
    msg = paste(
      "suggest_theory_from_problem is deprecated.",
      "Use suggest_theory_from_mappings instead."
    )
  )

  suggest_theory_from_mappings(problem, evidence, mappings = NULL)
}
