% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/stan_models.R
\name{stan_sarima}
\alias{stan_sarima}
\title{Fitting a Multiplicative Seasonal ARIMA model.}
\usage{
stan_sarima(
  ts,
  order = c(1, 0, 0),
  seasonal = c(0, 0, 0),
  xreg = NULL,
  period = 0,
  chains = 4,
  iter = 4000,
  warmup = floor(iter/2),
  adapt.delta = 0.9,
  tree.depth = 10,
  stepwise = TRUE,
  prior_mu0 = NULL,
  prior_sigma0 = NULL,
  prior_ar = NULL,
  prior_ma = NULL,
  prior_sar = NULL,
  prior_sma = NULL,
  prior_breg = NULL,
  series.name = NULL,
  ...
)
}
\arguments{
\item{ts}{a numeric or ts object with the univariate time series.}

\item{order}{A specification of the non-seasonal part of the ARIMA model: the
three components (p, d, q) are the AR order, the number of differences, and the
MA order.}

\item{seasonal}{A specification of the seasonal part of the ARIMA model,same as
order parameter:  the three components (p, d, q) are the seasonal AR order,
the degree of seasonal differences, and the seasonal MA order.}

\item{xreg}{Optionally, a numerical matrix of external regressors,
which must have the same number of rows as ts. It should not be a data frame.}

\item{period}{an integer specifying the periodicity of the time series by
default the value frequency(ts) is used.}

\item{chains}{An integer of the number of Markov Chains chains to be run,
by default 4 chains are run.}

\item{iter}{An integer of total iterations per chain including the warm-up,
by default  the number of iterations are 2000.}

\item{warmup}{A positive integer specifying number of warm-up (aka burn-in)
iterations. This also specifies the number of iterations used for step-size
adaptation, so warm-up samples should not be used for inference. The number
of warmup should not be larger than \code{iter} and the default is
\code{iter/2}.}

\item{adapt.delta}{An optional real value between 0 and 1, the thin of the jumps
in a HMC method. By default is 0.9.}

\item{tree.depth}{An integer of the maximum  depth of the trees  evaluated
during each iteration. By default is 10.}

\item{stepwise}{If TRUE, will do stepwise selection (faster). Otherwise, it searches
over all models. Non-stepwise selection can be very slow, especially for seasonal models.}

\item{prior_mu0}{The prior distribution for the location parameter in an ARIMA model. By default
the value is set \code{NULL}, then the default student(7,0,1) prior is used.}

\item{prior_sigma0}{The prior distribution for the scale parameter in an ARIMA model. By default
the value is set \code{NULL}, then the default student(7,0,1) prior is used.}

\item{prior_ar}{The prior distribution for the auto-regressive parameters in an ARIMA model.
By default the value is set \code{NULL}, then the default normal(0,0.5) priors are used.}

\item{prior_ma}{The prior distribution for the moving average parameters in an ARIMA model.
By default the value is set \code{NULL}, then the default normal(0,0.5) priors are used.}

\item{prior_sar}{The prior distribution for the seasonal auto-regressive parameters in a
SARIMA model. By default the value is set \code{NULL}, then the default normal(0,0.5) priors
are used.}

\item{prior_sma}{The prior distribution for the seasonal moving average parameters in a
SARIMA model. By default the value is set \code{NULL}, then the default normal(0,0.5) priors
are used.}

\item{prior_breg}{The prior distribution for the regression coefficient parameters in a
ARIMAX model. By default the value is set \code{NULL}, then the default student(7,0,1) priors
are used.}

\item{series.name}{an optional string vector with the series names.}

\item{...}{Further arguments passed to  \code{varstan} function.}
}
\value{
A \code{varstan} object with the fitted SARIMA model.
}
\description{
Fitting a SARIMA model  in \pkg{Stan}.
}
\details{
The function returns a \code{varstan} object with the fitted model.

If \code{xreg} option is used, the model by default will cancel the
seasonal differences adjusted (D = 0). If a value \code{d} > 0 is used, all
the regressor variables in \code{xreg} will be difference as well.

The default priors used in Sarima are:

\itemize{
\item{ar ~ normal(0,0.5)}
\item{ma ~ normal(0,0.5)}
\item{mu0 ~ t-student(0,2.5,6)}
\item{sigma0 ~ t-student(0,1,7)}
\item{sar ~ normal(0,0.5)}
\item{sma ~ normal(0,0.5)}
\item{breg ~ t-student(0,2.5,6)}
}
}
\examples{
\donttest{
 library(astsa)
 # Declare a multiplicative seasonal ARIMA model for the birth data.
 sf1 = stan_sarima(birth,order = c(0,1,2),seasonal = c(1,1,1))


 #Declare an Dynamic Harmonic Regression model for the birth data.
 sf2 = stan_sarima(birth,order = c(1,0,1),xreg = fourier(birth,K = 2))
}

}
\references{
Box, G. E. P. and Jenkins, G.M. (1978). Time series analysis: Forecasting and
control. San Francisco: Holden-Day. \emph{Biometrika}, 60(2), 297-303.
\code{doi:10.1093/biomet/65.2.297}.

Kennedy, P. (1992). Forecasting with dynamic regression models: Alan Pankratz, 1991.
\emph{International Journal of Forecasting}. 8(4), 647-648.
\code{url: https://EconPapers.repec.org/RePEc:eee:intfor:v:8:y:1992:i:4:p:647-648}.

Hyndman, R. & Khandakar, Y. (2008). Automatic time series forecasting: the
forecast package for \code{R}. \emph{Journal of Statistical Software}. 26(3),
1-22.\code{doi:	10.18637/jss.v027.i03}
}
\seealso{
\code{\link{garch}} \code{\link{set_prior}}
}
\author{
Asael Alonzo Matamoros
}
