\name{growth_form_table}
\alias{growth_form_table}
\title{Plant growth form occurrence matrices for TERN AusPlots
}
\description{Generates occurrence matrices for NVIS plant growth forms in plots as desired based on presence/absence, percent cover or species richness (i.e., the number of species assigned to a particular growth form). The input is a data frame of raw point intercept data from AusPlots generated using the \code{\link{get_ausplots}} function. Alternatively, cover by strata can be calculated.
}
\usage{
growth_form_table(veg.PI, m_kind=c("PA", "percent_cover", "richness"),	
cover_type=c("PFC", "OCC"), cumulative=TRUE, by_strata=FALSE, 
species_name=c("SN","HD","GS"))
}
\arguments{
  \item{veg.PI}{The input raw point intercept data generated by \code{\link{get_ausplots}}
}
  \item{m_kind}{The desired scoring method: binary (presence/absence; PA), percent cover or species richness.
}
  \item{cover_type}{A choice between PFC: 'projected foliage cover' and OCC: 'opaque canopy cover' and only applies to percent_cover. If 'PFC', the default, is selected (suitable as an abundance or cover measure for most applications), hits scored as 'in canopy sky' are removed, whereas they are retained as cover for that species for 'OCC' (suitable for application such as comparisons to remotely sensed cover).
}
\item{cumulative}{Logical. Should the cover of each growth form be calculated cumulatively across species (i.e., the sum of species covers in that growth form category; the default) or as absolute cover (percent of points where that growth form was hit)? Ignored except when \code{m_kind="percent_cover"}.
}
\item{by_strata}{Logical. If set to \code{TRUE}, cover is calculated by strata (as defined by growth forms).  Ignored except when \code{m_kind="percent_cover"}.
}
\item{species_name}{Select "SN" to use "standardised_names" (the default), "HD" to use "herbarium_determination", and "GS" to use "genus_species". See details for an explanation of each setting.
}
}
\details{Probably the most useful data output from this function is the matrix of plant growth forms against sites with percent cover. This could be used in manova, ordination, classification etc. It also provides a multivariate structural and functional response variable.

\code{species_name} provides multiple options for calculating plant growth form cover. If \code{species_name="HD"}, the function will use the "herbarium_determination" provided by state herbaria to identify unique species. "herbarium_determination" are made to the lowest possible taxonomic rank, which may be the family, genus, species, subspecies or variety. If \code{species_name="SN"}, the default, the function will use the the "standardised_name" to identify unique species. Values are based on the \code{veg.PI$herbarium_determination} but have been matched to the accepted scientific name according to the Australian Plant Census (APC: https://www.anbg.gov.au/cpbr/program/hc/hc-APC.html) and the Australian Plant Name Index (APNI: https://www.anbg.gov.au/cpbr/program/hc/hc-APNI.html). The standardised name field therefore helps to ensure taxonomic consistency. It excludes any records that don't match the APC. If \code{species_name="GS"}, only the standardised genus species rank ("genus_species" in the veg.PI input table) is used to identify species. \code{species_name="GS"} maximises consistency in species comparisons between plots, but reduces the taxonomic detail.

	Note sometimes the same species is recorded with different growth forms in a plot and therefore the rowSums of the occurrence matrix when m_kind is set to "richness" can be higher than the observed species richness because the same species can count towards the weights for multiple growth forms.
	
	By default, cumulative percent cover (percent of hits out of total number of point intercepts) is returned, meaning multiple species of the same growth form hit at a point add to the score, which can therefore exceed 100 percent. If \code{cumulative=FALSE}, non-cumulative cover is returned, whereby only one hit is counted per growth form per point-intercept, with a maximum of 100.
	
	If \code{by_strata=TRUE}, returns percent cover of three pre-defined strata: Upper, Mid and Lower, based on the crude assignment of growth forms such that Upper includes "Tree Mallee", "Tree/Palm", "Tree-fern", "Epiphyte"; Mid includes "Shrub Mallee", "Shrub", "Grass-tree", "Chenopod", "Heath-shrub"; and Lower includes "Tussock grass", "Forb", "Vine", "Hummock grass", "Fern", "Sedge", "Rush". The argument \code{cumulative} still applies for cover by strata, where the default (\code{TRUE}) is to calculate the cumulative cover of species in each stratum. Absolute cover (i.e., maximum value of 100) of strata can be calculated by setting \code{cumulative=FALSE}.
	
	The returned occurrence matrix (plant growth forms against sites) can be converted to long format (with rows as individual occurrences of growth forms) using:

\code{my.matrix <- growth_form_table(my.veg.PI, m_kind="percent_cover",
cover_type="PFC")} 

\code{library(reshape2)}

\code{my.matrix <- as.data.frame(melt(as.matrix(my.matrix)))}

\code{my.matrix <- my.matrix[-which(my.matrix $value == 0),]}

The \code{\link{round}} function can be used on the returned matrix to estimate percent covers to the nearest percent if desired (i.e., with digits=0)

}
\value{Returns a data frame with plant growth forms as columns and sites as rows. Values are determined by argument m_kind. For 'PA', binary presence/absence (0/1) is returned; for 'richness', the numbers of species (integer) in growth form categories are returned; for 'percent_cover', percent cover from point-intercept is returned (see \code{Details}).
}
\references{NVIS Technical Working Group (2017) Australian Vegetation Attribute Manual: National Vegetation Information System, Version 7.0. Department of the Environment and Energy, Canberra. Prep by Bolton, M.P., deLacey, C. and Bossard, K.B. (Eds)
}
\author{Greg R. Guerin, Samantha E.M. Munroe
}

\seealso{\code{\link{get_ausplots}}
	\cr
   \cr
    \code{\link{single_cover_value}}
}
\examples{
\dontrun{
#get veg.PI data for selected plots:
my.data <- get_ausplots(my.Plot_IDs=c("SATFLB0004", "QDAMGD0022", "NTASTU0002"),
	veg.PI=TRUE, site_info=FALSE)

#generate a percent cover occurrence matrix for plant growth forms:
growth_form_table(my.data$veg.PI, m_kind="percent_cover", cover_type="PFC",
cumulative=FALSE, species_name="SN")

#calculate cover of strata:
growth_form_table(my.data$veg.PI, m_kind="percent_cover", 
cover_type="PFC", cumulative=FALSE, by_strata=TRUE, species_name="SN")
}
}
