% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulate_wald_pvalues.R
\name{simulate_wald_pvalues}
\alias{simulate_wald_pvalues}
\title{Generate Wald test P-Values with Monte Carlo Simulations}
\usage{
simulate_wald_pvalues(
  model,
  nsim = 1000,
  responses = NULL,
  test_coefficients = NULL,
  refit_fn = NULL,
  coef_fn = get_fixef,
  vcov_fn = get_vcov,
  show_progress = TRUE,
  plot.it = TRUE,
  ...
)
}
\arguments{
\item{model}{A fitted model object that will be used to simulate responses.}

\item{nsim}{The number of simulations to perform.}

\item{responses}{An optional list of values to be used as response variables to refit the model.}

\item{test_coefficients}{Numeric vector. A vector with values to be used to compute
the test statistic. It should be the coefficients that was used to compute
the fitted values of the response. If \code{NULL} defaults to \code{coef_fn(model)}.}

\item{refit_fn}{Function to refit the model with new responses. If \code{NULL},
defaults to \code{get_refit(model, y, ...)}.}

\item{coef_fn}{Function that retrieves the coefficients of the model.}

\item{vcov_fn}{Function that computes the variance-covariance matrix for the models
adjusted in the simulations.}

\item{show_progress}{Display a progress bar for the simulation iteration.}

\item{plot.it}{Logical. Generate ecdf plot for joint Wald test.}

\item{...}{Additional arguments to be passed to \code{refit_fn}.}
}
\value{
An object of class \code{AD_pvalues}, which contains the following components:
\describe{
\item{test_coefficients}{Vector of coefficients being tested.}
\item{pvalues_matrix}{Matrix of p-values where each column corresponds to a
simulation and each row corresponds to a coefficient.}
\item{pvalues_joint}{Vector containing the joint p-values obtained from each simulation.}
\item{simulation_fixef}{List of fixed effect coefficient estimates from each simulation.}
\item{simulation_vcov}{List of covariance matrices estimated from each simulation.}
\item{simulation_warning}{Vector of boolean indicating if a simulation threw a warning.}
\item{converged}{Logical vector indicating whether model refitting converged for each simulation.}
\item{responses}{Simulated responses used for refitting the model.}
}
}
\description{
This function performs Monte Carlo simulations to generate p-values for
model coefficients by refitting the model with new simulated responses and
computing the Wald test statistics for each simulation.
It's standard behavior verify if the type I error from Wald tests are under control, considering the provided model
as "true", i.e., the model assumptions are valid.
It supports univariate and joint Wald tests, using chi-squared
distributions to calculate p-values.
}
\details{
If \code{responses} is provided, the function refits the model with these new
response vectors. Otherwise, it generates new responses with \code{\link[stats:simulate]{stats::simulate()}}.

For each new response calls \code{\link[=get_refit]{get_refit()}} to generate a new model with the new
response. It gets the fixed effects and the variance and covariance matrix with
\code{\link[=get_fixef]{get_fixef()}} and \code{\link[=get_vcov]{get_vcov()}}.

Each simulated model is refitted using the specified \code{refit_fn}
(or the default refit function) and the fixed effects coefficients and
variance-covariance matrix are extracted using \code{coef_fn} and \code{vcov_fn}, respectively.
The univariate Wald test is computed from the Wald statistic for each
coefficient, while the joint Wald test uses the inverse variance-covariance
matrix to compute a Wald statistic for the test_coefficients.
P-values are calculated from a chi-squared distribution with appropriate degrees of freedom.
}
\examples{
# from help("glm")
counts <- c(18, 17, 15, 20, 10, 20, 25, 13, 12)
outcome <- gl(3, 1, 9)
treatment <- gl(3, 3)
model <- glm(counts ~ outcome + treatment, family = poisson())
new_responses <- replicate(100, MASS::rnegbin(fitted.values(model), theta = 4.5), simplify = FALSE)
simulate_wald_pvalues(model, responses = new_responses, nsim = 100)

## Using custom refit_fn
if (require("survival")) {
  fit <- survreg(Surv(futime, fustat) ~ ecog.ps + rx, ovarian,
    dist = "exponential"
  )
  fitted_rate <- 1 / fitted(fit)
  new_responses <- replicate(100, rexp(length(fitted_rate), fitted_rate), simplify = FALSE)
  refit_surv_ovarian <- function(.y) {
    survreg(Surv(.y, fustat) ~ ecog.ps + rx, ovarian, dist = "exponential")
  }
  simulate_wald_pvalues(fit, responses = new_responses, refit_fn = refit_surv_ovarian)
}
}
\seealso{
\code{\link[=plot.AD_pvalues]{plot.AD_pvalues()}} for plotting.
}
