#'@title Generates a "diff" between a pair of revisions
#'
#'@description
#'wiki_diff generates a diff between two revisions in a MediaWiki page.
#'This is provided as an XML-parsable blob inside the returned JSON object.
#'
#'@param con A connector object, generated by \code{\link{wiki_con}},
#'that corresponds to the project you're trying to query.
#'
#'@param revisions The revision IDs of each "start" revision.
#'
#'@param properties Properties you're trying to retrieve about that revision, should you want to;
#'options include "ids" (the revision ID of the revision...which is pointless),
#'"flags" (whether the revision was 'minor' or not), "timestamp" (the timestamp of the revision,
#'which can be parsed with \code{\link{wiki_timestamp}}),"user" (the username of the person
#'who made that revision), "userid" (the userID of the person who made the revision),
#'"size" (the size, in uncompressed bytes, of the revision), "sha1" (the SHA-1 hash of
#'the revision text), "contentmodel" (the content model of the page, usually "wikitext"),
#'"comment" (the revision summary associated with the revision), "parsedcomment" (the same,
#'but parsed, generating HTML from any wikitext in that comment), "tags" (any tags associated
#'with the revision) and "flagged" (the revision's status under Flagged Revisions).
#'
#'@param direction The direction you want the diff to go in from the revisionID you have provided.
#'Options are "prev" (compare to the previous revision on that page), "next" (compare to the next
#'revision on that page) and "cur" (compare to the current, extant version of the page).
#'
#'@section Warnings:
#'
#'MediaWiki's API is deliberately designed to restrict users' ability to make computing-intense requests
#'- such as diff computation. As a result, the API only allows requests for one uncached diff in
#'each request. If you ask for multiple diffs, some uncached and some cached, you will be provided
#' with the cached diffs, one of the uncached diffs, and a warning.
#' 
#'If you're going to be asking for a lot of diffs, some of which may not be cached, it may be more
#'sensible to retrieve the revisions themselves using \code{\link{wiki_revision}} and compute the
#'diffs yourself.
#'
#'@export

wiki_diff <- function(con,
                      revisions,
                      properties = c("ids","flags","timestamp","user","userid","size",
                                             "sha1","contentmodel","comment","parsedcomment",
                                             "tags","flagged"),
                      direction = c("prev","next","cur")){
  
  #Match args
  direction <- match.arg(direction)
  properties <- match.arg(properties, several.ok = TRUE)
  
  #Save the desired properties and check revisions
  properties <- paste(properties, collapse = "|")
  revisions <- LimitHandler(revisions, 50)
    
  #Construct the URL
  diff_url <- paste(con$URL,"&action=query&prop=revisions&rvprop=",properties,"&rvdiffto=",direction,"&rvcontentformat=text/css&revids=",revisions, sep = "")
  
  #Retrieve the content
  diff_content <- wiki_call(URL = diff_url, con$CurlOpts)
  
  #Check for invalid RevIDs
  InvalidRevIDsHandler(diff_content)
  
  #Check for uncached diffs
  UncachedDiffsHandler(diff_content)
  
  #Return
  return(diff_content)
}