#' Integer representing shape of a tree
#'
#' Returns an integer that uniquely represents the shape of an _n_-tip
#' binary tree, ignoring tip labels.
#'
#' Rooted trees are numbered working up from the root.
#'
#' The root node divides _n_ tips into two subtrees.  The smaller subtree
#' may contain $a = 1, 2, ..., n/2$ tips, leaving $b = n - a$ tips in
#' These options are worked through in turn.
#'
#' For the first shape of the smaller subtree, work through each possible shape
#' for the larger subtree.  Then, move to the next shape of the smaller subtree,
#' and work through each possible shape of the larger subtree.
#'
#' Stop when the desired topology is encountered.
#'
#' Unrooted trees are numbered less elegantly.  Each cherry (i.e. node
#' subtending a pair of tips) is treated in turn.  The subtended tips are
#' removed, and the node treated as the root of a rooted tree.  The number of
#' this rooted tree is then calculated.  The tree is assigned a _key_
#' corresponding to the lowest such value.  The keys of all unrooted tree shapes
#' on _n_ tips are ranked, and the unrooted tree shape is assigned a _number_
#' based on the rank order of its key among all possible keys, counting from
#' zero.
#'
#' If `UnrootedTreeShape` or `UnrootedTreeKey` is passed a rooted tree,
#' the position of the root will be ignored.
#'
#' @template treeParam
#'
#' @return `TreeShape` returns an integer specifying the shape of a tree,
#' ignoring tip labels.
#'
#' @examples
#' RootedTreeShape(PectinateTree(8))
#' plot(RootedTreeWithShape(0, nTip = 8L))
#'
#' NRootedShapes(8L)
#' # Shapes are numbered from 0 to NRootedShapes(n) - 1.  The maximum shape is:
#' RootedTreeShape(BalancedTree(8))
#'
#' # Unique shapes of unrooted trees:
#' NUnrootedShapes(8L)
#'
#' # Keys of these trees:
#' UnrootedKeys(8L)
#'
#' # A tree may be represented by multiple keys.
#' # For a one-to-one correspondence, use a number instead:
#' allShapes <- lapply(seq_len(NUnrootedShapes(8L)) - 1L,
#'                     UnrootedTreeWithShape, 8L)
#' plot(allShapes[[1]])
#' sapply(allShapes, UnrootedTreeShape)
#' sapply(allShapes, UnrootedTreeKey) # Key >= number
#'
#'
#' @seealso [`TreeNumber`]
#'
#' @template MRS
#' @name TreeShape
#' @export
RootedTreeShape <- function (tree) {
  edge <- tree$edge
  nTip <- NTip(tree)
  edge <- PostorderEdges(edge)
  edge_to_rooted_shape(edge[, 1], edge[, 2], nTip)
}

#' @rdname TreeShape
#' @param shape Integer specifying shape of tree, perhaps generated by
#'  `TreeShape`.
#' @param nTip Integer specifying number of tips.
#' @return `RootedTreeWithShape` returns a tree of class `phylo` corresponding to the
#' shape provided.  Tips are unlabelled.
#' @export
RootedTreeWithShape <- function (shape, nTip, tipLabels = rep('', nTip)) {
  structure(list(edge = rooted_shape_to_edge(shape, nTip),
                 Nnode = nTip - 1,
                 tip.label = tipLabels),
            class = 'phylo')
}

#' @rdname TreeShape
#' @template tipLabelsParam
#' @return `UnrootedTreeWithShape` returns a tree of class `phylo` corresponding
#' to the shape provided.  Tips are unlabelled.
#' @export
UnrootedTreeWithShape <- function (shape, nTip, tipLabels = rep('', nTip)) {
  nShapes <- NUnrootedShapes(nTip)
  if (shape >= nShapes) {
    stop("Shape must be between 0 and ", nShapes)
  }

  UnrootedTreeWithKey(UnrootedKeys(nTip)[shape + 1L], nTip, tipLabels)
}

#' @rdname TreeShape
#' @param key Integer specifying the _key_ (not number) of an unrooted tree.
#' @return `UnrootedTreeWithKey` returns a tree of class `phylo` corresponding
#' to the key provided.  Tips are unlabelled.
#' @export
UnrootedTreeWithKey <- function (key, nTip, tipLabels = rep('', nTip)) {
  AddRoot <- function (x) {
    x$root.edge <- 1L
    x
  }
  SingleTaxonTree(tipLabels[1]) + SingleTaxonTree(tipLabels[2]) +
    AddRoot(RootedTreeWithShape(key, nTip - 2L, tipLabels[-c(1, 2)]))
}

#' @rdname TreeShape
#' @importFrom ape drop.tip root
#' @export
UnrootedTreeShape <- function (tree) {
  which(UnrootedKeys(NTip(tree)) == UnrootedTreeKey(tree)) - 1L
}

#' @rdname TreeShape
#' @importFrom ape drop.tip root
#' @export
UnrootedTreeKey <- function (tree) {
  tree <- Preorder(tree) # Guarantee unique representation of tree
  edge <- PostorderEdges(tree$edge)
  nTip <- NTip(tree)
  parent <- edge[, 1]
  child <- edge[, 2]
  nEdge <- length(child)
  unrooted <- nEdge %% 2L
  nodeFirst <- c(rep(c(TRUE, FALSE), nEdge / 2L), logical(as.integer(unrooted)))
  nodeSecond <- !nodeFirst
  nodeNumbers <- unique(parent)
  if (unrooted) {
    nodeFirst [nEdge - 0:2] <- FALSE
    nodeSecond[nEdge - 0:2] <- FALSE
    nodeNumbers <- nodeNumbers[-(nTip - 2L)]
  }

  RootedNumber <- function (nodeChildren) {
    RootedTreeShape(Postorder(drop.tip(root(tree, nodeChildren[1]), nodeChildren)))
  }

  basalTipEdges <- nEdge - (seq_len(4L - unrooted) - 1L)
  rootCandidate <- if (sum(child[basalTipEdges] <= nTip) == 2) {
    RootedNumber(child[basalTipEdges][child[basalTipEdges] <= nTip])
  } else {
    double(0)
  }

  cherryNodes <- nodeNumbers[child[nodeFirst] <= nTip & child[nodeSecond] <= nTip]
  allKeys <- c(vapply(cherryNodes, function (node) {
    RootedNumber(child[parent == node])
  }, double(1)), rootCandidate)

  # Return:
  min(allKeys)
}

#' @rdname TreeShape
#' @keywords internal
#' @export
.UnrootedKeys <- function (nTip) {
  if (nTip > 5) {
    #TODO make efficient - this is horrible!
    shapes <- vapply(seq_len(NRootedShapes(nTip)) - 1L, function (shape)
      UnrootedTreeKey(RootedTreeWithShape(shape, nTip)), double(1))
    uniqueShapes <- unique(shapes)
  } else {
    uniqueShapes <- 0
  }

  # Return:
  sort(uniqueShapes)
}

#' @rdname TreeShape
#' @param \dots Value of `nTip`, to pass to memoized `.UnrootedKeys`.
#' @param envir Unused; passed to [`addMemoization`].
#' @return `UnrootedKeys` returns a vector of integers corresponding to the
#' keys (not shape numbers) of unrooted tree shapes with `nTip` tips.
#' It is a wrapper to `.UnrootedKeys`, with memoization, meaning that results
#' once calculated are cached and need not be calculated on future calls to
#' the function.
#' @importFrom R.cache addMemoization
#' @export
UnrootedKeys <- addMemoization(.UnrootedKeys, envir = 'package:TreeTools')

#' @rdname TreeShape
#' @return `TreeShapes` returns an integer specifying the number of unique
#' unrooted tree shapes with `nTip` tips.
#' @export
NUnrootedShapes <- function (nTip) {
  length(UnrootedKeys(nTip))
}

#' @rdname TreeShape
#' @return `RootedTreeShapes` returns an integer specifying the number of unique
#' rooted tree shapes with `nTip` tips.
#' @export
NRootedShapes <- function (nTip) {
  n_rooted_shapes(as.integer(nTip))
}
