% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dist_wass_ipot.R
\name{ipot}
\alias{ipot}
\alias{ipotD}
\title{Wasserstein Distance by Inexact Proximal Point Method}
\usage{
ipot(X, Y, p = 2, wx = NULL, wy = NULL, lambda = 1, ...)

ipotD(D, p = 2, wx = NULL, wy = NULL, lambda = 1, ...)
}
\arguments{
\item{X}{an \eqn{(M\times P)} matrix of row observations.}

\item{Y}{an \eqn{(N\times P)} matrix of row observations.}

\item{p}{an exponent for the order of the distance (default: 2).}

\item{wx}{a length-\eqn{M} marginal density that sums to \eqn{1}. If \code{NULL} (default), uniform weight is set.}

\item{wy}{a length-\eqn{N} marginal density that sums to \eqn{1}. If \code{NULL} (default), uniform weight is set.}

\item{lambda}{a regularization parameter (default: 0.1).}

\item{...}{extra parameters including \describe{
\item{maxiter}{maximum number of iterations (default: 496).}
\item{abstol}{stopping criterion for iterations (default: 1e-10).}
\item{L}{small number of inner loop iterations (default: 1).}
}}

\item{D}{an \eqn{(M\times N)} distance matrix \eqn{d(x_m, y_n)} between two sets of observations.}
}
\value{
a named list containing\describe{
\item{distance}{\eqn{\mathcal{W}_p} distance value}
\item{iteration}{the number of iterations it took to converge.}
\item{plan}{an \eqn{(M\times N)} nonnegative matrix for the optimal transport plan.}
}
}
\description{
Due to high computational cost for linear programming approaches to compute
Wasserstein distance, Cuturi (2013) proposed an entropic regularization
scheme as an efficient approximation to the original problem. This comes with
a regularization parameter \eqn{\lambda > 0} in the term
\deqn{\lambda h(\Gamma) = \lambda \sum_{m,n} \Gamma_{m,n} \log (\Gamma_{m,n}).}
IPOT algorithm is known to be relatively robust to the choice of
regularization parameter \eqn{\lambda}. Empirical observation says that
very small number of inner loop iteration like \code{L=1} is sufficient.
}
\examples{
\donttest{
#-------------------------------------------------------------------
#  Wasserstein Distance between Samples from Two Bivariate Normal
#
# * class 1 : samples from Gaussian with mean=(-1, -1)
# * class 2 : samples from Gaussian with mean=(+1, +1)
#-------------------------------------------------------------------
## SMALL EXAMPLE
set.seed(100)
m = 20
n = 30
X = matrix(rnorm(m*2, mean=-1),ncol=2) # m obs. for X
Y = matrix(rnorm(n*2, mean=+1),ncol=2) # n obs. for Y

## COMPARE WITH WASSERSTEIN 
outw = wasserstein(X, Y)
ipt1 = ipot(X, Y, lambda=1)
ipt2 = ipot(X, Y, lambda=10)

## VISUALIZE : SHOW THE PLAN AND DISTANCE
pmw = paste0("wasserstein plan ; dist=",round(outw$distance,2))
pm1 = paste0("ipot lbd=1 ; dist=",round(ipt1$distance,2))
pm2 = paste0("ipot lbd=10; dist=",round(ipt2$distance,2))

opar <- par(no.readonly=TRUE)
par(mfrow=c(1,3))
image(outw$plan, axes=FALSE, main=pmw)
image(ipt1$plan, axes=FALSE, main=pm1)
image(ipt2$plan, axes=FALSE, main=pm2)
par(opar)
}

}
\references{
\insertRef{xie_fast_2020}{T4transport}
}
\concept{dist_wass}
