% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/batty_karlstrom_entropy.R
\name{batty}
\alias{batty}
\title{Batty's entropy.}
\usage{
batty(data, category = 1, cell.size = 1, partition = 10)
}
\arguments{
\item{data}{If data are lattice, a data matrix, which can be numeric, factor, character, ...
If the dataset is a point pattern, \code{data} is a \code{ppp} object.}

\item{category}{A single value matching the data category of interest for computing Batty's entropy.
Default to 1. If the dataset is an unmarked point pattern, this argument must not be changed from the default.}

\item{cell.size}{A single number. If data are lattice, the length of the side of each pixel.
Default to 1. Ignored if data are points.}

\item{partition}{Input defining the partition into subareas. If an integer, it defines the
number of sub-areas that are randomly generated by \link{areapart}; if a two column matrix
with coordinates, they are the centroids of the subareas built by \link{areapart}. Alternatively,
it can be the output of \link{areapart}, or a \code{tess} object built by the user.
The default option is \code{partition=areapart(data, G=10)}, which generates 10 random sub-areas.}
}
\value{
A list of four elements:
\itemize{
\item \code{area.tess} a \code{tess} object with the area partition
\item \code{areas.freq} a dataframe giving, for each sub-area, the absolute and relative frequency of
the points/pixels of interest and the sub-area size
for each sub-area
\item \code{batty} Batty's entropy
\item \code{rel.batty} Batty's entropy divided by \eqn{\log(\sum Tg)} for comparison across observation areas.
}
}
\description{
This function computes Batty's spatial entropy, following Batty (1976), see also Altieri et al (2017 and following)
(references are under the topic \code{\link{SpatEntropy}}).
}
\details{
Batty's spatial entropy measures the heterogeneity in the spatial distribution
of a phenomenon of interest, with regard to an area partition. It is high
when the phenomenon is equally intense over the sub-areas, and low when
it concentrates in one or few sub-areas. This function allows to compute Batty's entropy as
\deqn{H_B=\sum p_g \log(T_g/p_g)}
where \eqn{p_g} is the probability of occurrence of the phenomenon over sub-area \eqn{g},
and \eqn{T_g} is the sub-area size.
When data are categorical, the phenomenon of interest corresponds to
one category, which must be specified. If data are an unmarked
point pattern, a fake mark vector is be created with the same category for all points.
For comparison purposes, the relative version of Batty's entropy is also returned, i.e.
Batty's entropy divided by its maximum \eqn{\log(\sum T_g)}.
Note that when the total observation area is 1, then \eqn{\log(\sum T_g)=0}, therefore
in that case during the computation all \eqn{T_g}s are multiplied by 100 and a warning is produced.
The function is able to work with grids containing missing data, specified as NA values.
All NAs are ignored in the computation.
}
\examples{
#LATTICE DATA

data=matrix((sample(c("a","b","c"), 100, replace=TRUE)), nrow=10)
batty.entropy=batty(data, category="a")
##to plot
data.binary=matrix(as.numeric(data=="a"), nrow(data))
plot(as.im(data.binary, W=batty.entropy$areas.tess$window), main="",
     col=gray(seq(1,0,l=length(unique(c(data.binary))))), ribbon=FALSE)
plot(batty.entropy$areas.tess, add=TRUE, border=2)

#POINT DATA

#unmarked pp
data=ppp(x=runif(100, 0, 10), y=runif(100, 0, 10), window=square(10))
batty.entropy=batty(data)
##to plot
plot(data)
plot(batty.entropy$areas.tess, add=TRUE, border=2)

#marked pp
data=ppp(x=runif(100, 0, 10), y=runif(100, 0, 10), window=square(10),
         marks=(sample(1:5, 100, replace=TRUE)))
plot(data) #see ?plot.ppp for options
#if you want to compute the entropy on all points
batty.entropy=batty(unmark(data))
#if you want to compute the entropy on a category, say 3
batty.entropy=batty(data, category=3)
##to plot using the selected category
ind=which(marks(data)==3)
data.binary=unmark(data[ind])
plot(data.binary)
plot(batty.entropy$areas.tess, add=TRUE, border=2)

}
