% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/spread-process.R
\docType{methods}
\name{spread}
\alias{spread}
\alias{spread,RasterLayer-method}
\title{Simulate a spread process on a landscape.}
\usage{
spread(landscape, loci = NA_real_, spreadProb = 0.23, persistence = 0,
  mask = NA, maxSize = 100000000L, directions = 8L,
  iterations = 1000000L, lowMemory = getOption("spades.lowMemory"),
  returnIndices = FALSE, mapID = FALSE, plot.it = FALSE,
  spreadProbLater = NA_real_, spreadState = NA, ...)

\S4method{spread}{RasterLayer}(landscape, loci = NA_real_,
  spreadProb = 0.23, persistence = 0, mask = NA, maxSize = 100000000L,
  directions = 8L, iterations = 1000000L,
  lowMemory = getOption("spades.lowMemory"), returnIndices = FALSE,
  mapID = FALSE, plot.it = FALSE, spreadProbLater = NA_real_,
  spreadState = NA, ...)
}
\arguments{
\item{landscape}{A \code{RasterLayer} object.}

\item{loci}{A vector of locations in \code{landscape}}

\item{spreadProb}{Numeric or rasterLayer.  The overall probability of
spreading, or probability raster driven. Default is 0.23.
If a \code{spreadProbLater} is provided, then this is
only used for the first iteration. Also called Escape
probability.}

\item{persistence}{A probability that an active cell will continue to burn,
per time step.}

\item{mask}{non-NULL, a \code{RasterLayer} object congruent with
\code{landscape} whose elements are \code{0,1},
where 1 indicates "cannot spread to".
Currently not implemented.}

\item{maxSize}{Vector of the maximum number of pixels for a single or
all events to be spread.
Recycled to match \code{loci} length.}

\item{directions}{The number adjacent cells in which to look;
default is 8 (Queen case).}

\item{iterations}{Number of iterations to spread.
Leaving this \code{NULL} allows the spread to continue
until stops spreading itself (i.e., exhausts itself).}

\item{lowMemory}{Logical. If true, then function uses package \code{ff}
internally. This is slower, but much lower memory footprint.}

\item{returnIndices}{Logical. Should the function return a data.table with
indices and values of successful spread events, or
return a raster with values. See Details.}

\item{mapID}{Logical. If TRUE, returns a raster of events ids.
If FALSE, returns a raster of iteration numbers,
i.e., the spread history of one or more events. NOTE:
this is overridden if \code{returnIndices} is \code{TRUE}.}

\item{plot.it}{If TRUE, then plot the raster at every iteraction,
so one can watch the spread event grow.}

\item{spreadProbLater}{Numeric or rasterLayer. If provided, then this
will become the spreadProb after the first iteration. See details.}

\item{spreadState}{Data.table. This should be the output of a previous call to
\code{spread}. See Details.}

\item{...}{Additional parameters.}
}
\value{
Either a \code{RasterLayer} indicating the spread of the process in
the landscape or a \code{data.table}. If a \code{RasterLayer}, then it represents
every pixel in which a successful spread event occurred. For the case of, say, a fire
this would represent every pixel that burned. If \code{returnIndices} is \code{TRUE},
then this function returns a \code{data.table} with columns:

\code{indices} is the pixel indices of pixels that have been touched by the spread
algorithm.

\code{eventID} is an arbitrary ID \code{1:length(loci)} identifying unique clusters
of spread events, i.e., all pixels that have been spread into that have a common
initial pixel.

\code{active} is a logical indicating whether the pixel is active (i.e., could still
be a source for spreading) or not (no spreading will occur from these pixels).

\code{initialIndex}, the initial pixel number of that particular spread event.
}
\description{
This can be used to simulated fires or other things.
Essentially, it starts from a collection of cells (\code{loci}) and spreads
to neighbours, according to the \code{directions} and \code{spreadProbLater} arguments.
This can become quite general, if \code{spreadProbLater} is 1 as it will expand
from every loci until all pixels in the landscape have been covered.
With \code{mapID} set to \code{TRUE}, the resulting map will be classified
by the index of the pixel where that event propagated from.
This can be used to examine things like fire size distributions.
}
\details{
For large rasters, a combination of \code{lowMemory = TRUE} and
\code{returnIndices = TRUE} will use the least amount of memory.

This function can be interrupted before all active cells are exhausted if
the \code{iterations} value is reached before there are no more active
cells to spread into. If this is desired, \code{returnIndices} should be
\code{TRUE} and the output of this call can be passed subsequently as an input
to this same function. This is intended to be used for situations where external
events happen during a spread event, or where one or more arguments to the spread
function change before a spread event is completed. For example, if it is
desired that the \code{spreadProb} change before a spread event is completed because,
for example, a fire is spreading, and a new set of conditions arise due to
a change in weather.
}
\examples{
library(raster)
library(RColorBrewer)

# Make random forest cover map
a <- raster(extent(0,1e2,0,1e2), res = 1)
hab <- gaussMap(a,speedup = 1) # if raster is large (>1e6 pixels), use speedup>1
names(hab) = "hab"
cells <- loci <- b <- as.integer(sample(1:ncell(a),1e1))
mask <- raster(a)
mask <- setValues(mask, 0)
mask[1:5000] <- 1
numCol <- ncol(a)
numCell <- ncell(a)
directions <- 8

# Transparency involves putting two more hex digits on the color code: 00 is fully transparent.
setColors(hab) <- paste(c("#FFFFFF", brewer.pal(8, "Greys")), c("00", rep("FF", 8)), sep = "")

#dev(4)
Plot(hab, new = TRUE, speedup = 3) # note speedup is equivalent to making pyramids,
                             # so, some details are lost

# initiate 10 fires at to loci
fires <- spread(hab, loci = as.integer(sample(1:ncell(hab), 10)),
                0.235, 0, NULL, 1e8, 8, 1e6, mapID = TRUE)
#set colors of raster, including a transparent layer for zeros
setColors(fires, 10) <- c("#00000000", brewer.pal(8,"Reds")[5:8])
Plot(fires)
Plot(fires,addTo = "hab")

#alternatively, set colors using cols= in the Plot function
Plot(hab, new = TRUE)
Plot(fires) # default color range makes zero transparent.
# Instead, to give a color to the zero values, use \\code{zero.color=}
Plot(fires, addTo = "hab",
     cols = colorRampPalette(c("orange","darkred"))(10))
hab2 <- hab
Plot(hab2)
Plot(fires, addTo = "hab2", zero.color = "white",
     cols = colorRampPalette(c("orange","darkred"))(10))
# or overplot the original (NOTE: legend stays at original values)
Plot(fires,
     cols = topo.colors(10))

## Use interrupt a spread event using iterations - need returnIndices=TRUE to use outputs
##   as new inputs in next iteration
fires <- spread(hab, loci = as.integer(sample(1:ncell(hab), 10)), returnIndices=TRUE,
                0.235, 0, NULL, 1e8, 8, iterations = 3, mapID = TRUE)
fires[,list(size=length(initialLocus)), by=eventID]  # See sizes of fires

## Continue event by passing interrupted object into spreadState
fires2 <- spread(hab, loci=NA_real_, returnIndices=TRUE, 0.235,
                 0, NULL, 1e8, 8, iterations = 2, mapID = TRUE,
                 spreadState=fires)
# NOTE events are assigned arbitrary IDs, starting at 1

## Add new fires to the already burning fires
fires3 <- spread(hab, loci = as.integer(sample(1:ncell(hab), 10)), returnIndices=TRUE,
                     0.235, 0, NULL, 1e8, 8, iterations = 1, mapID = TRUE,
                                          spreadState=fires)
fires3[,list(size=length(initialLocus)), by=eventID]  # See sizes of fires
# NOTE old eventIDs are maintained, new events get ids begining above previous
# maximum (e.g., new fires 11 to 20 here)

## Use data.table and loci...
fires <- spread(hab, loci = as.integer(sample(1:ncell(hab), 10)), returnIndices=TRUE,
                0.235, 0, NULL, 1e8, 8, iterations = 2, mapID = TRUE)
fullRas <- raster(hab)
fullRas[] <- 1:ncell(hab)
burned <- fires[active == FALSE]
burnedMap <- rasterizeReduced(burned, fullRas, "eventID", "indices")
Plot(burnedMap, new=TRUE)
}
\author{
Steve Cumming \email{Steve.Cumming@sbf.ulaval.ca}

Eliot McIntire
}

