% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/nonlinear_RPCA.R
\name{do.rpca}
\alias{do.rpca}
\title{Robust Principal Component Analysis}
\usage{
do.rpca(X, lambda = sqrt(1/(max(dim(X)))), preprocess = c("center",
  "decorrelate", "whiten"))
}
\arguments{
\item{X}{an \eqn{(n\times p)} matrix or data frame whose rows are observations and columns represent independent variables.}

\item{lambda}{parameter for the sparsity term \eqn{\|S\|_1}. Default value is given accordingly to the referred paper.}

\item{preprocess}{an additional option for preprocessing the data. Default is ``center'', and other methods of ``decorrelate'' and ``whiten'' are supported. See also \code{\link{aux.preprocess}} for more details.}
}
\value{
a named list containing
\describe{
\item{L}{an \eqn{(n\times p)} low-rank matrix.}
\item{S}{an \eqn{(n\times p)} sparse matrix.}
\item{cvxr.status}{``optimal'' denotes the problem was solved. See\code{\link[CVXR]{psolve}} for more details on solvability.}
\item{cvxr.niters}{the number of iterations taken.}
\item{cvxr.solver}{type of solver used by \pkg{CVXR}.}
}
}
\description{
Robust PCA (RPCA) is not like other methods in this package as finding explicit low-dimensional embedding with reduced number of columns.
Rather, it is more of a decomposition method of data matrix \eqn{X}, possibly noisy, into low-rank and sparse matrices by
solving the following,
\deqn{\textrm{minimize}\quad \|L\|_* + \lambda \|S\|_1}
\deqn{\textrm{subject to}\quad L+S=X}
where \eqn{L} is a low-rank matrix, \eqn{S} is a sparse matrix and \eqn{\|\cdot\|_*} denotes nuclear norm, i.e., sum of singular values. Therefore,
it should be considered as \emph{preprocessing} procedure of denoising. Note that after RPCA is applied, \eqn{L} should be used
as kind of a new data matrix for any manifold learning scheme to be applied. \pkg{CVXR} was used for \code{do.rpca}.
}
\examples{
\dontrun{
## Load Iris data and put some noise
data(iris)
noise = 0.2
X = as.matrix(iris[,1:4])
X = X + matrix(noise*rnorm(length(X)), nrow=nrow(X))

## Compare 2 methods : {PCA} vs {RPCA + PCA}
rpca     = do.rpca(X)
out_pca  = do.pca(X,      ndim=2, preprocess="center")
out_rpca = do.pca(rpca$L, ndim=2, preprocess="center")

## Visualize
par(mfrow=c(1,2))
plot(out_pca$Y[,1], out_pca$Y[,2], main="PCA")
plot(out_rpca$Y[,1], out_rpca$Y[,2], main="RPCA")
}

}
\references{
\insertRef{candes_robust_2011}{Rdimtools}
}
\author{
Kisung You
}
