#' Intermediate Header
#'
#' Combines multiple descriptive tables into a single table with customized row group headers and styling. 
#' This function accepts a list of tables and corresponding group headers, applies consistent styling, 
#' and outputs a styled `gt` table.
#'
#' @param tbls A list of descriptive tables (generated by `RastaRocket::desc_var` or similar functions).
#' @param group_header A character vector specifying the headers for each group of tables. 
#'                     Must be the same length as `tbls`.
#' @param color A character string specifying the background color for the row group headers. 
#'              Default is `"#8ECAE6"`.
#' @param size An integer specifying the font size for the row group headers. Default is `16`.
#' @param align A character string specifying text alignment for the row group headers. 
#'              Options are `"left"`, `"center"`, or `"right"`. Default is `"center"`.
#' @param weight A character string specifying the font weight for the row group headers. 
#'               Options include `"normal"`, `"bold"`, etc. Default is `"bold"`.
#'
#' @return A styled `gt` table combining the input tables with row group headers.
#' @import gt
#' @export
#'
#' @examples
#' # Load necessary libraries
#' library(RastaRocket)
#' library(dplyr)
#'
#' # Generate sample data
#' data <- data.frame(
#'   Age = c(rnorm(45, mean = 50, sd = 10), rep(NA, 5)),
#'   sexe = sample(c("Femme", "Homme"), 50, replace = TRUE, prob = c(0.6, 0.4)),
#'   quatre_modalites = sample(c("A", "B", "C", "D"), 50, replace = TRUE)
#' )
#'
#' # Create descriptive tables
#' tb1 <- data %>%
#'   dplyr::select(Age, sexe) %>%
#'   RastaRocket::desc_var(table_title = "Demographics", group = FALSE)
#'
#' tb2 <- data %>%
#'   dplyr::select(quatre_modalites) %>%
#'   RastaRocket::desc_var(table_title = "Modalities", group = FALSE)
#'
#' # Combine and style tables
#' intermediate_header(
#'   tbls = list(tb1, tb2),
#'   group_header = c("Demographics", "Modalities")
#' )

intermediate_header <- function(tbls,
                                group_header,
                                color = "#8ECAE6",
                                size = 16,
                                align = 'center',
                                weight = "bold"){
  
  if(length(group_header) != length(tbls)){
    stop("tbls and group_header must be the same length")
  }
  
  gtsummary::tbl_stack(tbls,
                       group_header = group_header) %>%
    gtsummary::as_gt() %>%
    gt::tab_style(
      style = gt::cell_text(weight = weight, align = align, size = size), # Texte en gras, centré, taille de police plus grande
      locations = gt::cells_row_groups(groups = everything())
    ) %>%
    gt::tab_style(
      style = gt::cell_fill(color = color), # Fond bleu avec les valeurs RGB spécifiées
      locations = gt::cells_row_groups(groups = everything())
    ) %>%
    return()
}