% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/transfer_entropy.R
\name{transfer_entropy}
\alias{transfer_entropy}
\title{Function to estimate Shannon and Renyi transfer entropy between two time
series x and y.}
\usage{
transfer_entropy(x, y, lx = 1, ly = 1, q = 0.1,
  entropy = "Shannon", shuffles = 100, type = "quantiles",
  quantiles = c(5, 95), bins = NULL, limits = NULL, nboot = 300,
  burn = 50, quiet = NULL, seed = NULL)
}
\arguments{
\item{x}{a vector of numeric values, ordered by time.
Also allowed are \code{\link[xts]{xts}}, \code{\link[zoo]{zoo}},
or \code{\link[stats]{ts}} objects.}

\item{y}{a vector of numeric values, ordered by time.
Also allowed are \code{\link[xts]{xts}}, \code{\link[zoo]{zoo}},
or \code{\link[stats]{ts}} objects.}

\item{lx}{Markov order of x, i.e. the number of lagged values affecting the
current value of x. Default is \code{lx = 1}.}

\item{ly}{Markov order of y, i.e. the number of lagged values affecting the
current value of y. Default is \code{ly = 1}.}

\item{q}{a weighting parameter used to estimate Renyi transfer entropy,
parameter is between 0 and 1. For \code{q = 1}, Renyi transfer
entropy converges to Shannon transfer entropy. Default is
\code{q = 0.1}.}

\item{entropy}{specifies the transfer entropy measure that is estimated,
either 'Shannon' or 'Renyi'. The first character can be used
to specify the type of transfer entropy as well. Default is
\code{entropy = 'Shannon'}.}

\item{shuffles}{the number of shuffles used to calculate the effective
transfer entropy. Default is \code{shuffles = 100}.}

\item{type}{specifies the type of discretization applied to the observed time
series:'quantiles', 'bins' or 'limits'. Default is
\code{type = 'quantiles'}.}

\item{quantiles}{specifies the quantiles of the empirical distribution of the
respective time series used for discretization.
Default is \code{quantiles = c(5,95)}.}

\item{bins}{specifies the number of bins with equal width used for
discretization. Default is \code{bins = NULL}.}

\item{limits}{specifies the limits on values used for discretization.
Default is \code{limits = NULL}.}

\item{nboot}{the number of bootstrap replications for each direction of
the estimated transfer entropy. Default is \code{nboot = 300}.}

\item{burn}{the number of observations that are dropped from the beginning of
the bootstrapped Markov chain. Default is \code{burn = 50}.}

\item{quiet}{if FALSE (default), the function gives feedback.}

\item{seed}{a seed that seeds the PRNG (will internally just call set.seed),
default is \code{seed = NULL}.}
}
\value{
an object of class transfer_entropy, containing the transfer entropy
        estimates in both directions, the effective transfer entropy
        estimates in both directions, standard errors and p-values based on
        bootstrap replications of the Markov chains under the null hypothesis
        of statistical independence, an indication of statistical
        significance, and quantiles of the bootstrap samples
        (if \code{nboot > 0}).
}
\description{
Function to estimate Shannon and Renyi transfer entropy between two time
series x and y.
}
\examples{
# construct two time-series
set.seed(1234567890)
n <- 500
x <- rep(0, n + 1)
y <- rep(0, n + 1)

for (i in seq(n)) {
  x[i + 1] <- 0.2 * x[i] + rnorm(1, 0, 2)
  y[i + 1] <- x[i] + rnorm(1, 0, 2)
}

x <- x[-1]
y <- y[-1]

# Calculate Shannon's Transfer Entropy
te_result <- transfer_entropy(x, y, nboot = 100)
te_result

summary(te_result)

\donttest{
  # Parallel Processing using the future-package
  library(future)
  plan(multiprocess)

  te_result2 <- transfer_entropy(x, y, nboot = 100)
  te_result2

  # revert back to sequential execution
  plan(sequential)

  te_result2 <- transfer_entropy(x, y, nboot = 100)
  te_result2

  # General set of quiet
  set_quiet(TRUE)
  a <- transfer_entropy(x, y, nboot = 0)

  set_quiet(FALSE)
  a <- transfer_entropy(x, y, nboot = 0)
}
}
\seealso{
\code{\link{coef}}, \code{\link{print.transfer_entropy}}
}
